package server.util;

import java.util.HashSet;
import java.util.Set;

/**
 * This class can be extended to implement a
 * push notification service for other classes interested
 * in actions that are performed.
 * The classes interested in the notification must
 * implement {@link Observer}.
 * 
 * The method name schema is taken from {@link java.util.Observable}.
 *
 * @param <S> The implementing subclass which can be observed.
 * @param <D> Used to specify the type of the data used as third parameter in {@link Observer#update(Observable, Action, Object)}.
 */
public abstract class Observable<S extends Observable<S,D>,D> {
	
	/**
	 * Possible action types.
	 */
	public enum Action {
		ADD,
		DELETE,
		MODIFY;
	}
	
	// It's not necessary to synchronize the set because all
	// methods using the set are synchronized in this class.
	private Set<Observer<S,D>> observers = new HashSet<Observer<S,D>>();

	/**
	 * Adds an observer to the set of observers for this object, provided that it is not the same as some observer already in the set.
	 * @param observer an observer to be added.
	 */
	public synchronized void addObserver( Observer<S,D> observer ) {
		this.observers.add( observer );
	}
	
	/**
	 * Deletes an observer from the set of observers of this object. Passing null to this method will have no effect.
	 * @param observer the observer to be deleted.
	 */
	public synchronized void deleteObserver( Observer<S,D> observer ) {
		this.observers.remove( observer );
	}
	
	/**
	 * Clears the observer list so that this object no longer has any observers.
	 */
	public synchronized void deleteObservers() {
		this.observers.clear();
	}
	
	/**
	 * Returns the number of observers of this Observable object.
	 * @return The number of observers of this object.
	 */
	public synchronized int countObservers() {
		return this.observers.size();
	}
	
	/**
	 * If this object has changed then notify all of its observers.
	 * Each observer has its {@link Observer#update(Observable, Action, Object)} method called with three arguments: this observable object, the action and the data argument.
	 * @param action One of the actions specified in this class.
	 * @param data Any object.
	 */
	public synchronized void notifyObservers( Action action, D data ) {
		for ( Observer<S,D> observer : this.observers ) {
			observer.update( this, action, data );
		}
	}
}
