package server.theoremprover;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import server.parser.CompletenessSentenceFormula;
import server.parser.Formula;
import server.parser.ProverNineFormula;
import server.parser.node.AtomPredicateNode;
import server.parser.node.DisjunctorNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.UniversalQuantifiedNode;
import server.parser.node.VariableNode;

import exception.ProverException;

/**
 * Schnittstelle zum Theorembeweiser Prover9.
 */
public class ProverNine extends TheoremProver {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.theoremprover");
	
	/**
	 * Pfad zum Prover9
	 */
	private static final String PROVER9_EXE = "Prover9/prover9";
	
	/**
	 * Maximale Anzahl paralleler Prover9-Aufrufe
	 */
	private static final int NUMBER_OF_CONCURRENT_PROVES = 4;

	/**
	 * Liefert einen String zurueck, der allgemeine Optionen fuer den Prover9
	 * enthaelt.
	 * 
	 * @return Optionen fuer den Prover9 (String in Prover9-Syntax)
	 */
	private String getProver9Options() {
		String prover9Options = "";
		
		// Allgemeine Optionen fuer Prover9
		// wenn es mehr als eine negative Klausel gibt, bewirkt diese Eingabe, dass der Prover9 diese nicht einzeln testet
		prover9Options += "clear(auto_denials).\n\n";
		prover9Options += "set(prolog_style_variables).\n\n";
		// Verhindern, dass der Prover9 abgeleitete Klauseln ignoriert und somit falsche Ergebnisse erzeugt
		// kommt bei grossen Eingaben wie z.B. dem Completeness Sentence vor
		prover9Options += "assign(max_weight, 2147483647).\n\n";
		// Beschraenkung der Anzahl an Eingabe-Klauseln aufheben (fuer die offenen Zensoren, da diese bei der Standardeinstellung (20.000) 
		// an diese Grenze stossen)
		prover9Options += "assign(sos_limit, -1).\n";
		
		return prover9Options;
	}
	
	/**
	 * Bereitet die Eingabe fuer den Prover9 vor. Die Eingabe besteht aus 
	 * allgemeinen Optionen fuer den Prover9 und den Parametern sosInput und
	 * goalInput.
	 * 
	 * @param sosInput	Premissen, die bereits in Prover9-Syntax vorliegen
	 * @param goalInput Konklusion, die bereits in Prover9-Syntax vorliegt
	 * @return Prover9-Eingabe, die an den Prover9 weitergereicht werden kann
	 */
	private String prepareInput( String sosInput, String goalInput ) {
		String input = "";
		
		// Prover9 Optionen
		input += getProver9Options();
		
		// SOS
		input += "formulas(sos).\n";
		input += sosInput;
		input += "end_of_list.\n\n";
		
		// Goal
		input += "formulas(goals).\n";
		input += goalInput;
		input += "end_of_list.\n";
		
		return input;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ProverResult proveWithResult( String sos, String goal ) throws ProverException {
		String input = prepareInput( sos, goal );
				
		// Programm an Prover9 senden
		ProverNineTask proverNineTask = new ProverNineTask( input );
		ProverResult result = proverNineTask.call();
        return result;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ProverResult> proveWithResult( List<? extends Node> sos, List<? extends Node> goals, boolean uniqueNameAssumption ) throws ProverException {
		String sosInput = new String();
		
		if( uniqueNameAssumption ) {
			sosInput += getUniqueNameAssumption( sos, goals );
		}
		
		// Counter, der zur Unterscheidung der active domains bei mehreren completeness sentences dient
		int completenessSentenceCounter = 0;
		
		// SOS-Liste aufbauen
		for ( Node sosEntry : sos ) {
        	if( sosEntry.isCompletenessSentenceFormula() ) {
        		++completenessSentenceCounter;
        		
        		// Prover input bzgl. des completeness sentence anpassen zwecks Laufzeitoptimierung.
        		// Der Vollständigkeitssatz wird nicht in die Prover-Eingabe uebernommen, stattdessen
        		// werden fuer alle Kombinationen der Konstanten des completeness sentence explizit
        		// angegeben, ob diese Kombination gilt oder nicht.
        		// Fuer Konstanten, die nicht im completeness sentence enthalten sind, wird eine
        		// Aussage gemacht, dass die zum completeness sentence gehoerende Formel phi fuer 
        		// diese Kombination nicht gilt.
        		// Dies vermeidet die rechenaufwaendige Umwandlung des Completeness Sentence in KNF.
        		
        		CompletenessSentenceFormula cs = (CompletenessSentenceFormula) sosEntry;
        		
        		// generiere Formeln fuer die active domain des Completeness Sentence
        		sosInput += generateActiveDomainCS( cs, completenessSentenceCounter );
        		
        		// zerlege den Completeness Sentence in einzelne Formeln, indem jede
        		// nicht ausgeschlossene Kombination von Konstanten der active domain
        		// des CS in die Formel phi des CS eingesetzt wird und fuer diese
        		// ausgesagt wird, dass sie fuer diese Kombination nicht gilt
        		sosInput += generateNegatedFormulasFromCompletenessSentence( cs );
        		
        		// fuege Formeln hinzu, die aussagen, dass die Formel phi des Completeness Sentence
        		// fuer alle Kombinationen _nicht_ gilt, die mind. eine Konstante enthalten, die 
        		// nicht in der active domain des Completeness Sentence ist
        		sosInput += generateNegatedFormulasForActiveDomainComplement( cs, completenessSentenceCounter );
        	}
        	else {
        		// normale Formel (kein completeness sentence)
        		ProverNineFormula sosEntryProverNine = null;
        		sosEntryProverNine = new ProverNineFormula(sosEntry);
            	sosInput += sosEntryProverNine.toString() + ".\n";
        	}
        }
        
        // fuer jedes Goal den Prover einzeln starten
		List<ProverResult> proverResults = new LinkedList<ProverResult>();
		if( goals.size() <= 1 ) {
			// nur ein Goal (oder gar kein Goal, wenn Formelmenge auf Konsistenz geprueft werden soll)
			// -> direkt Prover9 aufrufen ohne zusaetzliche Threads
			
			ProverNineFormula goalProverNine = new ProverNineFormula( goals.get(0) );
			String goalInput = goalProverNine.toString() + ".\n";
			String input = prepareInput( sosInput, goalInput );
			
			ProverNineTask proverNineTask = new ProverNineTask( input );
			proverResults.add( proverNineTask.call() );
		}
		else {
			// mehrere Goals -> Prover9 in mehreren Threads parallel ausfuehren
			
			ExecutorService executor = Executors.newFixedThreadPool( NUMBER_OF_CONCURRENT_PROVES );
			List<Future<ProverResult>> futureList = new LinkedList<Future<ProverResult>>();
			for( Node goal : goals ) {
				ProverNineFormula goalProverNine = new ProverNineFormula( goal );
				String goalInput = goalProverNine.toString() + ".\n";
				
				String input = prepareInput( sosInput, goalInput );
				Future<ProverResult> future = executor.submit( new ProverNineTask(input) );
				futureList.add( future );
			}
			
			// Ergebnisse ermitteln
			try {
				for( Future<ProverResult> future : futureList ) {
					proverResults.add( future.get() );
				}
			}
			catch( ExecutionException e ) {
				Throwable throwable = e.getCause();
				if( throwable instanceof ProverException ) {
					throw (ProverException)throwable;
				}
				else {
					throw new ProverException( "Unknown error during execution of Prover9.", e);
				}
			}
			catch( InterruptedException e ) {
				throw new ProverException( "Prover9 execution was interrupted.", e );
			}
			
			// evtl. noch laufende Threads beenden, da Ergebnis schon feststeht
			executor.shutdownNow();
		}
		
		// wenn retVal true ist dann ist mind. eine Implikation erfuellt, andernfalls sind alle Implikationen nicht erfuellt		
		return proverResults;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ProverResult proveWithResult( List<? extends Node> sos, Node goal, boolean uniqueNameAssumption ) throws ProverException {
		List<Node> goals = new LinkedList<Node>();
		goals.add( goal );
		return proveWithResult(sos, goals, uniqueNameAssumption).get(0);
	}
	
	/**
	 * Generiert Formeln fuer die Unique Name Assumption und liefert diese in Form eines
	 * Strings zurueck, der direkt an den Prover9 weitergereicht werden kann.
	 * Die UNA wird dabei fuer alle Konstanten angewandt, die in der Praemisse sos oder
	 * der Konklusion goal vorkommen.
	 * 
	 * @param sos Premisse als Liste von Nodes / Formeln
	 * @param goal Konklusion als Node / Formel
	 * @return Prover9-Formel fuer die UNA
	 */
	private String getUniqueNameAssumption( List<? extends Node> sos, List<? extends Node> goals ) {
		// Menge der in sos und goal vorkommenden Konstanten ermitteln
		Set<String> constants = new HashSet<String>();
		for( Node node : sos ) {
			constants.addAll( node.getConstants() );
		}
		for( Node node : goals ) {
			constants.addAll( node.getConstants() );
		}
		
		if( constants.size() < 2 ) {
			// UNA ist trivialerweise schon erfuellt
			return "";
		}
		String una = "";
		
		Iterator<String> outerIterator = constants.iterator();
		while( outerIterator.hasNext() ) {
			String constant = outerIterator.next();
			outerIterator.remove();
			for( String constant2 : constants ) {
				una += "\"" + constant + "\"" + "!=" + "\"" + constant2 + "\".\n";
			}
		}
		
		
//		Bei kleinen Eingaben verursacht die Definition von distinct eine endlos Rekursion im Prover9 bei erhoehtem Wert von max_weight
//		Deshalb wurde auf die unten angegebene Version der UNA verzichtet
//		// Funktion 'distinct' definieren
//		una += "distinct([First, Second : Rest]) -> ( (First != Second) & distinct([First : Rest]) & distinct([Second : Rest])).\n";
//		// Liste mit den Konstanten erstellen
//		String constantlist = "";
//		String sep = "";
//		for(String s : constants) {
//			constantlist += sep + "\"" + s + "\"";
//			sep = ",";
//		}
//		// distinct-Funktion auf die Konstanten-Liste anwenden
//		una += "distinct([" + constantlist + "]).\n";
		
		return una;
	}
	
	/**
	 * Generiert Formeln fuer die active domain eines uebergebenen Completeness Sentence.
	 * 
	 * @param cs Completeness Sentence, zu dem die active domain gebildet werden soll
	 * @param activeDomainIndex Index fuer die active domain, um die active domains verschiedener CS unterscheiden zu koennen
	 * @return Formeln, die die active domain des uebergebenen Completeness Sentence beschreiben
	 */
	private String generateActiveDomainCS( CompletenessSentenceFormula cs, int activeDomainIndex ) {
		String activeDomainCS = "";
		
		Set<String> constants = cs.getSubstitutionConstants();
		// fuer jede Konstante c, fuege AD_index(c) hinzu
		for( String constant : constants ) {
        	activeDomainCS += "AD" + activeDomainIndex + "(\"" + constant + "\").\n";
        }
		
		// Vollstaendigkeitssatz fuer alle anderen Konstanten, die nicht in der active domain von phi sind
        String completenessSentence = "all X ((-AD" + activeDomainIndex + "(X))";
        for( String constant : constants ) {
        	completenessSentence += " | X=\"" + constant + "\"";
        }
        completenessSentence += ").\n";
        
        return activeDomainCS + completenessSentence;
	}
	
	/**
	 * Zerlegt einen uebergebenen Completeness Sentence in einzelne Formeln.
	 * Dabei wird fuer jede nicht ausgeschlossene Kombination von Konstanten der
	 * active domain des Completeness Sentence die Formel phi negiert und in die
	 * zurueckgegebene Formel integriert.
	 * 
	 * @param cs Completeness Sentence
	 * @return Formel, die den umgewandelten Completeness Sentence enthaelt
	 * @throws ProverException
	 */
	private String generateNegatedFormulasFromCompletenessSentence( CompletenessSentenceFormula cs ) throws ProverException {
		Set<String> freeVars = cs.getInteraction().getFreeVariables();
		int freeVarCount = freeVars.size();
		
		// generiere alle moeglichen Kombinationen der Konstanten des Completeness Sentence der Groesse freeVarCount
		String[] arr_constants = new String[0];
		arr_constants = cs.getSubstitutionConstants().toArray(arr_constants);
		Set<Map<String, String>> constantCombinations = new HashSet<Map<String, String>>();
		int[] lastCombination = new int[freeVarCount];
		for( int i=0; i<freeVarCount; ++i ) {
			lastCombination[i] = 0;
		}
		outerloop:
		while( lastCombination[0] < arr_constants.length ) {
			Map<String, String> combination = new HashMap<String, String>();
			int i = 0;
			for( String freeVar : freeVars ) {
				combination.put( freeVar, arr_constants[lastCombination[i++]] );
			}
			constantCombinations.add(combination);
			
			i = freeVarCount-1;
			while( ++lastCombination[i] >= arr_constants.length ) {
				lastCombination[i] = 0;
				--i;
				if( i < 0 )
					break outerloop;
			}
		}
		
		// entferne alle Kombinationen, die vom Completeness Sentence ausgeschlossen werden
		constantCombinations.removeAll(cs.getSubstitutions());
		
		// erstelle fuer alle uebrig gebliebenen Kombinationen eine Formel NOT phi(X_1, ..., X_n)
		String negatedFormulas = "";
		for( Map<String, String> combination : constantCombinations ) {
			Formula f = new Formula(cs.getInteraction());
    		f.substituteVariables(combination);
    		f.negate();

    		ProverNineFormula p9formula = new ProverNineFormula(f);
    		negatedFormulas += p9formula.toString() + ".\n";
		}
		
		return negatedFormulas;
	}
	
	/**
	 * Generiert Formeln, die fuer alle Kombinationen, die mindestens eine Konstante enthalten, die nicht
	 * in der active domain des Completeness Sentence ist, besagen, dass phi von dieser Kombination nicht gilt.
	 * Dabei ist phi die im Completeness Sentence gekapselte Formel. 
	 * 
	 * @param cs Completeness Sentence
	 * @param activeDomainIndex Index fuer die active domain, um die active domains verschiedener CS unterscheiden zu koennen
	 * @return Formel
	 */
	private String generateNegatedFormulasForActiveDomainComplement( CompletenessSentenceFormula cs, int activeDomainIndex ) {
        String negatedActiveDomainComplement = "";
		
		Set<String> freeVars = cs.getInteraction().getFreeVariables();
		
		for( String freeVar : freeVars ) {
        	Node phi = cs.getInteraction().clone();
            NegationNode negNode = new NegationNode( phi );
            AtomPredicateNode predNode = new AtomPredicateNode( "AD" + activeDomainIndex );
            predNode.addParameter( new VariableNode(freeVar) );
            DisjunctorNode disjNode = new DisjunctorNode();
            disjNode.addOperand( predNode );
            disjNode.addOperand( negNode );
            ArrayList<VariableNode> quantifiedVariables = new ArrayList<VariableNode>();
            for( String quanVar : freeVars ) {
            	quantifiedVariables.add( new VariableNode(quanVar) );
            }
            UniversalQuantifiedNode quantorNode = new UniversalQuantifiedNode(quantifiedVariables, disjNode);
            ProverNineFormula p9formula = new ProverNineFormula(quantorNode);
            negatedActiveDomainComplement += p9formula.toString() + ".\n";
        }
		
		return negatedActiveDomainComplement;
	}
	
	
	/**
	 * Fuehrt den Prover9-Aufruf durch und gibt dessen Resultat weiter.
	 * Diese Klasse kann von einem Executor als Callable ausgefuehrt werden, 
	 * sodass ein Zugriff auf den Rueckgabewert des Prover9 ueber das vom 
	 * Executor zurueckgelieferte Future-Objekt moeglich ist.
	 */
	private static class ProverNineTask implements Callable<ProverResult> {
		
		/**
		 * Eingabe fuer den Prover9
		 */
		String input;
		
		/**
		 * Konstruktor. Erstellt einen neuen ProverNineTask mit dem uebergebenen
		 * String als Eingabe fuer den Prover9.
		 * 
		 * @param input Eingabe fuer den Prover9
		 */
		public ProverNineTask(String input) {
			this.input = input;
		}
		
		/**
		 * Fuehrt den Prover9-Aufruf durch und gibt dessen Resultat zurueck.
		 * 
		 * @throws ProverException bei Auftritt eines Fehlers
		 */
		@Override
		public ProverResult call() throws ProverException {
			int exitCode = -1;
			Process p = null;
			
			StringBuilder output = new StringBuilder();
			try {
				// Prover9 Programm starten
				ProcessBuilder builder = new ProcessBuilder( PROVER9_EXE );
			    p = builder.start();
			    
			    // Eingabe in stdin von Prover9 schreiben
			    OutputStreamWriter prover9stdin = new OutputStreamWriter( p.getOutputStream() );
			    String line;
			    BufferedReader br = new BufferedReader( new StringReader(input) );
			    while ( (line = br.readLine()) != null ) {
			    	prover9stdin.write( line );
			    	logger.debug( line );
			    }
			    prover9stdin.close();
			    
			    // FIXME: warum ist das noetig?
			    if ( System.getProperty("os.name").contains("Windows") ) {
			    	Scanner s = new Scanner( p.getInputStream() ).useDelimiter( "\\Z" );
			    	s.next();
			    	//System.out.println( s.next() );
			    }
			    
//			    // InputStream leeren, weil dieser sonst ueberlaeuft und der Prover nicht mehr terminiert
			    InputStream inputStream = p.getInputStream();
			    BufferedReader bufferedReader = new BufferedReader( new InputStreamReader(inputStream) );
			    while( (line = bufferedReader.readLine()) != null ) {
//			    	// nix tun, nur den InputStream leeren...
			    	System.out.println("out: " + line);
			    	output.append(line + "\n");
			    }

			    // Auf Programmende warten und exit code speichern
			    p.waitFor();
			    exitCode = p.exitValue();
			}
			catch ( IOException e ) {
		        // Print out the exception that occurred
				throw new ProverException("Cought IO-Error while executing Prover9", e );
		    }
			catch ( InterruptedException e ) {
				if( p != null )
					p.destroy();
				//throw new ProverException( "Prover9 execution was interrupted.", e );
			}
			
			boolean result;
			switch( exitCode ) {
				case 0:
					// Implikation wahr
					result =  true;
					break;
				case 2:
					// Implikation nicht wahr
					result = false;
					break;				
				case 1: throw new ProverException("A fatal error occurred (user's syntax error or Prover9's bug)", exitCode );
				case 3: throw new ProverException("The max_megs (memory limit) parameter was exceeded.", exitCode );
				case 4: throw new ProverException("The max_seconds parameter was exceeded.", exitCode );
				case 5: throw new ProverException("The max_given parameter was exceeded.", exitCode );
				case 6: throw new ProverException("The max_kept parameter was exceeded.", exitCode );
				case 7: throw new ProverException("A Prover9 action terminated the search.", exitCode );
				case 101: throw new ProverException("Prover9 received an interrupt signal.", exitCode );
				case 102: throw new ProverException("Prover9 crashed, most probably due to a bug.", exitCode );
				default: throw new ProverException("Prover9 mit fatalem Fehler beendet.", exitCode );
			}
			
			ProverNineResult test = new ProverNineResult(result, output.toString());
			System.out.println("Start Proof:");
			try {
			System.out.print(test.getProof());
			}
			catch( ProverException e) {
				
			}
			System.out.println("End Proof");
			
			return new ProverNineResult(result, output.toString());
		}
	}
}
