package server.parser.node;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;

import javax.swing.tree.DefaultMutableTreeNode;

import org.apache.log4j.Logger;

import exception.DatabaseException;
import exception.IllegalFormulaOperationException;
import exception.UnsupportedFormulaException;

import server.database.cache.AppDatabaseSchemaCache;
import server.database.cache.Attribute;
import server.parser.Formula;
import server.parser.Formatter.FormulaFormatter;
import server.parser.Formatter.StandardFormatter;
import server.util.CloneableWithCloneMethod;
import server.util.Tuple;


public abstract class Node implements CloneableWithCloneMethod<Node>, Serializable {
	private static final long serialVersionUID = -61459759607116728L;
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.parser");


	// Schlüsselbegriffe bei der Umwandlung von einer Formel in einen String (toString()).
	public static final String SPACE = " "; 
	public static final String SEPARATOR = ",";
	public static final String LEFT_P 	= "(";
	public static final String RIGHT_P 	= ")";
	public static final String STRING_EXP = "\"";	
	public static final String NEGATION = "NOT";
	public static final String JUNCTOR_AND = "AND";
	public static final String JUNCTOR_OR = "OR";
	public static final String JUNCTOR_IMPL = "IMPL";
	public static final String JUNCTOR_EQUIV = "EQUIV";
	public static final String QUANTOR_EXISTS = "EXISTS";
	public static final String QUANTOR_FORALL = "FORALL";
	public static final String KRIPKE_OPERATOR = "K";

	/**
	 * Elter des aktuellen Knoten.
	 */
	private Node parent;

	/**
	 * Kinder des Knotens
	 */
	private List<Node> children = new ArrayList<Node>();
	
	/**
	 * variableCounter is used to generate unused variables
	 * for this node-tree. It is only used by FormulaUtil via the appropriate
	 * get() and set() methods.
	 */
	private int variableCounter = 0;
	
	
	

	/**
	 * Leerer Default-Konstruktor, der von jeder Kindklasse aufgerufen wird.
	 */
	protected Node() {
	}

	@Override
	public Node clone() {
		Node copy = null;

		try {
			copy = (Node)super.clone();
		} catch (CloneNotSupportedException e) {
			logger.fatal("Failed to clone node. Should not happen!");
			e.printStackTrace();
		}

		// Tiefe Kopie der Kindliste anlegen.
		List<Node> newChildren = new ArrayList<Node>();
		for( Node child : copy.getChildren() ) {
			newChildren.add( child.clone() );
		}

		// Alte flache Kopie loeschen und neue Kopien der Kinder einfuegen.
		copy.children = new ArrayList<Node>();
		copy.addChildren( newChildren );

		return copy;
	}
	
	/**
	 * Returns the counter for variable generation.
	 * @return
	 */
	public int getVariableCounter() {
		return this.variableCounter;
	}
	
	/**
	 * Sets the counter for variable generation to a new value.
	 * @param count
	 */
	public void setVariableCounter( int count ) {
		this.variableCounter = count;
	}
	
	
	/**
	 * Löst Implikationen auf: A IMPL B --> NOT A OR B
	 */
	public void transformImplication() {
		// Rekursiver Aufruf.
		List<Node> children = new ArrayList<Node>( this.getChildren() );
		for ( Node child : children ) {
			child.transformImplication();
		}
	}
	
	/**
	 * Löst Äquivalenzen auf: A EQUIV B --> (NOT A OR B) AND (NOT B OR A)
	 */
	public void transformEquivalence() {
		// Rekursiver Aufruf.
		List<Node> children = new ArrayList<Node>( this.getChildren() );
		for ( Node child : children ) {
			child.transformEquivalence();
		}
	}
	
	/**
	 * Schiebt Negationen direkt vor Atome. Dabei werden doppelte Negationen aufgeloest,
	 * Negationen vor Quantoren fueren zur Umwandlung des Quantors in den Gegenpart,
	 * Negationen vor Junktoren fuehren ebenfalls zur Umwandlung des Junktors in den Gegenpart.
	 * 
	 * Die Formeln MUESSEN ohne Implikation und Aequivalenz vorliegen.
	 * 
	 * @throws UnsupportedFormulaException Wird erzeugt, falls der Teilbaum Implikationen oder Aequivalenzen enthaelt.
	 */
	public void pushNegations() throws UnsupportedFormulaException {
		this.pushNegations( false, false );
	}
	
	/**
	 * Wie pushNegations(), jedoch werden Existenzquantoren mit vorangehender Negation nicht in den Gegenpart umgewandelt.
	 * 
	 * @see #pushNegations()
	 */
	public void pushNegationsSRNF() throws UnsupportedFormulaException {
		this.pushNegations( false, true );
	}
	
	/**
	 * @param foundNegation Gibt an, ob bereits eine Negation gefunden wurde. Entsprechend muessen Quantoren und Junktoren umgewandelt und doppelte Negationen entfernt werden.
	 * @param skipNegatedExistentialQuantifiers Falls true, wird ein Existenzquantor mit vorangehender Negation _nicht_ in den Gegenpart umgeformt
	 * @throws UnsupportedFormulaException Wird erzeugt, falls der Teilbaum Implikationen oder Aequivalenzen enthaelt.
	 * 
	 * @see #pushNegations()
	 * @see #pushNegationsSRNF()
	 */
	protected void pushNegations( boolean foundNegation, boolean skipNegatedExistentialQuantifiers ) throws UnsupportedFormulaException {
		throw new UnsupportedFormulaException(this.toString());
	}

	/**
	 * Die Formel wird so umgeformt, dass jedes AND, OR und EXISTS kein Kind mit dem gleichen Typ hat.
	 * Zudem werden alle Klammern eliminiert.
	 * 
	 * Die Default-Implementierung in Node ruft einfach rekursiv flatten() auf allen Kindern auf.
	 */
	public Node flatten() {
		ArrayList<Node> newChildren = new ArrayList<Node>();

		for ( Node child : this.getChildren() ) {
			newChildren.add( child.flatten() );
		}

		this.replaceChildren( newChildren );

		return this;
	}

	/**
	 * Die Methode rr() berechnet die Menge der range restricted Variablen.
	 * Wird zur Umwandlung einer Formel in RANF benoetigt.
	 * 
	 * @return Menge der range restricted Variablen.
	 */
	public Set<String> rr() {
		throw new UnsupportedOperationException("Unsupported");
	}

	/**
	 * Fuegt ein neues Kind am Ende der Kindliste hinzu.
	 * @param node Eine Kopie der uebergebenen Node wird als Kind hinzugefuegt.
	 */
	public void addChild(Node node) {
		addChild(node, children.size());
	}

	/**
	 * Fuegt eine flache(!) Kopie eines Knoten als neues Kind an einer bestimmten Stelle hinzu.
	 * @param node Node, von der eine flache(!) Kopie erzeugt wird, die dann in die Liste der Kinder aufgenommen wird.
	 * @param i Stelle, an der das Kind aufgenommen werden soll.
	 */
	public void addChild(Node node, int i) {
		if ( node.isRootNode() ) {
			node = ((Formula)node).getRootChild();
		}
		
		this.getChildren().add(i, node);
		node.setParent(this);
	}

	/**
	 * Fuege eine Reihe von Nodes am Ende der Kindliste hinzu.
	 * @param nodes Liste von neuen Kindern.
	 */
	public void addChildren( List<Node> nodes ) {
		for ( Node node : nodes ) {
			this.addChild( node );
		}
	}

	/**
	 * Fuegt eine Menge von Kindern in beliebiger Reihenfolge hinzu (z.B. fuer Konjunktion oder Disjunktion).
	 * @param nodes Menge der Kinder, die hinzugefuegt werden sollen.
	 */
	public void addChildren( Set<Node> nodes ) {
		for ( Node node : nodes ) {
			this.addChild( node );
		}
	}

	public List<Node> getChildren() {
		return children;
	}

	public int getChildrenNumber() {
		return children.size();
	}

	public Node getChild(int i) {
		return children.get(i);
	}
    
	/**
	 * Die uebergebenen Kinder werden zu den aktuellen Kindern und bekommen 
	 * das Knoten als parent in addChild.
	 * Die parent-Zeiger der aktuellen Kinder werden NICHT modifiziert und zeigen weiterhin 
	 * auf diesen Knoten. Diese sollten anschließend manuell auf NULL gesetzt werden.
	 * 
	 * @param children
	 */
	public void replaceChildren( List<Node> children ) {
		this.children = new ArrayList<Node>();
		this.addChildren( children );    	
	}
	
	/**
	 * Ersetzt das aktuelle Kind oldChild durch das neue Kind newChild.
	 * Der parent-Zeiger des aktuellen Kinds wird NICHT modifiziert und zeigt weiterhin 
	 * auf diesen Knoten. Dieser sollte anschließend manuell auf NULL gesetzt werden.
	 * 
	 * @param children
	 */
	public boolean replaceChild(Node oldchild, Node newChild) {
		boolean retVal = false;
		int index = children.indexOf(oldchild);
		retVal = children.remove(oldchild);
		this.addChild(newChild, index);
		return retVal;
	}

	/**
	 * Loescht das Kind an dem angegebenen Index.
	 * Der parent-Zeiger des aktuellen Kindes wird NICHT modifiziert und zeigt weiterhin 
	 * auf diesen Knoten. Dieser sollte anschließend manuell auf NULL gesetzt werden.
	 * 
	 * @param index
	 */
	public void removeChild(int index) {
		this.children.remove(index);
	}

	/**
	 * Loescht das uebergebene Kind.
	 * Der parent-Zeiger des Kindes wird NICHT modifiziert und zeigt weiterhin 
	 * auf diesen Knoten. Dieser sollte anschließend manuell auf NULL gesetzt werden.
	 * 
	 * @param child
	 */
	public void removeChild(Node child) {
		this.children.remove(child);
	}

	/**
	 * Loescht alle Kinder des aktuellen Knotens.
	 * Die parent-Zeiger der aktuellen Kinder werden NICHT modifiziert und zeigen weiterhin 
	 * auf diesen Knoten. Diese sollten anschließend manuell auf NULL gesetzt werden.
	 */
	public void removeAllChildren() {
		this.children.clear();
	}

	/**
	 * Ersetzt den aktuellen Knoten im Syntaxbaum durch einen neuen.
	 * @param newNode Der neue Knoten, der an die Stelle dieses Knotens gesetzt wird.
	 * @throws IllegalFormulaOperationException 
	 */
	public void replace( Node newNode ) {
		Node parent = this.getParent();
		if ( parent == null ) {
			// Wir sind im Wurzelknoten. Diesen zu ersetzen macht keinen Sinn.
			throw new IllegalFormulaOperationException("Error while replacing node: root node can not be replaced!", this);
		}

		parent.replaceChild( this, newNode );
	}

	/**
	 * Liefert den Elternknoten zurueck. Bei Formula-Objekten wird null zurueckgegeben.
	 * @return Elternknoten von diesem Knoten.
	 * @throws IllegalFormulaOperationException falls kein Elternknoten existiert (jede Formel/Node muss zumindest ein Formula-Objekt als Root-Node besitzen)
	 */
	public Node getParent() throws IllegalFormulaOperationException {
		if( this.parent == null ) {
			throw new IllegalFormulaOperationException("Every formula needs a Formula-Object as root node! Check your code!", this);
		}
				
		return parent;
	}

	/**
	 * Setzt einen neuen Elternknoten.
	 * @param parent Neuer Elternknoten.
	 * @throws IllegalFormulaOperationException Wird geworfen, falls der Parent eines Formula-Objekts gesetzt werden soll. 
	 */
	public void setParent(Node parent) throws IllegalFormulaOperationException {
		this.parent = parent;
	}

	/**
	 * Gibt die Menge aller Variablen (egal ob gebunden oder nicht) zurueck, die in diesem Baum vorkommen.
	 * 
	 * @return Die Menge aller Variablen (frei und gebunden) dieses (Teil-)Baums.
	 */
	public Set<String> getVariables() {
		Set<String> vars = new HashSet<String>();

		for ( Node child : this.getChildren() ) {
			vars.addAll( child.getVariables() );
		}

		return vars;
	}


	/**
	 * Gibt die Menge aller freien Variablen zurueck. Standardmaessig wird diese Methode rekursiv
	 * fuer jedes Kind des Knotens aufgerufen. Hat ein Knoten direkt freie Variablen (wie z.B.
	 * eine AtomPredicateNode), so werden diese der Menge hinzugefuegt. Quantoren (QuantifiedNode)
	 * hingegen bewirken, dass eine Variable aus der Menge entfernt wird.
	 * 
	 * @return Die Menge der freien Variablen dieses (Teil-)Baums.
	 */
	public Set<String> getFreeVariables() {
		Set<String> free = new HashSet<String>();

		for ( Node child : this.getChildren() ) {
			free.addAll( child.getFreeVariables() );
		}

		return free;
	}

	/**
	 * Gibt die Menge aller gebundenen Variablen zurueck. Standardmaessig wird diese Methode
	 * rekursiv fuer jedes Kind des Knotens aufgerufen. Bei jeder QuantifiedNode wird die gebundene
	 * Variable eingesammelt. Wird diese Variable in einem Teilbaum nicht verwendet, so kommt
	 * diese auch nicht vor. Beispiel: EXISTS Z P(X)  hat {} als Menge der gebundenen Variablen
	 * und {X} als Menge der freien Variablen.
	 * 
	 * @return Die Menge der gebundenen Variablen dieses (Teil-)Baums.
	 */
	public Set<String> getBoundVariables() {
		Set<String> bound = new HashSet<String>();

		for ( Node child : this.getChildren() ) {
			bound.addAll( child.getBoundVariables() );
		}

		return bound;
	}

	/**
	 * Liefert zurueck, ob eine Formel geschlossen (keine freien Variablen) ist.
	 * 
	 * @return true, falls Formel geschlossen ist, sonst false.
	 */
	public boolean isClosedFormula() {
		return this.getFreeVariables().isEmpty();
	}

	/**
	 * Liefert zurueck, ob eine Formel offen (es existiert mind. eine Variable) ist.
	 *
	 * @return true, falls Formel offen ist, sonst false.
	 */
	public boolean isOpenFormula() {
		return !this.getFreeVariables().isEmpty();
	}

	/**
	 * Gibt zurueck, ob es sich bei dem Knoten um die Wurzel handelt.
	 * 
	 * @return true, falls es sich um den Wurzelknoten (Formula) handelt, sonst false.
	 */
	public boolean isRootNode() {
		return false;
	}

	public boolean isVariableNode() {
		return false;
	}

	public boolean isConstantNode() {
		return false;
	}

	public boolean isRelationnameNode() {
		return false;
	}

	public boolean isAtomPredicateNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine konjunktive Formel repraesentiert
	 * */
	public boolean isConjunctorNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine disjunktive Formel repraesentiert
	 * */
	public boolean isDisjunctorNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine implizierte Formel repraesentiert
	 * */
	public boolean isImplicationNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine aequivalente Formel repraesentiert
	 * */
	public boolean isEquivalenceNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine quantifizierte Formel repraesentiert
	 * */
	public boolean isQuantifiedNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine universell quantifizierte Formel repraesentiert
	 * */
	public boolean isUniversalQuantifiedNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine existiell quantifizierte Formel repraesentiert
	 * */
	public boolean isExistentialQuantifiedNode() {
		return false;
	}

	public boolean isIntegerNode() {
		return false;
	}

	public boolean isStringNode() {
		return false;
	}

	public boolean isBooleanNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine verknuepfte Formel repraesentiert
	 * */
	public boolean isConnectorNode() {
		return false;
	}

	/**
	 * Praesentiert der Knoten eine atomare Wissensformel
	 * */
	public boolean isAtomicKripkeNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine negierte Formel repraesentiert
	 * */
	public boolean isNegationNode() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine elementare Formel repraesentiert. Wird verwendet, um Klammern nicht
	 * doppelte oder um Atome zu setzen.
	 */
	public boolean isElementaryFormula() {
		return false;
	}

	/**
	 * Gibt an, ob der Knoten eine Gleichheits-Formel repraesentiert
	 * */
	public boolean isEqualityNode() {
		return false;
	}
	
	/**
	 * Gibt an, ob die Formel ein Vollstaendigkeitssatz ist.
	 * 
	 * @return true, wenn die Formel ein Vollstaendigkeitssatz ist, false sonst
	 */
	public boolean isCompletenessSentenceFormula() {
		return false;
	}


	/**
	 * Ersetzt Variablennamen nach der uebergebenen Abbildung.
	 * 
	 * @param mapping Bijektive Abbildung alte Variablennamen -> neue Variablennamen.
	 */
	public void renameVariables(Map<String,String> mapping) {
		for ( Node child : this.getChildren() ) {
			child.renameVariables(mapping);
		}
	}

	/**
	 * Wandelt die Formel als Syntaxbaum in textuelle Darstellung um
	 * @param formulaformatter Formatiert die Formel nach einem bestimmten Ausgabeformat.
	 */
	public abstract String toString(FormulaFormatter formulaformatter);


	public String toString() {
		return toString(new StandardFormatter()); // + " {" +getClass().getName()+"}";
	}

	public void createTreeDump(DefaultMutableTreeNode top) {
		String treeValue = this.getClass().getSimpleName();

		// Bei Konstanten, Variablen und Relationen zusätzlich den Wert anzeigen.
		if ( this instanceof RelationnameNode ) {
			treeValue = treeValue + ": " + ((RelationnameNode)this).getRelationname();
		} else if ( this instanceof VariableNode ) {
			treeValue = treeValue + ": " + ((VariableNode)this).getVariable();
		} else if ( this instanceof ConstantNode ) {
			treeValue = treeValue + ": " + ((ConstantNode)this).getConstant();
		} else if ( this instanceof QuantifiedNode ) {
			treeValue = treeValue + ": " + ((QuantifiedNode)this).getQuantifiedVariables();
		}

		DefaultMutableTreeNode thisNode = new DefaultMutableTreeNode( treeValue );
		top.add( thisNode );
		for ( Node child : this.getChildren() ) {
			child.createTreeDump( thisNode );
		}
	}


	/**
	 * Ersetzt die durch die im Key als String spezifizierten freien Variablen durch 
	 * die als Value angegebene ConstantNode.
	 * 
	 * @param mapping Abbildung: Varialbenname -> Konstante
	 */
	public void substituteVariables(Map<String, String> mapping) {
		for ( Node child : new ArrayList<Node>(this.getChildren()) ) {
			child.substituteVariables( mapping );
		}
	}
	
	/**
	 * Substitutes one variable with a constant from a dictionary that is linked
	 * to the specific column. If no dictionary is found the String "FIXME" will be
	 * inserted.
	 * 
	 * @param var Name of the variable to be substituted.
	 */
	public void substituteVariable( AppDatabaseSchemaCache schema, String var ) throws DatabaseException {
		this.substituteVariable( schema, new Tuple<String,String>(var,null) );
	}
	
	/**
	 * Recursive call: Tries to find the variable in an AtomPredicateNode. If a variable
	 * is found for the first time, the second entry of the tupel will be set to the
	 * substituted value and will be used in all further substitutions. Until then the
	 * second entry is null.
	 * 
	 * @param mapping
	 */
	protected void substituteVariable( AppDatabaseSchemaCache schema, Tuple<String,String> mapping ) throws DatabaseException {
		for ( Node child : this.getChildren() ) {
			child.substituteVariable( schema, mapping );
		}
	}
	

	/**
	 * Negiert jedes Vorkommen eines uebergebenen Literals.
	 * 
	 * Veraendert den zugrundeliegenden Syntaxbaum!
	 * Der aktuelle Knoten wird NICHT veraendert! Man sollte also mit einer
	 * Formula starten, um sicherzustellen, dass alles passt.
	 * 
	 * @see #neg(AtomPredicateNode)
	 * @param literal Literal, dessen beinhaltetes Atom im Syntaxbaum nergiert werden soll.
	 * @return Neuer Syntaxbaum, in dem das in dem Literal enthaltene Atom negiert wurde.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn der uebergebene Knoten kein Literal ist.
	 */
	protected Node negateLiteral( Node literal ) throws UnsupportedFormulaException {
		return this.negateLiteral( Node.getAtomFromLiteral(literal) );
	}

	/**
	 * Negiert jedes Vorkommen einer uebergebenen AtomPredicateNode.
	 * Wird in der Literatur haeufig als "Variable Negation" beschrieben.
	 * 
	 * Veraendert den zugrundeliegenden Syntaxbaum!
	 * Der aktuelle Knoten wird NICHT veraendert! Man sollte also mit einer
	 * Formula starten, um sicherzustellen, dass alles passt.
	 * 
	 * @param atom Atom, das im Syntaxbaum negiert werden soll.
	 * @return Neuer Syntaxbaum, in dem das Atom negiert wurde.
	 */
	protected Node negateLiteral( AtomPredicateNode atom ) {
		ArrayList<Node> newChildren = new ArrayList<Node>();

		for ( Node child : this.getChildren() ) {
			if ( child.isAtomPredicateNode() ) {
				// Beim richtigem Atom NegationNode einfuegen.
				if ( child.equals(atom) ) {
					logger.debug(child + " vs. " + atom);

					// Wenn wir selbst eine Negation sind direkt das Atom zurueckgeben (einziges Kind).
					if ( this.isNegationNode() ) {
						return child;
					} else {
						// Ansonsten neue Negation an dieser Stelle einfuegen.
						newChildren.add( new NegationNode(child) );
					}
				} else {
					// Bei Atomen auf keinen Fall weiter rekursiv absteigen.
					newChildren.add( child );
				}
			} else {
				// Kein Atom als Kind => rekursiv negieren.
				newChildren.add( child.negateLiteral(atom) );
			}
		}

		// Kinder neu setzen (aktualisiert auch Parent-Verweis)
		this.replaceChildren( newChildren );

		return this;
	}

	/* (non-Javadoc) automatisch generierter Eclipse-Code.
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
		+ ((children == null) ? 0 : children.hashCode());
		return result;
	}

	/* (non-Javadoc) automatisch generierter Eclipse-Code.
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Node other = (Node) obj;
		if (children == null) {
			if (other.children != null)
				return false;
		} else if (!children.equals(other.children))
			return false;

		// Parent wird nicht ueberprueft, da es sonst eine endlose Rekursion geben wuerde (immer hoch
		// und runter im Baum).

		return true;
	}




	/**
	 * Gibt die Menge aller Relationennamen zurueck, die in diesem Baum vorkommen.
	 * 
	 * @return Die Menge aller Relationennamen dieses (Teil-)Baums.
	 */
	public Set<String> getRelationnames() {
		Set<String> relationnames = new HashSet<String>();

		for ( Node child : this.getChildren() ) {
			relationnames.addAll( child.getRelationnames() );
		}

		return relationnames;
	}

	/**
	 * Gibt die Menge aller Konstanten zurueck, die in diesem Baum vorkommen.
	 * 
	 * @return Die Menge aller Konstanten dieses (Teil-)Baums.
	 */
	public Set<String> getConstants() {
		Set<String> constants = new HashSet<String>();

		for ( Node child : this.getChildren() ) {
			constants.addAll( child.getConstants() );
		}

		return constants;
	}

	/**
	 * Gibt alle Kombinationen von Relationenname und Attribut zurueck, in denen
	 * die Variable var vorkommt.
	 * Beispiel:
	 * Formel p1(X, "c1", "c2, X, "c3") AND p2(Y, "c1", X) ergibt {"p1":{0,3}, "p2":{2}} fuer Parameter "X".
	 * @param var Variablenname, zu der die Relationennamen und Attribute ihrer Vorkommen ermittelt werden sollen
	 * @return Map aus Relationennamen und einer Attributmenge in denen die Variable var vorkommt
	 */
	public Map<String, SortedSet<Integer>> getRelationnamesAndAttributesContainingVar(String var) {
		Map<String, SortedSet<Integer>> relationnamesAndAttributes = new HashMap<String, SortedSet<Integer>>();

		for ( Node child : this.getChildren() ) {
			Map<String, SortedSet<Integer>> relAtt = child.getRelationnamesAndAttributesContainingVar(var);
			for( Map.Entry<String, SortedSet<Integer>> entry : relAtt.entrySet()) {
				if( relationnamesAndAttributes.containsKey(entry.getKey()) ) {
					relationnamesAndAttributes.get(entry.getKey()).addAll(entry.getValue());
				}
				else {
					relationnamesAndAttributes.put(entry.getKey(), entry.getValue());
				}
			}
		}

		return relationnamesAndAttributes;
	}
	
	/**
	 * Calculates the active domain values for the current formula. They
	 * will be added to the schema cache, which might not be empty beforehand.
	 * Managing the schema cache must be done within your part of the code ;)
	 * @param schema
	 */
	public void calculateActiveDomain( AppDatabaseSchemaCache schema ) {
		for ( Node child : this.getChildren() ) {
			child.calculateActiveDomain(schema);
		}
	}
	
	/**
	 * Calculates all possible values (taken from the active domain, which must 
	 * be calculated beforehand with {@link #calculateActiveDomain(AppDatabaseSchemaCache)})
	 * of each variable occuring in this syntax tree.
	 * @param schema
	 * @return
	 * @throws DatabaseException 
	 */
	public HashMap<String,Set<String>> calculateVariableMappingsForActiveDomain( AppDatabaseSchemaCache schema ) throws DatabaseException {
		HashMap<String,Set<String>> variableToActiveDomainMapping = new HashMap<String,Set<String>>();
		HashMap<String,Set<Attribute>> mapping = this.calculateVariableMappings(schema);
		
		for ( Map.Entry<String,Set<Attribute>> entry : mapping.entrySet() ) {
			Set<String> activeDomain = new HashSet<String>();
			for ( Attribute attribute : entry.getValue() ) {
				activeDomain.addAll( attribute.getActiveDomain() );
			}
			variableToActiveDomainMapping.put( entry.getKey(), activeDomain );
		}
		
		return variableToActiveDomainMapping;
	}
	
	/**
	 * Calculates all attributes occuring at a position of a variable for each variable
	 * occuring in this syntax tree.
	 * 
	 * @param schema
	 * @return
	 */
	public HashMap<String,Set<Attribute>> calculateVariableMappings( AppDatabaseSchemaCache schema ) {
		HashMap<String,Set<Attribute>> mapping = new HashMap<String,Set<Attribute>>();
		this.calculateVariableMappings( schema, mapping );
		return mapping;
	}
	
	/**
	 * Recursive algorithm to solve {@link #calculateVariableMappings(AppDatabaseSchemaCache)}.
	 * @param schema
	 * @param mapping
	 */
	protected void calculateVariableMappings( AppDatabaseSchemaCache schema, Map<String, Set<Attribute>> mapping ) {
		for ( Node child : this.getChildren() ) {
			child.calculateVariableMappings(schema, mapping);
		}
	}
	
	public static HashMap<String,Set<String>> calculateVariableMappingsForActiveDomain( AppDatabaseSchemaCache schema, Collection<Node> nodes ) throws DatabaseException {
		HashMap<String,Set<String>> variableToActiveDomainMapping = new HashMap<String,Set<String>>();
		
		// First calculate the active domain.
		schema.resetCustomActiveDomain();
		for ( Node node : nodes ) {
			node.calculateActiveDomain(schema);
		}
		
		// Now calculate the mapping for the active domain.
		for ( Node node : nodes ) {
			HashMap<String,Set<String>> mapping = node.calculateVariableMappingsForActiveDomain(schema);
			
			for ( Map.Entry<String,Set<String>> entry : mapping.entrySet() ) {
				if ( variableToActiveDomainMapping.containsKey(entry.getKey()) ) {
					variableToActiveDomainMapping.get(entry.getKey()).addAll( entry.getValue() );
				} else {
					variableToActiveDomainMapping.put( entry.getKey(), entry.getValue() );
				}
			}
		}
		
		// Now reset the active domain (we don't need it anymore).
		schema.resetCustomActiveDomain();
		
		return variableToActiveDomainMapping;
	}
	
	/**
	 * Gibt das in einem Literal enthaltene Atom zurueck.
	 * Ein Literal ist ein Atom oder ein negiertes Atom.
	 * Die Methode funktioniert ebenfalls mit Formula-Objekten,
	 * die ein Literal enthalten.
	 * 
	 * @param literal Knoten, der ein Literal sein muss.
	 * @return Das Atom, dass in dem Literal enthalten ist.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn es sich bei dem uebergebenen Knoten nicht um ein Literal handelt.
	 */
	public static AtomPredicateNode getAtomFromLiteral( Node literal ) throws UnsupportedFormulaException {
		if ( literal.isNegationNode() ) {
			Node negNode = ((NegationNode)literal).getNegatedFormula();
			if ( negNode.isAtomPredicateNode() ) {
				return (AtomPredicateNode)negNode;
			}
		} else if ( literal.isAtomPredicateNode() ) {
			return (AtomPredicateNode)literal;
		} else if ( literal.isRootNode() ) {
			return getAtomFromLiteral( ((Formula)literal).getRootChild() );
		}

		throw new UnsupportedFormulaException( "Not a literal: "+literal.toString() );
	}
}