package server.parser.node;

import java.util.ArrayList;
import java.util.List;

import exception.UnsupportedFormulaException;
import server.parser.Formula;
import server.parser.Formatter.FormulaFormatter;

/**
 * Wird nur fuer die Liste von Literalen bei Updates verwendet.
 * Stellt keine Formel in Sinne einer logischen Formel dar (also keine
 * Aussagen- oder Praedikatenlogik).
 * 
 * Die Kinder des Knoten sind in jedem Fall Literale (d.h. entweder direkt
 * eine AtomPredicateNode oder eine NegationNode mit einer AtomPredicateNode
 * als Kind.
 *
 */
public class LiteralSequenceNode extends Node {
	private static final long serialVersionUID = -960071167240321111L;
	
	/**
	 * Fuegt eine Liste von Literalen hinzu. Es wird ueberprueft, ob es sich bei der Liste
	 * wirklich um Literale handelt.
	 * @param literals
	 * @throws UnsupportedFormulaException
	 */
	public LiteralSequenceNode( List<Node> literals ) throws UnsupportedFormulaException {
		for ( Node literal : literals ) {
			// Pruefen, ob es sich wirklich um ein Literal handelt.
			if ( literal.isAtomPredicateNode() || (literal.isNegationNode() && ((NegationNode)literal).getNegatedFormula().isAtomPredicateNode()) ) {
				this.addChild( literal );
			} else {
				throw new UnsupportedFormulaException( literal.toString() );
			}
		}
	}
	
	/**
	 * Fuegt die Liste von Literalen ohne zusaetzliche Checks hinzu. Der Wert des Parameter nochecks wird ignoriert.
	 * @param literals
	 * @param nochecks Wird ignoriert.
	 */
	public LiteralSequenceNode( List<Node> literals, boolean nochecks ) {
		this.addChildren( literals );
	}
	
	/**
	 * Erstellt eine Kopie der als Liste vorliegenden Literale und wandelt sie dabei in gueltige Einzelformeln um.
	 * @return
	 */
	public List<Formula> getLiteralList() {
		ArrayList<Formula> literals = new ArrayList<Formula>();
		
		for ( Node child : this.getChildren() ) {
			literals.add( new Formula(child.clone()) );
		}
		
		return literals;
	}
	
	@Override
	public String toString(FormulaFormatter formulaformatter) {
		StringBuilder builder = new StringBuilder();
		builder.append( formulaformatter.getBeforeLiteralSequence() );
		
		String separator = "";
		for ( Node child : this.getChildren() ) {
			builder.append( separator );
			builder.append( child.toString(formulaformatter) );
			separator = SEPARATOR;
		}
		
		builder.append( formulaformatter.getAfterLiteralSequence() );
		return builder.toString();
	}
	
}
