package server.parser.node;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import exception.DatabaseException;

import server.database.cache.AppDatabaseSchemaCache;
import server.database.cache.Attribute;
import server.parser.Formatter.FormulaFormatter;
import server.util.Tuple;


public class AtomPredicateNode extends Node {
	private static final long serialVersionUID = -8112967923895757629L;
	
	public AtomPredicateNode( String relationname ) {
		this.addChild( new RelationnameNode(relationname) );
	}
	
	@Override
	public boolean isAtomPredicateNode() {
		return true;
	}
	
	public String getRelationname() {
		return ((RelationnameNode)this.getChild(0)).getRelationname();
	}
	
	public void setRelationname( String relationname ){
		((RelationnameNode)this.getChild(0)).setRelationname(relationname);
	}
	
	public List<TermNode> getParameterNodes() {
		List<TermNode> params = new ArrayList<TermNode>();
		
		for ( Node child : this.getChildren() ) {
			if ( child instanceof TermNode ) {
				params.add( (TermNode)child );
			}
		}
		
		return params;
	}
	
	public void addParameter(TermNode parameter) {
		this.addChild(parameter);
	}
	
	@Override
	public String toString(FormulaFormatter formulaformatter) {
		StringBuilder builder = new StringBuilder();
		builder.append(getRelationname());
		if (!getParameterNodes().isEmpty()) {
			builder.append(formulaformatter.getLeftParenthesis());
			Node firstParam = getParameterNodes().get(0);
			builder.append(firstParam.toString(formulaformatter));
			for (int i = 1; i < getParameterNodes().size(); i++) {
				builder.append(SEPARATOR);
				Node nextChild = getParameterNodes().get(i);
				builder.append(nextChild.toString(formulaformatter));
			}
			builder.append(formulaformatter.getRightParenthesis());
		}
		return builder.toString();
	}
	
	@Override
	public Set<String> getFreeVariables() {
		Set<String> free = new HashSet<String>();
		
		for ( Node parameter : this.getParameterNodes() ) {
			if ( parameter.isVariableNode() ) {
				free.add( ((VariableNode)parameter).getVariable() );
			}
		}
		
		return free;
	}
	
	/**
	 * Zu diesem Zeitpunkt die Menge der Variablen mit der Menge der
	 * freien Variablen identisch. Gebundene Variablen kann es auf Ebene
	 * der Atome nicht geben.
	 */
	@Override
	public Set<String> getVariables() {
		return this.getFreeVariables();
	}
	
	@Override
	public boolean isElementaryFormula() {
		return true;
	}
	
	@Override
	public Set<String> rr() {
		Set<String> rr = new HashSet<String>();
		
		for ( Node child : this.getChildren() ) {
			if ( child.isVariableNode() ) {
				rr.add( ((VariableNode)child).getVariable() );
			}
		}
	
		return rr;
	}
	
	/**
     * Gibt alle Kombinationen von Relationenname und Attribut zurück, in denen
     * die Variable var vorkommt.
     * Beispiel:
     * Für die Formel p("c1", "c2", X, "c4") liefert der Aufruf mit Parameter X "p -> 2" zurück.
     * @param var Variablenname, zu der die Relationennamen und Attribute ihrer Vorkommen ermittelt werden sollen
     * @return Map aus Relationennamen und einer Attributmenge in denen die Variable var vorkommt
     */
	@Override
    public Map<String, SortedSet<Integer>> getRelationnamesAndAttributesContainingVar(String var) {
    	Map<String, SortedSet<Integer>> relationnamesAndAttributes = new HashMap<String, SortedSet<Integer>>();
    	
    	SortedSet<Integer> attributes = new TreeSet<Integer>();
    	int i = 0;
    	for ( Node parameter : this.getParameterNodes() ) {
    		if ( parameter.isVariableNode() && ((VariableNode)parameter).getVariable().equals(var) ) {
    			attributes.add( i );
    		}
    		++i;
    	}
    	if( attributes.size() > 0)
    		relationnamesAndAttributes.put( getRelationname(), attributes );
    	
        return relationnamesAndAttributes;
    }
	
	@Override
	protected void pushNegations( boolean foundNegation, boolean skipNegatedExistentialQuantifiers ) {
		// Wir sind in einem Blatt des Baums.
		
		if ( foundNegation ) {
			// Es wurde vorher eine Negation gefunden, die jetzt vor dem Atom eingefuegt
			// werden muss.
			Node parent = this.getParent();
			NegationNode neg = new NegationNode(this);
			parent.replaceChild(this, neg);
		}
		
		// In jedem Fall sind wir an dieser Stelle fertig (kein rekursiver Aufruf).
	}
	
	@Override
	protected void substituteVariable( AppDatabaseSchemaCache schema, Tuple<String,String> mapping ) throws DatabaseException {
		List<Node> children = new ArrayList<Node>();
		
		List<Attribute> attributes = schema.getRelation( this.getRelationname() );
		Iterator<Attribute> attributeIter = attributes.iterator();
		
		for ( Node child : this.getChildren() ) {
			if ( child.isRelationnameNode() ) {
				children.add( child );
				continue;
			}
			
			Attribute attribute = attributeIter.next();
			
			if ( !child.isVariableNode() ) {
				children.add( child );
				continue;
			}
			
			if ( child.isVariableNode() ) {
				VariableNode varNode = (VariableNode)child;
				if ( varNode.getVariable().compareToIgnoreCase(mapping.getFirstElement()) == 0 ) {
					String constant = mapping.getSecondElement();
					// Get constant from dictionary if neccessary.
					if ( constant == null ) {
						constant = attribute.getPossibleValue();
					}
					children.add( new StringNode(constant) );
				} else {
					children.add( child );
				}
			}
		}
		
		this.replaceChildren( children );
	}
	
	@Override
	public void calculateActiveDomain( AppDatabaseSchemaCache schema ) {
		List<Attribute> attributes = schema.getRelation( this.getRelationname() );
		Iterator<Attribute> attributeIter = attributes.iterator();
		
		for ( Node child : this.getParameterNodes() ) {
			Attribute attribute = attributeIter.next();
			
			if ( child.isStringNode() ) {
				attribute.addCustomActiveDomainValue( ((StringNode)child).getConstant() );
			}
		}
	}
	
	@Override
	protected void calculateVariableMappings( AppDatabaseSchemaCache schema, Map<String, Set<Attribute>> mapping ) {
		List<Attribute> attributes = schema.getRelation( this.getRelationname() );
		Iterator<Attribute> attributeIter = attributes.iterator();
		
		for ( Node child : this.getParameterNodes() ) {
			Attribute attribute = attributeIter.next();
			
			if ( child.isVariableNode() ) {
				String var = ((VariableNode)child).getVariable();
				
				if ( mapping.containsKey(var) ) {
					mapping.get(var).add( attribute );
				} else {
					HashSet<Attribute> attrs = new HashSet<Attribute>();
					attrs.add( attribute );
					mapping.put(var, attrs );
				}
			}
		}
	}
}