package server.parser;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import exception.CompletenessSentenceParseException;

import server.parser.node.ConjunctorNode;
import server.parser.node.ConstantNode;
import server.parser.node.DisjunctorNode;
import server.parser.node.EqualityNode;
import server.parser.node.ImplicationNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.StringNode;
import server.parser.node.UniversalQuantifiedNode;
import server.parser.node.VariableNode;


public class CompletenessSentenceFormula extends Formula {

	private static final long serialVersionUID = 3228358122067164474L;
	
	protected Formula interaction;
	protected Set<Map<String, String>> substitutions;
	
	public CompletenessSentenceFormula( Formula interaction, Set<Map<String, String>> substitutions ) {
		this.interaction = interaction;
		this.substitutions = substitutions;
		if( substitutions.isEmpty() ) {
			this.formulaType = FormulaType.CS_WITHOUT_RULEOUT_COMBINATIONS;
		}
		else {
			this.formulaType = FormulaType.CS_WITH_RULEOUT_COMBINATIONS;
		}
		this.buildCompletenessSentence();
	}
	
	public CompletenessSentenceFormula( CompletenessSentenceFormula copy ) {
		super( copy );
		
		this.interaction = (Formula)copy.interaction.clone();
		this.substitutions = new HashSet<Map<String, String>>();
		for( Map<String, String> copySubstitution : copy.substitutions ) {
			Map<String, String> copiedSubstitution = new HashMap<String, String>();
			for( Map.Entry<String, String> copySubstitutionEntry : copySubstitution.entrySet() ) {
				copiedSubstitution.put(copySubstitutionEntry.getKey(), copySubstitutionEntry.getValue());
			}
			this.substitutions.add( copiedSubstitution );
		}
	}
	
	private void buildCompletenessSentence() {
		// freie Variable ermitteln
		Set<String> freeVars = this.interaction.getFreeVariables();
		
		// Formel fuer Complete(phi(X_1, ..., X_n), k) aufbauen
		// Ziel: FORALL X_1, ..., X_n ( ( [X_1 != c_x1_1 OR X_2 != c_x2_1 OR ... OR X_n != c_xn_1] AND
		//								  [X_1 != c_x1_2 OR X_2 != c_x2_2 OR ... OR X_n != c_xn_2] )
		//								=> NOT phi(X_1, ..., X_n) )
		
		// urspruengliche Formel negieren
		Node negatedInteraction;
		if( this.interaction.getRootChild().isNegationNode() ) {
			NegationNode negNode = (NegationNode) this.interaction.getRootChild();
			negatedInteraction = negNode.getNegatedFormula();
		}
		else {
			negatedInteraction = new NegationNode(this.interaction.getRootChild());
		}
		
		// zu quantifizierende Formel bilden
		Node toQuantify;
		if( this.substitutions.isEmpty() ) {
			// keine auszuschliessenden Konstanten vorhanden
			// => zu quant. Formel besteht nur aus der negierten urspruenglichen Anfrage
			toQuantify = negatedInteraction;
		}
		else {
			// mind. eine auszuschliessende Kombination vorhanden
			
			// Konjunktion erstellen: [X_1 != c_x1_1 OR X_2 != c_x2_1 OR ... OR X_n != c_xn_1] AND ... [...]
			ConjunctorNode conNode = new ConjunctorNode();
			DisjunctorNode disjNode;
			for( Map<String, String> substitution : this.substitutions ) {
				disjNode = new DisjunctorNode();
				for( String freeVar : freeVars ) {
					EqualityNode eqNode = new EqualityNode(new VariableNode(freeVar), new StringNode(substitution.get(freeVar)), false);
					
					if( freeVars.size() == 1 ) {
						// Disjunktion wuerde nur aus einem Operanden bestehen => Disjunktion direkt weglassen
						conNode.addOperand(eqNode);
					}
					else {
						disjNode.addOperand(eqNode);
					}
				}
				if( freeVars.size() == 1 ) {
					// nichts mehr zu tun, da die einzige Ungleichheit bereits in der Schleife zur conNode hinzugefuegt wurde
				}
				else {
					conNode.addOperand(disjNode);
				}
			}
			
			// Implikation erstellen
			ImplicationNode impNode = new ImplicationNode();
			impNode.addOperand(conNode);
			impNode.addOperand(negatedInteraction);
			
			toQuantify = impNode;
		}
		
		// ALL-Quantor vor zu quant. Formel haengen
		ArrayList<VariableNode> varList = new ArrayList<VariableNode>();
		for( String var : freeVars) {
			varList.add(new VariableNode(var));
		}
		UniversalQuantifiedNode quantorNode = new UniversalQuantifiedNode(varList, toQuantify);
		
		this.setRootChild( quantorNode );
	}
	
	public Formula getInteraction() {
		return this.interaction;
	}
	
	public Set<Map<String, String>> getSubstitutions() {
		return this.substitutions;
	}
	
	public void addSubstitution(Map<String, String> substitution) {
		if( this.substitutions.add(substitution) ) {
			// Substitutionen haben sich veraendert -> Formel neu aufbauen
			this.buildCompletenessSentence();
		}
	}
	
	public void addSubstitutions(Set<Map<String, String>> substitutions) {
		if( this.substitutions.addAll(substitutions) ) {
			// Substitutionen haben sich veraendert -> Formel neu aufbauen
			this.buildCompletenessSentence();
		}
	}
	
	public void removeSubstitution(Map<String, String> substitution) {
		if( this.substitutions.remove(substitution) ) {
			// Substitutionen haben sich veraendert -> Formel neu aufbauen
			this.buildCompletenessSentence();
		}
	}
	
	public void removeSubstitutions( Set<Map<String, String>> substitutions ) {
		if( this.substitutions.removeAll(substitutions) ) {
			// Substitutionen haben sich veraendert -> Formel neu aufbauen
			this.buildCompletenessSentence();
		}
	}
	
	/**
	 * Liefert eine Menge aller Konstanten zurueck, die in den Substitutionen vorkommen.
	 * 
	 * @return Menge der in den Substitutionen vorkommenden Konstanten
	 */
	public Set<String> getSubstitutionConstants() {
		Set<String> substitutionConstants = new HashSet<String>();
		for( Map<String, String> entry : substitutions ) {
			substitutionConstants.addAll( entry.values() );
		}
		
		return substitutionConstants;
	}
	
	@Override
	public boolean isCompletenessSentenceFormula() {
		return true;
	}
	
	
	private static Map<String, String> parseInequality( EqualityNode equalityNode ) throws CompletenessSentenceParseException {
		Map<String, String> varConstantMapping = new HashMap<String, String>();
		
		if( ! equalityNode.getEquals() && equalityNode.getLeftOperand().isVariableNode() && equalityNode.getRightOperand().isConstantNode() ) {
			// extrahiere Variable und Konstante
			VariableNode variableNode = (VariableNode) equalityNode.getLeftOperand();
			String variable = variableNode.getVariable();
			ConstantNode constantNode = (ConstantNode) equalityNode.getRightOperand();
			String constant = constantNode.getConstant();
			// fuege Variable und Konstante zum varConstantMapping hinzu
			varConstantMapping.put(variable, constant);
		}
		else {
			// ungueltiger CompletenessSentence
			throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula! Error while parsing inequality in substitutions: " + equalityNode.toString());
		}
		
		return varConstantMapping;
	}
	
	private static Map<String, String> parseSubstitution( Node substitution, int substitutionSize ) throws CompletenessSentenceParseException {
		Map<String, String> varConstantMapping = new HashMap<String, String>();
		
		if( substitution.isEqualityNode() && substitutionSize == 1 ) {
			EqualityNode equalityNode = (EqualityNode) substitution;
			varConstantMapping.putAll( parseInequality(equalityNode) );
		}
		else if( substitution.isDisjunctorNode() && substitutionSize > 1 ) {
			DisjunctorNode disjunctorNode = (DisjunctorNode) substitution;
			for( Node operand : disjunctorNode.getOperands() ) {
				if( operand.isEqualityNode() ) {
					EqualityNode equalityNode = (EqualityNode) operand;
					varConstantMapping.putAll( parseInequality(equalityNode) );
				}
				else {
					// ungueltiger CompletenessSentence
					throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula! Error while parsing substitution disjunction: " + disjunctorNode.toString());
				}
			}
		}
		else {
			// ungueltiger CompletenessSentence
			throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula! Error while parsing substitution: " + substitution.toString());
		}
		
		return varConstantMapping;
	}
		
	private static Set<Map<String, String>> parseSubstitutions(Node substitutionsNode, int substitutionSize) throws CompletenessSentenceParseException {
		Set<Map<String, String>> substitutions = new HashSet<Map<String,String>>();
		
		if( substitutionsNode.isDisjunctorNode() || substitutionsNode.isEqualityNode() ) {
			// nur eine auszuschliessende Kombination vorhanden (es ex. in diesem Fall keine umschliessende ConjunctorNode)
			substitutions.add( parseSubstitution(substitutionsNode, substitutionSize) );
		}
		else if( substitutionsNode.isConjunctorNode() ) {
			// mehrere auszuschliessende Kombinationen vorhanden
			ConjunctorNode conjunctorNode = (ConjunctorNode) substitutionsNode;
			for( Node substitution : conjunctorNode.getOperands() ) {
				substitutions.add( parseSubstitution(substitution, substitutionSize) );
			}
		}
		else {
			// ungueltiger CompletenessSentence
			throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula! Error while parsing substitutions: " + substitutionsNode.toString());
		}
		
		return substitutions;
	}
	
	private static Formula parseInteraction( Node interactionNode ) {
		// negiere interactionNode, da der Konstruktor der CompletenessSentenceFormula diese wieder zurueck negiert
		if( interactionNode.isNegationNode() ) {
			NegationNode negInteraction = (NegationNode) interactionNode;
			return new Formula( negInteraction.getNegatedFormula()) ;
		}
		else {
			interactionNode = new NegationNode(interactionNode);
			return new Formula(interactionNode);
		}
	}
	
	public static CompletenessSentenceFormula parseCompletenessSentenceFormula( Formula formula, boolean hasRuleOutCombinations ) throws CompletenessSentenceParseException {
		Set<Map<String, String>> substitutions = new HashSet<Map<String, String>>();
		Formula interaction = null;
		Node quantifiedFormula;
		Set<String> quantifiedVariables;
		
		// ermittle quantifizierte Variablen und quantifizierte Formel
		if( formula.getRootChild().isUniversalQuantifiedNode() ) {
			UniversalQuantifiedNode universalQuantifier =  (UniversalQuantifiedNode) formula.getRootChild();
			quantifiedFormula = universalQuantifier.getQuantifiedFormula();
			quantifiedVariables = universalQuantifier.getQuantifiedVariables();
		}
		else {
			// ungueltiger CompletenessSentence
			throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula: " + formula.toString());
		}
		
		// auszuschliessende Kombinationen vorhanden?
		if( hasRuleOutCombinations ) {
			// splitte formula auf in vorderen Teil, der die Substitutionen enthaelt und hinteren Teil, der die eigentliche interaction enthaelt
			Node substitutionsNode, interactionNode;
			if( quantifiedFormula.isImplicationNode() && quantifiedFormula.getChildrenNumber() == 2 ) {
				substitutionsNode = quantifiedFormula.getChild(0);
				interactionNode = quantifiedFormula.getChild(1);
			}
			else {
				// ungueltiger CompletenessSentence
				throw new CompletenessSentenceParseException("Unable to parse CompletenessSentenceFormula: " + formula.toString());
			}
			
			// parse Substitutionen
			substitutions = parseSubstitutions(substitutionsNode, quantifiedVariables.size());
			
			// parse interaction
			interaction = parseInteraction(interactionNode);
		}
		else {
			// keinen auszuschliessenden Substitutionen vorhanden
			interaction = parseInteraction(quantifiedFormula);
		}
		
		return new CompletenessSentenceFormula(interaction, substitutions);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		CompletenessSentenceFormula other = (CompletenessSentenceFormula) obj;
		if( ! this.interaction.equals(other.interaction) )
			return false;
		if( ! this.substitutions.equals(other.substitutions) )
			return false;

		return true;
	}
	
	@Override
	public CompletenessSentenceFormula clone() {
		CompletenessSentenceFormula clone = (CompletenessSentenceFormula)super.clone();
		
		clone.interaction = (Formula)this.interaction.clone();
		clone.substitutions = new HashSet<Map<String, String>>();
		for( Map<String, String> substitution : this.substitutions ) {
			Map<String, String> clonedSubstitution = new HashMap<String, String>();
			for( Map.Entry<String, String> substitutionEntry : substitution.entrySet() ) {
				clonedSubstitution.put(substitutionEntry.getKey(), substitutionEntry.getValue());
			}
			clone.substitutions.add( clonedSubstitution );
		}
		
		return clone;
	}
}
