package server.database.sql;

import java.util.LinkedList;
import java.util.List;

import server.database.schema.SchemaColumn;
import server.database.sql.SQLWhereClause.Condition;

/**
 * This class must be used as parent class to all SQL expressions
 * that can contain a WHERE clause. Those include SELECT, DELETE and UPDATE.
 *
 * @param <T> This MUST be the type of the child class. Otherwise method chaining WILL FAIL.
 */
public abstract class SQLWhereExpression<T> extends SQLBaseExpression<T> {

	private String whereOverride;
	private List<SQLWhereClause> where;
	private List<Object> whereBindVariables;
	
	public SQLWhereExpression( SQLDatabase db ) {
		super(db);
		
		this.whereOverride = null;
		this.where = new LinkedList<SQLWhereClause>();
		this.whereBindVariables = new LinkedList<Object>();
	}
	
	// FIXME: Translate everything and add missing documentation.
	// FIXME: refactor to allow nested AND and OR statements.
	
	/**
	 * Returns the list of bind variables (the question marks) values used in the SQL statement.
	 * Be aware of their order!
	 * @return List with values.
	 */
	public List<Object> getWhereBindVariables() {
		return new LinkedList<Object>( this.whereBindVariables );
	}
	
	/**
	 * Sets the values of the bind variables (the question marks) used in the SQL statement.
	 * Be aware of their order!
	 * @param bindVariables List of values.
	 */
	public void setWhereBindVariables( List<Object> bindVariables ) {
		this.whereBindVariables = new LinkedList<Object>( bindVariables );
	}
	
	
	/**
	 * Allows you to override the other where() command and specify your own where clause.
	 * Please use bind variable, which can be set with the method {@link #setWhereVariables(List)}.
	 * @param whereClause WHERE Expression (must be valid SQL).
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public T where( String whereClause ) {
		this.whereOverride = whereClause;
		return (T)this;
	}
	
	
	
	public T where( SchemaColumn column, String value ) {
		return this.where( column.getFullName(), value, Condition.EQUAL, Condition.AND, false );
	}
	
	public T where( String column, String value ) {
		return this.where( column, value, Condition.EQUAL, Condition.AND, false );
	}
	
	public T where( SchemaColumn column, int number ) {
		return this.where( column.getFullName(), String.valueOf(number), Condition.EQUAL, Condition.AND, true );
	}
	
	public T where( SchemaColumn column, String value, Condition comparison ) {
		return this.where( column.getFullName(), value, comparison, Condition.AND, false );
	}
	
	public T where( SchemaColumn column, int number, Condition comparison ) {
		return this.where( column.getFullName(), String.valueOf(number), comparison, Condition.AND, true );
	}
	
	public T or_where( SchemaColumn column, String value ) {
		return this.where( column.getFullName(), value, Condition.EQUAL, Condition.OR, false );
	}
	
	public T or_where( SchemaColumn column, int number ) {
		return this.where( column.getFullName(), String.valueOf(number), Condition.EQUAL, Condition.OR, true );
	}
	
	public T or_where( SchemaColumn column, String value, Condition comparison ) {
		return this.where( column.getFullName(), value, comparison, Condition.OR, false );
	}
	
	public T or_where( SchemaColumn column, int number, Condition comparison ) {
		return this.where( column.getFullName(), String.valueOf(number), comparison, Condition.OR, true );
	}
	
	public T where( SchemaColumn column, String value, Condition comparison, Condition connective, boolean noprime ) {
		return this.where(column.getFullName(), value, comparison, connective, noprime);
	}
	
	
	public T where(String column, int number) {
		return this.where( column, String.valueOf(number), Condition.EQUAL, Condition.AND, true );
	}
	
	public T where( String column, String value, Condition comparison ) {
		return this.where( column, value, comparison, Condition.AND, false );
	}
	
	public T where( String column, int number, Condition comparison ) {
		return this.where( column, String.valueOf(number), comparison, Condition.AND, true );
	}
	
	public T or_where( String column, String value ) {
		return this.where( column, value, Condition.EQUAL, Condition.OR, false );
	}
	
	public T or_where(String column, int number) {
		return this.where( column, String.valueOf(number), Condition.EQUAL, Condition.OR, true );
	}
	
	public T or_where( String column, String value, Condition comparison ) {
		return this.where( column, value, comparison, Condition.OR, false );
	}
	
	public T or_where( String column, int number, Condition comparison ) {
		return this.where( column, String.valueOf(number), comparison, Condition.OR, true );
	}
	
	/**
	 * Genau wie where(), allerdings wird bei noprime = true der zweite Wert nicht in einfache Anführungszeichen
	 * (') gesetzt. Wird z.B. für den Join von Tabellen benutzt.
	 * @param column
	 * @param value
	 * @param noprime
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public T where( String column, String value, Condition comparison, Condition connective, boolean noprime ) {
		// Variable fuer bind-Variable festlegen.
		this.whereBindVariables.add( value );

		if ( !noprime ) {
			value = "'"+value+"'";
		}
		
		this.where.add( new SQLWhereClause(column, value, comparison, connective) );
	
		return (T)this;
	}
	
	
	/**
	 * Gibt alle WHERE-Bedingungen 
	 * @return
	 */
	public List<SQLWhereClause> getWhere() {
		return this.where;
	}
	
	
	/**
	 * Gibt den WHERE-Teil von SQL-Ausdruecken wie SELECT, DELETE oder UPDATE zurueck.
	 * @param bindVars true, falls bind-Variablen (Fragezeichen im SQL-Ausdruck) verwendet werden sollen, ansonsten false.
	 * @return SQL-WHERE-Klausel eines SQL-Ausdruck.
	 */
	protected String generateWhereSQL( boolean bindVars ) {
		String result = "";
		
		// If the user wants to use his own WHERE clause let them.
		if ( whereOverride != null ) {
			return " WHERE "+this.whereOverride;
		}
		
		// Otherwise generate our own.
		if ( this.getWhere().size() > 0 ) {
			result += " WHERE ";
			
			boolean firstrun = true;
			for ( SQLWhereClause tupel : this.getWhere() ) {
				if ( !firstrun ) {
					result += tupel.getConnective();
				}
				result += tupel.getLeftOperand() + tupel.getComparison();
				if ( bindVars ) {
					result += "?";
				} else {
					result += tupel.getRightOperand();
				}
				
				firstrun = false;
			}
		}
		
		return result;
	}
}
