package server.database.sql;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

import exception.DatabaseException;

import server.database.schema.SchemaColumn;
import server.database.schema.TableSchema;
import server.util.Tuple;

/**
 * Resembles an SQL UPDATE statement.
 * Please see the note for method chaining in the description
 * of SQLBaseExpression.
 *
 * @see SQLBaseExpression
 */
public class SQLUpdate extends SQLWhereExpression<SQLUpdate> {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");

	private List<Tuple<String, String>> columnValues;
	private List<Object> setBindVariables;
	
	/**
	 * Creates a new SQL UPDATE expression.
	 * @param db Database connection. Must already be established and usable.
	 */
	public SQLUpdate( SQLDatabase db ) {
		super(db);
		
		this.columnValues = new LinkedList<Tuple<String, String>>();
		this.setBindVariables = new LinkedList<Object>();
	}
	
	
	/**
	 * Returns a list of columns and their values that would be used upon update.
	 * @return List of tuples containing a column and the assigned value.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 */
	public List<Tuple<String, String>> getColumnValues() {
		return this.columnValues;
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(SchemaColumn column, String value) {
		return this.set( column, value, false );
	}
	
	/**
	 * Sets the value of a specific column to a number.
	 * @param column Column that will be assigned with a value.
	 * @param number The numeric value the column should have.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(SchemaColumn column, int number) {
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @param noprime If true '' around the value will be omitted, otherwise they will be automatically added.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(SchemaColumn column, String value, boolean noprime) {
		return this.set( column.name, value, noprime );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(String column, String value) {
		return this.set( column, value, false );
	}
	
	/**
	 * Sets the value of a specific column to a number.
	 * @param column Column that will be assigned with a value.
	 * @param number The numeric value the column should have.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(String column, int number) {
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @param noprime If true '' around the value will be omitted, otherwise they will be automatically added.
	 * @return Current SQLUpdate object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLUpdate set(String column, String value, boolean noprime) {
		// Every value can be replaced by bind variables. Prepare this here.
		this.setBindVariables.add( value );
		
		// Add '' if necessary.
		if ( !noprime ) {
			value = "'"+value+"'";
		}

		this.columnValues.add( new Tuple<String, String>(column, value) );
		
		return this;
	}

	
	@Override
	protected String toSQL( boolean bindVars ) throws DatabaseException {
		String result = "";
		String sep = "";
		
		if ( this.getTable() == null || this.columnValues.isEmpty() ) {
			throw new DatabaseException("No table for UPDATE specified or empty UPDATE statement.", null);
		}
		
		// UPDATE ..
		result = "UPDATE " + this.getTable();
		
		// SET ..
		result += " SET ";
		for ( Tuple<String, String> columnValue : this.columnValues ) {
			result += sep + columnValue.getFirstElement() + "=";
			if ( bindVars ) {
				result += "?";
			} else {
				result += columnValue.getSecondElement();
			}
			sep = ",";
		}
				
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		return result;
	}

	/**
	 * Executes the UPDATE statement and returns the number of rows updated.
	 * @return Number of rows updated.
	 * @throws DatabaseException
	 * @see #update(String)
	 * @see #update(TableSchema)
	 */
	public int update() throws DatabaseException {
		// Create SQL statement.
		String sql = this.toSQL( true );
		
		List<Object> vars = new LinkedList<Object>();
		vars.addAll( this.setBindVariables );
		vars.addAll( this.getWhereBindVariables() );
		
		logger.debug("SQL UPDATE: " + sql);
		logger.debug("Bind Values: "+this.getWhereBindVariables());
		
		return this.db.dataManipulationQuery( sql, vars );
	}
	
	/**
	 * Executes the UPDATE statement and returns the number of rows updated.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)} is overridden with the specified value.
	 * Try to use {@link #update(TableSchema)} if possible.
	 * @param table Table in which the data will be updated.
	 * @return Number of rows updated.
	 * @throws DatabaseException
	 * @see #update()
	 * @see #update(TableSchema)
	 */
	public int update( String table ) throws DatabaseException {
		this.table( table );
		return this.update();
	}
	
	/**
	 * Executes the UPDATE statement and returns the number of rows updated.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)} is overridden with the specified value.
	 * @param table Table in which the data will be updated.
	 * @return Number of rows updated.
	 * @throws DatabaseException
	 * @see #update(String)
	 * @see #update()
	 */
	public int update( TableSchema table ) throws DatabaseException {
		this.table( table.name );
		return this.update();
	}
}
