package server.database.sql;

import org.apache.log4j.Logger;

import exception.DatabaseException;
import server.database.schema.TableSchema;

/**
 * Resembles an SQL DELETE statement.
 * Please see the note for method chaining in the description
 * of SQLBaseExpression.
 *
 * @see SQLBaseExpression
 */
public class SQLDelete extends SQLWhereExpression<SQLDelete> {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");

	/**
	 * Creates a new SQL DELETE expression.
	 * @param db Database connection. Must already be established and usable.
	 */
	public SQLDelete( SQLDatabase db ) {
		super( db );
	}
	
	@Override
	protected String toSQL( boolean bindVars ) throws DatabaseException {
		String result = "";
		
		if ( this.getTable() == null ) {
			throw new DatabaseException("No table for DELETE specified.", null);
		}
		
		// DELETE FROM ..
		result = "DELETE FROM "+this.getTable();
		
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		return result;
	}
	
	/**
	 * Executes the DELETE statement and returns the number of deleted rows.
	 * @return Number of deleted rows.
	 * @throws DatabaseException
	 * @see #delete(String)
	 * @see #delete(TableSchema)
	 */
	public int delete() throws DatabaseException {
		// Create SQL statement.
		String sql = this.toSQL( true );
		
		logger.debug("SQL DELETE: " + sql);
		logger.debug("Bind Values: "+this.getWhereBindVariables());
		
		return this.db.dataManipulationQuery( sql, this.getWhereBindVariables() );
	}
	
	/**
	 * Executes the DELETE statement and returns the number of deleted rows.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)}is overridden with the specified value.
	 * Try to use {@link #delete(TableSchema)} if possible.
	 * @param table Table in which the data will be deleted.
	 * @return Number of deleted rows.
	 * @throws DatabaseException
	 * @see #delete()
	 * @see #delete(TableSchema)
	 */
	public int delete( String table ) throws DatabaseException {
		this.table( table );
		return this.delete();
	}
	
	/**
	 * Executes the DELETE statement and returns the number of deleted rows.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)}is overridden with the specified value.
	 * @param table Table in which the data will be deleted.
	 * @return Number of deleted rows.
	 * @throws DatabaseException
	 * @see #delete()
	 * @see #delete(String)
	 */
	public int delete( TableSchema table ) throws DatabaseException {
		this.table( table.name );
		return this.delete();
	}
}
