package server.database.schema.app;

import exception.DatabaseException;
import exception.UnsupportedFormulaException;
import server.censor.signature.experiment.ExperimentPreparationDBSEC12;
import server.database.AppDatabase;
import server.database.NewMaintenanceDatabase;
import server.database.OracleSQLAppDatabase;
import server.database.schema.app.dict.Sig_KrankheitDictionary;
import server.database.sql.SQLDatabase;

/**
 * Datenbank-Schema fuer den SignatureCensor und der krankheit-Tabelle.
 * 
 * Erweitert die Klasse SignatureExperimentSchema dadurch, dass zusaetzlich zur
 * Erstellung des Schemas die Tabellen mit einem konkreten Beispiel aus dem
 * Experiment zum SignatureCensor gefuellt werden.
 * 
 * Achtung: Fuer eine korrekte Funktionsweise muss zusaetzlich zur Generierung
 * dieses Schemas auch die sql-Datei 'staticSignatureCensorExperimentCreate.sql', 
 * die u.a. die Trigger fuer den SignatureCensor enthaelt, in der Datenbank
 * ausgefuehrt werden.
 */
public class SignatureExperimentExample extends SignatureExperimentSchema {

	private final int NUMBER_OF_INSTANCE_COPIES = 1;
	
	public final Sig_KrankheitDictionary sig_krankheitDict;
	
	public SignatureExperimentExample() throws DatabaseException {
		super("Signature Experiment Schema with concrete example");
		
		this.sig_krankheitDict = new Sig_KrankheitDictionary();
		
		
		this.maintenanceTables.add( this.sig_krankheitDict );
	}
	
	@Override
	public void fillWithContents(NewMaintenanceDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		this.fillMaintenanceDBWithContents(maintenanceDB);
		
		try {
			// delegate creation of the example to the ExperimentPreparation class
			// unflexible approach
			AppDatabase applicationDB = new OracleSQLAppDatabase(appDB);
			ExperimentPreparationDBSEC12 experimentPreparation = new ExperimentPreparationDBSEC12(maintenanceDB, applicationDB, false, 8);
			experimentPreparation.prepareExperiment(NUMBER_OF_INSTANCE_COPIES);
			// flexible approach
			experimentPreparation = new ExperimentPreparationDBSEC12(maintenanceDB, applicationDB, true, 7);
			experimentPreparation.prepareExperiment(NUMBER_OF_INSTANCE_COPIES);
		}
		catch( UnsupportedFormulaException e ) {
			throw new DatabaseException("Error while preparing SignatureExperiment!", e);
		}
	}
	
	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		// Create default access rights
		super.createDefaultAccessRights(db);
		
		// Create default users
		super.createDefaultUsers(db);
		
		super.linkDictionary( db, this.sig_krankheitDict, Krankheit2TableSchema.KRANKHEIT );
	}
	
	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		// has been filled during the preparation of the signature experiment in method fillWithContents()
	}
}
