package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.GermanAuswirkungDictionary;
import server.database.schema.app.dict.GermanBehandlungDictionary;
import server.database.schema.app.dict.GermanKrankheitDictionary;
import server.database.schema.app.dict.GermanNachnameDictionary;
import server.database.schema.app.dict.GermanVornameDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class PreCQEKrankheitInventionExample extends MaintenanceDatabaseSchema {

	public final Krankheit2TableSchema krankheit;
	public final BehandlungTableSchema behandlung;
	public final AuswirkungTableSchema auswirkung;
	public final GermanNachnameDictionary nameDict;
	public final GermanVornameDictionary vornameDict;
	public final GermanKrankheitDictionary krankheitDict;
	public final GermanBehandlungDictionary behandlungDict;
	public final GermanAuswirkungDictionary auswirkungDict;
	
	public PreCQEKrankheitInventionExample() throws DatabaseException {
		super("Kombination aus PreCQE Krankheit Example und PreCQE Invention Example");
		
		this.krankheit = new Krankheit2TableSchema();
		this.behandlung = new BehandlungTableSchema();
		this.auswirkung = new AuswirkungTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.krankheit );
		this.appTables.add( this.behandlung );
		this.appTables.add( this.auswirkung );
		
		this.nameDict = new GermanNachnameDictionary();
		this.vornameDict = new GermanVornameDictionary();
		this.krankheitDict = new GermanKrankheitDictionary();
		this.behandlungDict = new GermanBehandlungDictionary();
		this.auswirkungDict = new GermanAuswirkungDictionary();
		
		this.maintenanceTables.add( this.nameDict );
		this.maintenanceTables.add( this.vornameDict );
		this.maintenanceTables.add( this.krankheitDict );
		this.maintenanceTables.add( this.behandlungDict );
		this.maintenanceTables.add( this.auswirkungDict );
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillKrankheit(db);
		this.fillBehandlung(db);
		this.fillAuswirkung(db);
	}
	
	private void fillKrankheit(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Meier");
		sql.set(Krankheit2TableSchema.VORNAME, "Markus");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "RLS");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Becker");
		sql.set(Krankheit2TableSchema.VORNAME, "Bernd");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Aids");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Becker");
		sql.set(Krankheit2TableSchema.VORNAME, "Bernd");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Hirntumor");
		db.insert(krankheit);

		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Müller");
		sql.set(Krankheit2TableSchema.VORNAME, "Lisa");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Grippe");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Fischer");
		sql.set(Krankheit2TableSchema.VORNAME, "Fritz");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "AllergischerSchock");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Meier");
		sql.set(Krankheit2TableSchema.VORNAME, "Heinz");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Lungenentzündung");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Müller");
		sql.set(Krankheit2TableSchema.VORNAME, "Lisa");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Beinbruch");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Müller");
		sql.set(Krankheit2TableSchema.VORNAME, "Peter");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Magengeschwür");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Schurich");
		sql.set(Krankheit2TableSchema.VORNAME, "Daniel");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Hirntumor");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Schurich");
		sql.set(Krankheit2TableSchema.VORNAME, "Daniel");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Beinbruch");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(Krankheit2TableSchema.NAME, "Meier");
		sql.set(Krankheit2TableSchema.VORNAME, "Marie");
		sql.set(Krankheit2TableSchema.KRANKHEIT, "Aids");
		db.insert(krankheit);
		
	}
	
	private void fillBehandlung(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Meier");
		sql.set(BehandlungTableSchema.VORNAME, "Markus");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Carbamazepin");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Meier");
		sql.set(BehandlungTableSchema.VORNAME, "Markus");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Morphin");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Fischer");
		sql.set(BehandlungTableSchema.VORNAME, "Fritz");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Adrenalin");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Fischer");
		sql.set(BehandlungTableSchema.VORNAME, "Fritz");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Cortison");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Meier");
		sql.set(BehandlungTableSchema.VORNAME, "Heinz");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Antibiotika");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Müller");
		sql.set(BehandlungTableSchema.VORNAME, "Lisa");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Paracetamol");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Müller");
		sql.set(BehandlungTableSchema.VORNAME, "Peter");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Pantozol");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Müller");
		sql.set(BehandlungTableSchema.VORNAME, "Lisa");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Tamiflu");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Meier");
		sql.set(BehandlungTableSchema.VORNAME, "Marie");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "MedizinA");
		db.insert(behandlung);
		
		sql = db.sql();
		sql.set(BehandlungTableSchema.NAME, "Meier");
		sql.set(BehandlungTableSchema.VORNAME, "Marie");
		sql.set(BehandlungTableSchema.BEHANDLUNG, "Tamiflu");
		db.insert(behandlung);
		
	}
	
	private void fillAuswirkung(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Lehmann");
		sql.set(AuswirkungTableSchema.VORNAME, "Lars");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "Pflegefall");
		db.insert(auswirkung);
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Lehmann");
		sql.set(AuswirkungTableSchema.VORNAME, "Lars");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "Blindheit");
		db.insert(auswirkung);
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Lehmann");
		sql.set(AuswirkungTableSchema.VORNAME, "Lars");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "Taubheit");
		db.insert(auswirkung);
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Becker");
		sql.set(AuswirkungTableSchema.VORNAME, "Bernd");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "geringeLebenserwartung");
		db.insert(auswirkung);
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Schurich");
		sql.set(AuswirkungTableSchema.VORNAME, "Daniel");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "Arbeitsausfall");
		db.insert(auswirkung);
		
		sql = db.sql();
		sql.set(AuswirkungTableSchema.NAME, "Müller");
		sql.set(AuswirkungTableSchema.VORNAME, "Lisa");
		sql.set(AuswirkungTableSchema.AUSWIRKUNG, "Arbeitsausfall");
		db.insert(auswirkung);
	}
	

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		super.createAdminUser(db);
		int userId = super.createTestUser(db, 1, "PreCQE");
		try {
			this.fillPrior(db.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(db.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User does not exists, check your code.", e);
		}
		
		super.linkDictionary( db, this.nameDict, Krankheit2TableSchema.NAME );
		super.linkDictionary( db, this.vornameDict, Krankheit2TableSchema.VORNAME );
		super.linkDictionary( db, this.krankheitDict, Krankheit2TableSchema.KRANKHEIT );
		
		super.linkDictionary( db, this.nameDict, AuswirkungTableSchema.NAME );
		super.linkDictionary( db, this.vornameDict, AuswirkungTableSchema.VORNAME );
		super.linkDictionary( db, this.auswirkungDict, AuswirkungTableSchema.AUSWIRKUNG );
		
		super.linkDictionary( db, this.nameDict, BehandlungTableSchema.NAME );
		super.linkDictionary( db, this.vornameDict, BehandlungTableSchema.VORNAME );
		super.linkDictionary( db, this.behandlungDict, BehandlungTableSchema.BEHANDLUNG );

	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("FORALL X,Y (behandlung(X,Y,\"Morphin\") IMPL (krankheit(X,Y,\"RLS\") OR krankheit(X,Y,\"Rheuma\")))"));
			prior.add(new Formula("FORALL X,Y (behandlung(X,Y,\"Carbamazepin\") IMPL (krankheit(X,Y,\"RLS\") OR krankheit(X,Y,\"Epilepsie\")))"));
			prior.add(new Formula("FORALL X,Y ((krankheit(X,Y,\"Epilepsie\") OR krankheit(X,Y,\"Rheuma\")) IMPL (NOT behandlung(X,Y,\"Carbamazepin\")))"));
			prior.add(new Formula("FORALL X,Y (behandlung(X,Y,\"ChemoTherapie\") IMPL krankheit(X,Y,\"Krebs\"))"));
			prior.add(new Formula("FORALL X,Y (behandlung(X,Y,\"Pantozol\") IMPL krankheit(X,Y,\"Magengeschwür\"))"));
			prior.add(new Formula("FORALL X,Y (krankheit(X,Y,\"Beinbruch\") IMPL auswirkung(X,Y,\"Arbeitsausfall\"))"));
			prior.add(new Formula("FORALL X,Y ((krankheit(X, Y, \"QSL\") OR krankheit(X, Y, \"Demenz\") OR (auswirkung(X, Y, \"Blindheit\") AND auswirkung(X, Y, \"Taubheit\"))) IMPL auswirkung(X, Y, \"Pflegefall\"))"));
			prior.add(new Formula("FORALL X,Y (auswirkung(X, Y, \"geringeLebenserwartung\") IMPL (krankheit(X, Y, \"Hirntumor\") OR krankheit(X, Y, \"Aids\")))"));			
			prior.add(new Formula("FORALL X,Y (auswirkung(X, Y, \"geringeLebenserwartung\") AND behandlung(X, Y, \"ChemoTherapie\") IMPL krankheit(X, Y, \"Hirntumor\"))"));
			// Formeln fuers Erfinden
			prior.add(new Formula("EXISTS X,Y,Z,W (behandlung(X, Y, Z) AND krankheit(X, Y, W))"));
			prior.add(new Formula("EXISTS X,Y (behandlung(X, Y, \"Tamiflu\") AND krankheit(X, Y, \"Grippe\"))"));
			prior.add(new Formula("FORALL X,Y ((behandlung(X, Y, \"MedizinA\") AND behandlung(X, Y, \"Tamiflu\")) IMPL krankheit(X,Y,\"Aids\"))"));
			
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("krankheit(\"Becker\", \"Bernd\", \"Aids\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("krankheit(\"Becker\", \"Bernd\", \"Hirntumor\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("krankheit(\"Becker\", \"Bernd\", \"MS\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("auswirkung(\"Lehmann\", \"Lars\", \"Pflegefall\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("krankheit(\"Meier\",\"Markus\",\"RLS\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("krankheit(\"Müller\",\"Peter\",\"Aids\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			// Formeln fuers Erfinden
			policy.add(new Formula("krankheit(\"Müller\",\"Lisa\",\"Grippe\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("EXISTS X,Y (krankheit(X,Y,\"Aids\"))"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);

		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}