package server.database.schema.app;


import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.DiagnosisDictionary;
import server.database.schema.app.dict.NameDictionary;
import server.database.schema.app.dict.PrescriptionDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class PreCQEFirstExample extends MaintenanceDatabaseSchema {
	
	public final IllTableSchema ill;
	public final TreatTableSchema treat;
	public final NameDictionary nameDict;
	public final DiagnosisDictionary diagnosisDict;
	public final PrescriptionDictionary prescriptionDict;
	
	public PreCQEFirstExample() throws DatabaseException {
		super("PreCQE First Example");
		
		this.ill = new IllTableSchema();
		this.treat = new TreatTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.ill );
		this.appTables.add( this.treat );
		
		this.nameDict = new NameDictionary();
		this.diagnosisDict = new DiagnosisDictionary();
		this.prescriptionDict = new PrescriptionDictionary();
		
		this.maintenanceTables.add( this.nameDict );
		this.maintenanceTables.add( this.diagnosisDict );
		this.maintenanceTables.add( this.prescriptionDict );
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillTreat(db);
	}
	
	private void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "Aids" );
		db.insert(this.ill);
	}
	
	private void fillTreat(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineA" );
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Lisa" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
		db.insert(this.treat);
	}

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase mdb) throws DatabaseException {
		super.createDefaultAccessRights(mdb);
		super.createAdminUser(mdb);
		int userId = super.createTestUser(mdb, 1, "PreCQE");
		try {
			this.fillPrior(mdb.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(mdb.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User does not exists, check your code.", e);
		}
		
		super.linkDictionary( mdb, this.nameDict, IllTableSchema.NAME );
		super.linkDictionary( mdb, this.diagnosisDict, IllTableSchema.DIAGNOSIS );
		super.linkDictionary( mdb, this.nameDict, TreatTableSchema.NAME );
		super.linkDictionary( mdb, this.prescriptionDict, TreatTableSchema.PRESCRIPTION );
	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("EXISTS B (treat(B,\"MedicineA\") AND ill(B,\"Flu\"))"));
			prior.add(new Formula("NOT EXISTS B (treat(B,\"MedicineA\") AND treat(B,\"MedicineB\") AND NOT ill(B,\"Aids\"))"));
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("EXISTS B ill(B,\"Aids\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("ill(\"Lisa\",\"Flu\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}
