package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import server.data.ConfidentialityPolicy;
import server.data.Role;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.GermanKrankheitDictionary2;
import server.database.schema.app.dict.GermanNachnameDictionary2;
import server.database.schema.app.dict.GermanSymptomDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class KrankheitExample extends MaintenanceDatabaseSchema {

	public final KrankheitTableSchema krankheit;
	
	public final GermanSymptomDictionary symptomDict;
	public final GermanKrankheitDictionary2 diagnosisDict;
	public final GermanNachnameDictionary2 patientDict;
	
	
	public KrankheitExample() throws DatabaseException {
		super("Krankheit Example");
		
		this.krankheit = new KrankheitTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.krankheit );
		
		this.symptomDict = new GermanSymptomDictionary();
		this.diagnosisDict = new GermanKrankheitDictionary2();
		this.patientDict = new GermanNachnameDictionary2();
		
		this.maintenanceTables.add( this.symptomDict );
		this.maintenanceTables.add( this.diagnosisDict );
		this.maintenanceTables.add( this.patientDict );
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillKrankheit(db);
	}
	
	private void fillKrankheit(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set(KrankheitTableSchema.SYMPTOM, "Fieber");
		sql.set(KrankheitTableSchema.DIAGNOSIS, "Krebs");
		sql.set(KrankheitTableSchema.PATIENT, "Schmidt");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(KrankheitTableSchema.SYMPTOM, "Fieber");
		sql.set(KrankheitTableSchema.DIAGNOSIS, "Krebs");
		sql.set(KrankheitTableSchema.PATIENT, "Mueller");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(KrankheitTableSchema.SYMPTOM, "Blutarmut");
		sql.set(KrankheitTableSchema.DIAGNOSIS, "Krebs");
		sql.set(KrankheitTableSchema.PATIENT, "Schmidt");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(KrankheitTableSchema.SYMPTOM, "Atemnot");
		sql.set(KrankheitTableSchema.DIAGNOSIS, "Lungenentzuendung");
		sql.set(KrankheitTableSchema.PATIENT, "Schaefer");
		db.insert(krankheit);
		
		sql = db.sql();
		sql.set(KrankheitTableSchema.SYMPTOM, "Muedigkeit");
		sql.set(KrankheitTableSchema.DIAGNOSIS, "Burn-out-Syndrom");
		sql.set(KrankheitTableSchema.PATIENT, "Schneider");
		db.insert(krankheit);
	}

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		super.createAdminUser(db);
		
		String[] censors = {"Lying", "Refusal", "Combined", "OpenLying", "OpenRefusal", "OpenCombined"};
		
		Role stdRole = Role.load(db.getDb(), 1);
		for( String censor : censors ) {
			User tmpUser = db.getUserManagement().add(censor.toLowerCase());
			tmpUser.setPassword(censor.toLowerCase());
			tmpUser.setCensor(censor);
			tmpUser.setRole(stdRole);
			this.fillConfidentialityPolicy(tmpUser);
		}
		
		super.linkDictionary( db, this.symptomDict, KrankheitTableSchema.SYMPTOM );
		super.linkDictionary( db, this.diagnosisDict, KrankheitTableSchema.DIAGNOSIS );
		super.linkDictionary( db, this.patientDict, KrankheitTableSchema.PATIENT );
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("NOT krankheit(\"Uebelkeit\", \"Leberzyste\", \"Buchholz\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}
