package server.database.schema;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

import server.database.NewMaintenanceDatabase;
import server.database.schema.app.DictTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLInsert;
import exception.DatabaseException;

public abstract class DatabaseSchema {
	protected final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");
	
	private String name;
	protected List<TableSchema> appTables;
	protected List<TableSchema> maintenanceTables;
	
	public DatabaseSchema( String name ) {
		this.name = name;
		this.appTables = new LinkedList<TableSchema>();
		this.maintenanceTables = new LinkedList<TableSchema>();
	}
	
	public void fillWithContents(NewMaintenanceDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		this.fillMaintenanceDBWithContents(maintenanceDB);
		this.fillAppDBWithContents(appDB);
	}
	
	public abstract void fillAppDBWithContents(SQLDatabase db) throws DatabaseException;
	public abstract void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException;
	
	public void dropTables(SQLDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		for ( TableSchema table : this.maintenanceTables ) {
			this.dropTable(maintenanceDB, table);
		}
		for ( TableSchema table : this.appTables ) {
			this.dropTable(appDB, table);
		}
	}
	
	private void dropTable(SQLDatabase db, TableSchema table) throws DatabaseException {
		for ( String query : table.generateDropSQL() ) {
			logger.debug( query );
			try {
				db.rawQuery(query);
			} catch ( DatabaseException ex ) {
				// Oracle hack..
				if ( ex.getOriginalException().getMessage().startsWith("ORA-00942") ) {
					continue;
				} else if ( ex.getOriginalException().getMessage().startsWith("ORA-02289") ) {
					continue;
				} else {
					throw ex;
				}
			}
		}
	}
	
	public void initTables(SQLDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		for ( TableSchema table : this.maintenanceTables ) {
			this.initTable(maintenanceDB, table);
			
			// Fill dictionary if neccessary.
			if ( table instanceof DictTableSchema ) {
				this.initDictionary( maintenanceDB, (DictTableSchema)table );
			}
		}
		for ( TableSchema table : this.appTables ) {
			this.initTable(appDB, table);
		}
	}
	
	private void initTable(SQLDatabase db, TableSchema table) throws DatabaseException {
		for ( String query : table.generateCreateSQL() ) {
			logger.debug( query );
			db.rawQuery(query);
		}
	}
	
	/**
	 * Creates a new dictionary for the given relation and attribute. Also links the table-specific dictionary table to
	 * the global dictionary-table. The table name will be "DICT_RELATIONNAME_ATTRIBUTENAME".
	 * 
	 * @param mdb MaintenanceDatabase that will be used.
	 * @param relation Name of the relation.
	 * @param attribute Name of the attribute.
	 * @return The newly created TableSchema (should be saved inside your example).
	 * @throws DatabaseException
	 */
	private void initDictionary( SQLDatabase db, DictTableSchema table ) throws DatabaseException {
		SQLInsert sql;
		
		for ( String entry : table.getDictionaryContents() ) {			
			sql = db.newInsert();
			sql.set( DictTableSchema.ENTRY, entry );
			sql.insert( table );
		}
	}
	
	public void recreateTables(SQLDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		this.dropTables(maintenanceDB, appDB);
		this.initTables(maintenanceDB, appDB);
	}
	
	public void truncateTables(SQLDatabase maintenanceDB, SQLDatabase appDB) throws DatabaseException {
		this.truncateMaintenanceTables(maintenanceDB);
		this.truncateAppTables(appDB);
	}
	
	public void truncateMaintenanceTables(SQLDatabase maintenanceDB) throws DatabaseException {
		for ( TableSchema table : this.maintenanceTables ) {
			this.truncateTable(maintenanceDB, table);
		}
	}
	
	public void truncateAppTables(SQLDatabase appDB) throws DatabaseException {
		for ( TableSchema table : this.appTables ) {
			this.truncateTable(appDB, table);
		}
	}
	
	public void truncateTable(SQLDatabase db, TableSchema table) throws DatabaseException {
		for ( String query : table.generateTruncateSQL() ) {
			logger.debug( query );
			try {
				db.rawQuery(query);
			} catch ( DatabaseException ex ) {
				// Oracle hack..
				if ( ex.getOriginalException().getMessage().startsWith("ORA-00942") ) {
					continue;
				} else if ( ex.getOriginalException().getMessage().startsWith("ORA-02289") ) {
					continue;
				} else {
					throw ex;
				}
			}
		}
	}
	
	public String toString() {
		return this.name;
	}

}
