package server.database.cache;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;

import exception.DatabaseException;

import server.database.NewMaintenanceDatabase;
import server.database.sql.SQLDatabase;

public class Attribute {

	private AppDatabaseSchemaCache cache;
	private SQLDatabase appDB;
	private NewMaintenanceDatabase maintenanceDB;
	private String relation;
	private String name;
	private String type;
	private List<String> possibleValues;
	private Iterator<String> possibleValueIterator;
	private Set<String> activeDomainFromDB;
	private Set<String> customActiveDomain;
	
	
	public Attribute( AppDatabaseSchemaCache cache, SQLDatabase appDB, NewMaintenanceDatabase maintenanceDB, String relation, String name, String type ) {
		this.cache = cache;
		this.appDB = appDB;
		this.maintenanceDB = maintenanceDB;
		this.relation = relation;
		this.name = name;
		this.type = type;
		this.possibleValues = null;
		this.activeDomainFromDB = null;
		this.customActiveDomain = new HashSet<String>();
	}
	
	public String getRelation() {
		return this.relation;
	}
	
	public String getName() {
		return this.name;
	}
	
	public String getType() {
		return this.type;
	}
	
	/**
	 * Returns the list of possible values from the dictionary for this attribute.
	 * The values are loaded on-demand and cached. If the cache grows too large,
	 * values might get unloaded and will be loaded on the next access of this
	 * method.
	 * 
	 * @return List of possible values from the dictionary. Can be an empty list.
	 * @throws DatabaseException
	 */
	public List<String> getPossibleValues() throws DatabaseException {
		this.loadDictionary();
		
		return this.possibleValues;
	}
	
	/**
	 * Returns one value of the possible values. It will iterate over all possible
	 * values and start from the beginning, if needed.
	 * @return A possible value or "FIXME" if no dictionary is linked with the attribute or the linked dictionary is empty.
	 * @throws DatabaseException
	 */
	public String getPossibleValue() throws DatabaseException {
		this.loadDictionary();
		try {
			return this.possibleValueIterator.next();
		} catch ( NoSuchElementException ex ) {
			if ( this.possibleValues.isEmpty() ) {
				return "FIXME";
			} else {
				this.possibleValueIterator = this.possibleValues.iterator();
				return this.getPossibleValue();
			}
		}
	}
	
	/**
	 * Loads the dictionary if needed.
	 * @throws DatabaseException
	 */
	private void loadDictionary() throws DatabaseException {
		// Load dictionary on-demand.
		if ( this.possibleValues == null ) {
			this.possibleValues = this.maintenanceDB.getDictionary().getValues(this.relation, this.name);
			this.cache.loadedDictionary(this);
			this.possibleValueIterator = this.possibleValues.iterator();
		}
	}
	
	/**
	 * This method resets the cache of possible values. Should only be called by SQLSchemaCache.
	 */
	public void resetPossibleValues() {
		this.possibleValues = null;
	}
	
	/**
	 * Returns the current active domain.
	 * @return
	 * @throws DatabaseException 
	 */
	public Set<String> getActiveDomain() throws DatabaseException {
		this.loadActiveDomain();
		
		HashSet<String> result = new HashSet<String>();
		result.addAll( this.activeDomainFromDB );
		result.addAll( this.customActiveDomain );
		
		return result;
	}
	
	/**
	 * Adds a value to current active domain.
	 * @param value
	 */
	public void addCustomActiveDomainValue( String value ) {
		this.customActiveDomain.add( value );
	}
	
	/**
	 * Resets the current active domain.
	 */
	public void resetActiveDomainFromDB() {
		this.activeDomainFromDB = null;
	}
	
	public void resetCustomActiveDomain() {
		this.customActiveDomain.clear();
	}
	
	/**
	 * Loads the active domain from the db if needed.
	 * @throws DatabaseException
	 */
	private void loadActiveDomain() throws DatabaseException {
		// Load active domain on-demand.
		if ( this.activeDomainFromDB == null ) {
			this.activeDomainFromDB = this.appDB.getAttributeValues(this.relation, this.name);
			this.cache.loadedActiveDomain(this);
		}
	}

	
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result
				+ ((relation == null) ? 0 : relation.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Attribute other = (Attribute) obj;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (relation == null) {
			if (other.relation != null)
				return false;
		} else if (!relation.equals(other.relation))
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}
	
	
}
