package server.database;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import server.database.sql.Column;
import server.database.sql.OracleSQLDatabase;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLTable;
import server.parser.Formula;
import server.parser.FormulaToOracleSQL;
import server.parser.Formatter.ConstantWithoutQuotesFormatter;
import server.parser.node.AtomPredicateNode;
import server.parser.node.ConstantNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.TermNode;
import exception.DatabaseException;
import exception.UnsupportedFormulaException;

public class OracleSQLAppDatabase extends AppDatabase {
	
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");

	
	protected SQLDatabase db = null;
	protected FormulaToOracleSQL toSQL = null;
	protected ConstantWithoutQuotesFormatter formatter = null;
	
	/**
	 * Maximum size (number of entries) of the active domain cache.
	 * With 1000*1000 entries you have roughly 30mb cache with each item being 30 characters long.
	 */
	public final static int ACTIVE_DOMAIN_SIZE_LIMIT = 1000*10;
	
	// Stores active domain values.
	private Map<Column, Set<String>> activeDomain = null;
	private List<Column> loadedActiveDomainHistory = null;
	private int activeDomainSizeCounter = 0;
	
	public OracleSQLAppDatabase( DatabaseConfiguration config ) throws DatabaseException {
		this( new OracleSQLDatabase(config) );
	}
	
	public OracleSQLAppDatabase( SQLDatabase db ) {
		this.db = db;
		this.toSQL = new FormulaToOracleSQL( db );
		this.formatter = new ConstantWithoutQuotesFormatter();
		this.activeDomain = new HashMap<Column, Set<String>>();
		this.loadedActiveDomainHistory = new LinkedList<Column>();
		this.activeDomainSizeCounter = 0;
	}
	
	/**
	 * Returns the currently used sql database.
	 * @return SQL-Database instance.
	 */
	@Override
	public SQLDatabase getSQLDatabase() {
		return this.db;
	}

	@Override
	public boolean evalComplete(Formula interaction) throws DatabaseException, UnsupportedFormulaException {
		String sql = this.toSQL.translateFormula(interaction);
		logger.debug("Relational Calculus Anfrage: "+interaction);
		logger.debug("SQL-Anfrage: "+sql);
		SQLTable state = db.query( sql );
		// next() gibt true zurueck, wenn es noch eine weitere Zeile in der Antwort auf
		// die SQL-Anfrage gibt. Am Anfang steht der Cursor vor der ersten Zeile, d.h.
		// next() gibt in der hier verwendeten Weise true zurueck, wenn es mind. eine Zeile
		// im Ergebnis gibt und ansonsten false.
		// Das entspricht genau der Semantik von evalComplete(), deshalb koennen wir einfach
		// return verwenden.
		//return state.next();
		return state.getRowCount() != 0;
	}
	
	@Override
	public List<Formula> evalPosComplete(Formula interaction) throws DatabaseException, UnsupportedFormulaException {
		ArrayList<Formula> result = new ArrayList<Formula>();
		
		// Offene Anfrage an die Datenbank stellen.
		String sql = this.toSQL.translateFormula(interaction);
		SQLTable dbResult = db.query( sql );
		// Das Ergebnis beinhaltet eine Tabelle, die als
		// Spaltennamen die Variablennnamen haben.
		String[] variables = dbResult.getColumnNames();
		for ( String[] row : dbResult ) {
			HashMap<String,String> mapping = new HashMap<String,String>();
			int i = 0;
			for ( String columnValue : row ) {
				mapping.put(variables[i], columnValue);
				i++;
			}
			
			// Instanz des Constraints bilden und zum Ergebnis hinzufuegen.
			Formula instance = (Formula)interaction.clone();
			instance.substituteVariables(mapping);
			result.add( instance );
		}
		
		return result;
	}

	@Override
	public int evalIncomplete(Formula interaction) throws DatabaseException {
		// TODO Auto-generated method stub
		return 0;
	}
	
	/**
	 * Gibt ein Array der Attributnamen einer Relation zurück.
	 * @param relationname Name der Relation, dessen Attributnamen zurückgegeben werden sollen
	 * @return Array von Attributnamen der Relation relationname
	 */
	@Override
	public String[] getAttributeNames(String relationname) throws DatabaseException {
		return db.getColumnNames(relationname);
	}

	
	@Override
	public int updateComplete(List<Formula> literals) throws DatabaseException, UnsupportedFormulaException {
		Map<String,List<List<ConstantNode>>> positiveAtoms = new HashMap<String,List<List<ConstantNode>>>();
		Map<String,List<List<ConstantNode>>> negativeAtoms = new HashMap<String,List<List<ConstantNode>>>();
		int count = 0;

		// Zuerst wird zwischen positiven Atomen unterschieden, die als neue Tupel
		// eingefuegt werden muessen und negativen, die wir loeschen muessen.
		for ( Formula literalRoot : literals ) {
			Node literal = literalRoot.getRootChild();
			
			logger.debug("Update Literal: "+literal);
			
			AtomPredicateNode atom = null;
			Map<String,List<List<ConstantNode>>> atomMap = null;
			
			if ( literal.isAtomPredicateNode() ) {
				atom = (AtomPredicateNode)literal;
				atomMap = positiveAtoms;
			} else if ( literal.isNegationNode() ) {
				NegationNode negNode = (NegationNode)literal;
				if ( negNode.getNegatedFormula().isAtomPredicateNode() ) {
					atom = (AtomPredicateNode)negNode.getNegatedFormula();
					atomMap = negativeAtoms;
				}
			}
			
			// Kein Literal oder freie Variablen enthalten.
			if ( atom == null || atom.getFreeVariables().size() > 0 ) {
				throw new UnsupportedFormulaException( literal.toString() );
			}
			
			// TermNode in ConstantNode umwandeln.
			List<ConstantNode> params = new ArrayList<ConstantNode>();
			for ( TermNode param : atom.getParameterNodes() ) {
				params.add( (ConstantNode)param );
			}
			
			// Literal jetzt einordnen.
			if ( atomMap.containsKey(atom.getRelationname()) ) {
				atomMap.get(atom.getRelationname()).add( params );
			} else {
				List<List<ConstantNode>> tupleList = new ArrayList<List<ConstantNode>>();
				tupleList.add( params );
				atomMap.put( atom.getRelationname(), tupleList );
			}
		}
		
		// Alte Transaktion ggf. beenden.
		this.db.commit();
		
		try {
			// Jetzt kann das Einfuegen und Loeschen passieren (alle Literale okay).
			for ( Map.Entry<String, List<List<ConstantNode>>> entry : positiveAtoms.entrySet() ) {
				count += this.addTuples( entry.getKey(), entry.getValue() );
			}
			
			for ( Map.Entry<String, List<List<ConstantNode>>> entry : negativeAtoms.entrySet() ) {
				count += this.deleteTuples( entry.getKey(), entry.getValue() );
			}
		} catch ( DatabaseException e ) {
			this.db.rollback();
			throw e;
		}
		
		// Transaktion beenden.
		this.db.commit();
		
		return count;
	}
	
	/**
	 * Fuegt fuer eine Relation (Datenbanktabelle) mehrere Tupel ein.
	 * Jedes Tupel muss fuer jede Spalte einen Wert enthalten, ansonsten wird eine DatabaseException
	 * ausgeloest.
	 * @param relation Datenbanktabelle, in die Tupel einegefuegt werden sollen.
	 * @param tuples Liste der Tupel, die in die Tabelle einegefuegt werden sollen.
	 * @throws DatabaseException Wird geworfen, wenn ein Fehler in der Datenbank auftritt (z.B. Constraints verletzt oder Verbindungsabbruch) oder wenn die Tupel nicht die richtige Form haben.
	 * @return The number of tuples that were added.
	 */
	private int addTuples( String relation, List<List<ConstantNode>> tuples ) throws DatabaseException {
		List<Column> columns = this.db.getColumns( relation );
		int added = 0;
		
		for ( List<ConstantNode> tuple : tuples ) {
			if ( columns.size() != tuple.size() ) {
				throw new DatabaseException("Tuple has wrong number of columns", null);
			}
			
			SQLInsert sql = this.db.newInsert();
			Iterator<Column> columnIter = columns.iterator();
			for ( ConstantNode columnValue : tuple ) {
				sql.set( columnIter.next().getName(), columnValue.toString( formatter ) );
			}
			
			int insertCount = sql.insert( relation );
			added += insertCount;
		}
		
		return added;
	}
	
	/**
	 * Loescht fuer eine Relation (Datenbanktabelle) mehrere Tupel.
	 * Jedes Tupel muss fuer jede Spalte einen Wert enthalten, ansonsten wird eine DatabaseException
	 * ausgeloest.
	 * @param relation Datenbanktabelle, aus der Tupel geloescht werden sollen.
	 * @param tuples Liste der Tupel, die aus der Tabelle geloescht werden sollen.
	 * @throws DatabaseException Wird geworfen, wenn ein Fehler in der Datenbank auftritt (z.B. Constraints verletzt oder Verbindungsabbruch) oder wenn die Tupel nicht die richtige Form haben.
	 * @return The number of tuples that were deleted.
	 */
	private int deleteTuples( String relation, List<List<ConstantNode>> tuples ) throws DatabaseException {
		List<Column> columns = this.db.getColumns( relation );
		int deleted = 0;
		
		for ( List<ConstantNode> tuple : tuples ) {
			if ( columns.size() != tuple.size() ) {
				throw new DatabaseException("Tuple has wrong number of columns", null);
			}
			
			SQLDelete sql = this.db.newDelete();
			Iterator<Column> columnIter = columns.iterator();
			for ( ConstantNode columnValue : tuple ) {
				sql.where( columnIter.next().getName(), columnValue.toString( formatter ) );
			}
			
			int deleteCount = sql.delete( relation );
			deleted += deleteCount;
		}
		
		return deleted;
	}
	
	public void truncateRelation(String relationname) throws DatabaseException {
		this.db.rawQuery("TRUNCATE TABLE " + relationname);
	}
	
	
	
	
	
	/**
	 * Returns the current active domain for an attribute (column).
	 * The returned list is a copy and can be modified.
	 * @return
	 * @throws DatabaseException 
	 */
	public Set<String> getActiveDomain( Column column ) throws DatabaseException {
		// Load active domain on-demand.
		Set<String> columnActiveDomain = this.activeDomain.get( column );
		if ( columnActiveDomain == null ) {
			columnActiveDomain = this.db.getAttributeValues(column.getTable(), column.getName());
			this.loadedActiveDomain( column, columnActiveDomain );
		}
		
		return new HashSet<String>( columnActiveDomain );
	}
	
	/**
	 * Resets the current active domain.
	 */
	public void resetActiveDomain() {
		this.activeDomain = new HashMap<Column, Set<String>>();
	}
	
	/**
	 * This method MUST be called whenever an active domain is loaded on-demand.
	 * @param column Reference to the column for which the active domain was loaded.
	 * @param values Active domain values.
	 * @throws DatabaseException 
	 */
	private void loadedActiveDomain( Column column, Set<String> values ) throws DatabaseException {
		this.activeDomainSizeCounter += values.size();
		
		// Scale the dictionary down until we're in the size limit.
		while ( this.activeDomainSizeCounter > ACTIVE_DOMAIN_SIZE_LIMIT ) {
			if ( this.loadedActiveDomainHistory.isEmpty() ) {
				// Removed every element but still not enough size for the newly loaded dictionary. Bad luck.
				logger.warn("Trying to scale down cache, but newly loaded active domain exceeds cache limit.");
				break;
			}
			
			Column loadedColumn = this.loadedActiveDomainHistory.get(0);
			this.activeDomainSizeCounter -= this.activeDomain.get(loadedColumn).size();
			this.activeDomain.remove( loadedColumn );
		}
		
		this.activeDomain.put( column, values );
		this.loadedActiveDomainHistory.add( column );
	}
}
