package server.database;


import java.io.File;
import java.io.IOException;
import java.util.Vector;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import server.util.Tuple;


import exception.DatabaseConfigurationException;

/**
 * Enthaelt eine Liste von DatabaseConfiguration-Tupeln, die jeweils aus einem UserDatabase- (erstes Element)
 * und einer Database- (zweites Element) Konfiguration bestehen.
 */
public class DatabaseConfigurationList extends Vector<Tuple<DatabaseConfiguration,DatabaseConfiguration>> {
	/**
	 * Automatisch generierte Serial (wird zum Serialisieren benoetigt).
	 */
	private static final long serialVersionUID = 571306706821014298L;
	
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");
	
	/**
	 * Load default database configuration file.
	 */
	public static DatabaseConfigurationList DEFAULT_LIST = null;
	static {
		try {
			DEFAULT_LIST = new DatabaseConfigurationList("db.xml");
		} catch ( DatabaseConfigurationException e ) {
			logger.fatal("Unable to load default configuration file", e);
			System.exit(-1);
		}
	}
	
	public DatabaseConfigurationList( String filename ) throws DatabaseConfigurationException {
		this.loadFromFile( new File(filename) );
	}

	/**
	 * Laedt eine XML-Konfiguration aus einer Datei und fuegt alle Elemente dieser DatabaseConfigurationList
	 * hinzu. Wenn die Methode mehrmals fuer unterschiedliche Dateien aufgerufen wird, so wird die Liste
	 * vorher _nicht_ geloescht.
	 * @param xmlFile Einzulesende XML-Datei. Das Format muss der dbconfig.xsd entsprechen.
	 */
	public void loadFromFile( File xmlFile ) throws DatabaseConfigurationException {
		
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
				
		try {
			// XML-Dokument einlesen.
			builder = factory.newDocumentBuilder();
			Document doc = builder.parse( xmlFile );
			
			// Validierung der Konfigurationsdatei gegen das XML Schema.
			SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
			Schema schema = schemaFactory.newSchema(DatabaseConfigurationList.class.getResource(("dbconfig.xsd")));
			Validator validator = schema.newValidator();
			validator.validate(new DOMSource(doc));
			
			// Liste mit allen DatabaseConfiguration-Elementen erstellen.
			NodeList nodes = doc.getElementsByTagName("DatabaseConfiguration");
			
			for (int i = 0; i < nodes.getLength(); i++) {
				// Aktuelles DatabaseConfiguration-Element.
				Element currentElement = (Element) nodes.item(i);
				
				// MaintainDB und CQEDB aus der Liste der Kindknoten raussuchen.
				// In den Kindknoten sind auch "leere" Knoten (nur Whitespaces/Newlines) enthalten,
				// deshalb ist ein direkter Zugriff ueber den Index nicht moeglich / sinnvoll.
				Element idElement = (Element)currentElement.getElementsByTagName("Identifier").item(0);
				Element maintainElement = (Element)currentElement.getElementsByTagName("MaintainDB").item(0);
				Element cqeElement = (Element)currentElement.getElementsByTagName("CQEDB").item(0);

				// Konkrete Konfiguration aus den entsprechenden Kindknoten auslesen.
				String id = idElement.getFirstChild().getNodeValue();
				DatabaseConfiguration maintainDB = this.getDatabaseConfigurationFromNodeList( maintainElement.getChildNodes() );
				DatabaseConfiguration cqeDB = this.getDatabaseConfigurationFromNodeList( cqeElement.getChildNodes() );	
				maintainDB.setId( "MaintainDB: "+id );
				cqeDB.setId( "AppDB: "+id );
				
				// Fuege neues Tupel der beiden Konfigurationen in die Liste ein.
				Tuple<DatabaseConfiguration,DatabaseConfiguration> tupel = new Tuple<DatabaseConfiguration,DatabaseConfiguration>(maintainDB,cqeDB);
				this.add( tupel );
			}
			
		
		} catch (ParserConfigurationException e) {
			throw new DatabaseConfigurationException("Couldn't create DocumentBuilder instance", e);
		} catch (SAXException e) {
			throw new DatabaseConfigurationException("Configuration parse error (syntax)", e);
		} catch (IOException e) {
			throw new DatabaseConfigurationException("I/O error during configuration parsing", e);
		}
	}
	
	/**
	 * Liest aus den Kindknoten einer "MaintainDB" oder "CQEDB" die eigentliche Konfiguration (gleiche Syntax).
	 * Die Kindknoten koennen auch komplett "leer" sein (nur Whitespaces oder Newlines), deshalb
	 * ist ein Zugriff ueber den Index nicht moeglich / sinnvoll. Stattdessen wird ueber alle Knoten in einer
	 * Liste iteriert und nur die wichtigen rausgesucht.
	 * @param list
	 * @return
	 */
	private DatabaseConfiguration getDatabaseConfigurationFromNodeList( NodeList list ) {
		DatabaseConfiguration config = new DatabaseConfiguration();
		
		for ( int j = 0; j < list.getLength(); j++ ) {
			// Der Name des Knoten ist in der aktuellen Node, waehrend der eigentliche Inhalt
			// in dessen einzigem Kindknoten ist (getFirstChild).
			Node node = list.item( j );
			Node child = node.getFirstChild();
			String name = node.getNodeName();
			
			if ( name.equals("Type") ) {
				config.setType( child.getNodeValue() );
			} else if ( name.equals("Server") ) {
				config.setServer( child.getNodeValue() );
			} else if ( name.equals("Port") ) {
				config.setPort( Integer.parseInt(child.getNodeValue()) );
			} else if ( name.equals("DB") ) {
				config.setDb( child.getNodeValue() );
			} else if ( name.equals("Login") ) {
				config.setLogin( child.getNodeValue() );
			} else if ( name.equals("Password") ) {
				config.setPassword( child.getNodeValue() );
			}
		}
		
		return config;
	}
	
}
