package server.data;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import server.database.schema.Schema;
import server.database.schema.maintenance.UserTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.util.Observable;
import exception.DatabaseException;
import exception.UnknownUserException;

public class UserManagement extends Observable<UserManagement,User> {
	private SQLDatabase db;
	private PriorAll priorAll;
	private SchemaConstraints schemaConstraints;
	private final Map<Integer,User> userIdCache = Collections.synchronizedMap( new HashMap<Integer,User>() );
	private final Map<String,User> userUsernameCache = Collections.synchronizedMap( new HashMap<String,User>() );

	public UserManagement( SQLDatabase db, PriorAll priorAll, SchemaConstraints schemaConstraints ) {
		this.db = db;
		this.priorAll = priorAll;
		this.schemaConstraints = schemaConstraints;
	}
	
	/**
	 * Factory to load a user from the database.
	 * Already loaded users are cached, so all users
	 * have only one reference object.
	 * 
	 * @param id Id of the user.
	 * @return Reference to the user object.
	 * @throws DatabaseException 
	 * @throws UnknownUserException 
	 */
	public User load( int id ) throws DatabaseException, UnknownUserException {
		synchronized( this.userIdCache ) {
			// Try to answer request from cache.
			if ( this.userIdCache.containsKey(id) ) {
				return this.userIdCache.get(id);
			}

			SQLSelect sql = this.db.newSelect();
			sql.select(UserTableSchema.LOGIN).select(UserTableSchema.PASSWORD).select(UserTableSchema.ROLE_ID).select(UserTableSchema.CENSOR);
			sql.from(Schema.maintenanceDatabase.cqeUser).where(UserTableSchema.ID, id);
			SQLTable result = sql.get();

			if ( result.getRowCount() == 0 ) {
				throw new UnknownUserException(id);
			}

			String[] row = result.getFirstRow();
			return this.load( id, row[0], row[1], Integer.valueOf(row[2]), row[3] );
		}
	}
	
	/**
	 * Factory to load a user from the database.
	 * Already loaded users are cached, so all users
	 * have only one reference object.
	 * 
	 * @param username Name of the user.
	 * @return Reference to the user object.
	 * @throws DatabaseException 
	 * @throws UnknownUserException 
	 */
	public User load( String username ) throws DatabaseException, UnknownUserException {
		synchronized( this.userIdCache ) {
			// Try to answer request from cache.
			if ( this.userUsernameCache.containsKey(username) ) {
				return this.userUsernameCache.get(username);
			}

			SQLSelect sql = this.db.newSelect();
			sql.select(UserTableSchema.ID).select(UserTableSchema.LOGIN).select(UserTableSchema.PASSWORD).select(UserTableSchema.ROLE_ID).select(UserTableSchema.CENSOR);
			sql.from(Schema.maintenanceDatabase.cqeUser).where(UserTableSchema.LOGIN, username);
			SQLTable result = sql.get();

			if ( result.getRowCount() == 0 ) {
				throw new UnknownUserException(username);
			}

			String[] row = result.getFirstRow();
			return this.load( Integer.valueOf(row[0]), row[1], row[2], Integer.valueOf(row[3]), row[4] );
		}
	}
	
	/**
	 * Returns a list with all users available.
	 * The returned list can be modified without any changes to the database.
	 * On the other hand modifications of the user objects will lead to database modifications.
	 * 
	 * @return
	 * @throws DatabaseException
	 */
	public List<User> loadAll() throws DatabaseException {
		synchronized( this.userIdCache ) {
			SQLSelect sql = this.db.newSelect();
			sql.select(UserTableSchema.ID).select(UserTableSchema.LOGIN).select(UserTableSchema.PASSWORD).select(UserTableSchema.ROLE_ID).select(UserTableSchema.CENSOR);
			sql.from(Schema.maintenanceDatabase.cqeUser);
			SQLTable result = sql.get();

			LinkedList<User> users = new LinkedList<User>();

			for ( String[] row : result ) {
				int id = Integer.valueOf(row[0]);
				String username = row[1];
				String password = row[2];
				int roleId = Integer.valueOf(row[3]);
				String censor = row[4];

				User user = null;

				// Try to answer request from cache.
				if ( this.userIdCache.containsKey(id) ) {
					user = this.userIdCache.get(id);
				} else {
					user = this.load(id, username, password, roleId, censor);
				}

				users.add( user );
			}

			return users;
		}
	}
	
	/**
	 * Used to load the reference data (log, confPol, prior) for the user.
	 * @param id
	 * @param username
	 * @param password
	 * @param roleId
	 * @param censor
	 * @return
	 * @throws DatabaseException
	 */
	private User load( int id, String username, String password, int roleId, String censor ) throws DatabaseException {
		synchronized( this.userIdCache ) {
			Role role = Role.load( this.db, roleId );
			Log log = new Log( this.db, id );
			ConfidentialityPolicy confPol = new ConfidentialityPolicy( this.db, id );
			PriorUser priorUser = new PriorUser( this.db, id );
			User user = new User( this.db, id, username, password, censor, role, log, confPol, priorUser, this.priorAll, this.schemaConstraints );

			// Update cache.
			this.userIdCache.put( id, user );
			this.userUsernameCache.put( username, user );

			return user;
		}
	}
	
	/**
	 * Adds a new user.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param username
	 * @return
	 * @throws DatabaseException
	 */
	public User add( String username ) throws DatabaseException {
		synchronized( this.userIdCache ) {
			SQLInsert sql = this.db.newInsert();
			sql.setGenerated(UserTableSchema.ID);
			sql.set(UserTableSchema.LOGIN, username);
			sql.set(UserTableSchema.PASSWORD, "secret");
			sql.set(UserTableSchema.CENSOR, "DirectAccess");
			sql.set(UserTableSchema.ROLE_ID, 0);
			
			int num = sql.insert( Schema.maintenanceDatabase.cqeUser );
			SQLTable generatedValues = sql.getGeneratedValues();
			if ( num != 1 || generatedValues == null || generatedValues.getRowCount() != 1 ) {
				throw new DatabaseException("Failed to insert user. Rows inserted: "+num, null );
			}
			
			// Get ID for the entry and load user.
			int id = Integer.valueOf( generatedValues.getFirstRow()[0] );
			User user = this.load( id, username, "secret", 0, "AccessControlCensor" );
			
			// Notify observers.
			this.notifyObservers( Observable.Action.ADD, user );
			
			return user;
		}
	}
	
	/**
	 * Removes a user and all his log/confPol/prior entries.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param user
	 * @throws DatabaseException
	 */
	public void delete( User user ) throws DatabaseException {
		synchronized( this.userIdCache ) {
			// Clear all data.
			user.getLog().clear();
			user.getConfidentialityPolicy().clear();
			user.getPriorUser().clear();
			
			// Remove the user itself.
			SQLDelete sql = this.db.newDelete();
			sql.where(UserTableSchema.ID, user.getId());
			int deleteCount = sql.delete(Schema.maintenanceDatabase.cqeUser);

			if ( deleteCount != 1 ) {
				throw new DatabaseException("DELETE failed, wrong number of elements ("+deleteCount+") were deleted", null);
			}
			
			// Remove references to user object from cache
			this.userIdCache.remove(user.getId());
			this.userUsernameCache.remove(user.getUsername());
			
			// Notify observers.
			this.notifyObservers( Observable.Action.DELETE, user );
		}
	}
}
