package server.data;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownLogException;

import server.database.schema.Schema;
import server.database.schema.SchemaColumn;
import server.database.schema.maintenance.LogTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.parser.Formula;
import server.parser.ParseException;
import server.parser.Parser;
import server.parser.node.AtomPredicateNode;
import server.util.Observable;
import user.IdAndFormula;

public class Log extends Observable<Log,LogEntry> implements Iterable<LogEntry> {

	private CopyOnWriteArrayList<LogEntry> entries = null;
	private SQLDatabase db = null;
	private int userId;
	
	/**
	 * Loads the log of a user from the database.
	 * ONLY to be called by {@link User#load(int)} or {@link User#load(String)}.
	 * The visibility of the constructor is therefore limited to the data package.
	 * @param db DB from which the log will be loaded.
	 * @param userId Id of the user.
	 * @throws DatabaseException 
	 */
	Log( SQLDatabase db, int userId ) throws DatabaseException {
		this.entries = new CopyOnWriteArrayList<LogEntry>();
		this.db = db;
		this.userId = userId;
		
		// Load users log from database.
		SQLSelect sql = this.db.newSelect();
		sql.select(LogTableSchema.ID).select(LogTableSchema.INTERACTION).select(LogTableSchema.INTERACTIONTYPE).select(LogTableSchema.INTERACTIONDATE).select(LogTableSchema.ANSWER).select(LogTableSchema.ANSWERFORMULATYPE).select(LogTableSchema.EFFECTIVE_UPDATES);
		sql.from(Schema.maintenanceDatabase.log);
		sql.where(LogTableSchema.USER_ID, userId);
		
		SQLTable result = sql.get();
		for ( String[] row : result ) {
			try {
				int id = Integer.valueOf( row[0] );
				Formula interaction = new Formula( row[1] );
				String interactionType = row[2];
				String interactionDate = row[3];
				Formula answer = new Formula( row[4] );
				String answerType = row[5];
				Set<AtomPredicateNode> effectiveUpdates;
				if( row[6] == null )
					effectiveUpdates = new HashSet<AtomPredicateNode>();
				else
					effectiveUpdates = new Parser(row[6]).updateSet();
				// Create new LogEntry and add to list.
				LogEntry entry = new LogEntry(this, id, interaction, interactionType, interactionDate, answer, answerType, effectiveUpdates);
				this.entries.add( entry );
			} catch (ParseException e) {
				throw new DatabaseException("Unable to parse log entry (malformed effective update)", e);
			} catch (ParserException e) {
				throw new DatabaseException("Unable to parse log entry (malformed formula)", e);
			} catch (NumberFormatException e) {
				throw new DatabaseException("Unable to parse log entry (malformed id)", e);
			}
		}
	}
	
	/**
	 * Writes the changes of a LogEntry to the database and notifies all observers.
	 * This method has the package visibility because only methods in the LogEntry class should call it!
	 * 
	 * @param entry The LogEntry that has changed.
	 * @param column The column which will get a new value.
	 * @param value The new value of the column.
	 * @throws DatabaseException
	 */
	void updateEntry( LogEntry entry, SchemaColumn column, String value ) throws DatabaseException {
		SQLUpdate sql = this.db.newUpdate();
		sql.set(column, value);
		sql.where(LogTableSchema.ID, entry.getId());
		sql.update( Schema.maintenanceDatabase.log );
		
		this.notifyObservers( Observable.Action.MODIFY, entry );
	}
	
	/**
	 * Adds a new entry to the log.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param interaction The interaction of the user. The type of the interaction is read from this object too.
	 * @param interactionDate Date of the interaction. Format must be YYYY-MM-DD or null for the current date.
	 * @param answer Answer to the interaction. Can be null (no answer given).
	 * @param answerType The type of the answer (e.g. completeness sentence).
	 * @param effectiveUpdates Used only for updates. Stores the actually changed values.
	 * @return
	 * @throws DatabaseException
	 * @throws ParserException 
	 */
	public synchronized LogEntry add( String interaction, String interactionType, String interactionDate, String answer, String answerType, Set<AtomPredicateNode> effectiveUpdates ) throws DatabaseException, ParserException {
		// Copy all mutable objects.
		Formula interactionFormula = new Formula(interaction);
		interactionFormula.setInteractionType( CqeType.InteractionType.valueOf(interactionType) );
		Formula answerFormula = new Formula(answer);
		answerFormula.setFormulaType( Formula.FormulaType.valueOf(answerType) );
		
		return this.add(interactionFormula, interactionDate, answerFormula, effectiveUpdates);
	}

	/**
	 * Adds a new entry to the log.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param interaction The interaction of the user. The type of the interaction is read from this object too.
	 * @param interactionDate Date of the interaction. Format must be YYYY-MM-DD or null for the current date.
	 * @param answer Answer to the interaction. The type of the answer (e.g. completeness sentence) is read from this object too. Can be null (no answer given).
	 * @param effectiveUpdates Used only for updates. Stores the actually changed values.
	 * @return
	 * @throws DatabaseException
	 */
	public synchronized LogEntry add( Formula interaction, String interactionDate, Formula answer, Set<AtomPredicateNode> effectiveUpdates ) throws DatabaseException {
		// Copy all mutable objects.
		interaction = new Formula( interaction );
		if ( answer != null ) {
			answer = new Formula( answer );
		}
		
		Set<AtomPredicateNode> effUp = new HashSet<AtomPredicateNode>();
		if ( effectiveUpdates != null ) {
			for ( AtomPredicateNode update : effectiveUpdates ) {
				effUp.add( (AtomPredicateNode)update.clone() );
			}
		}
		
		// Create new database entry.
		SQLInsert sql = this.db.newInsert();
		sql.setGenerated( LogTableSchema.ID );
		sql.set( LogTableSchema.INTERACTION, interaction.toString() );
		sql.set( LogTableSchema.INTERACTIONTYPE, interaction.getInteractionType().toString() );
		if ( interactionDate != null ) {
			sql.set( LogTableSchema.INTERACTIONDATE, interactionDate );
		}
		if ( answer != null ) {
			sql.set( LogTableSchema.ANSWER, answer.toString() );
			sql.set( LogTableSchema.ANSWERFORMULATYPE, answer.getFormulaType().toString() );
		}
		if ( !effUp.isEmpty() ) {
			sql.set( LogTableSchema.EFFECTIVE_UPDATES, effUp.toString() );
		}
		sql.set( LogTableSchema.USER_ID, this.userId );
		
		int num = sql.insert( Schema.maintenanceDatabase.log );
		SQLTable generatedValues = sql.getGeneratedValues();
		if ( num != 1 || generatedValues == null || generatedValues.getRowCount() != 1 ) {
			throw new DatabaseException("Failed to insert log entry. Rows inserted: "+num, null );
		}
		
		// Get ID for the entry.
		int id = Integer.valueOf( generatedValues.getFirstRow()[0] );
		
		// Create new LogEntry and add to list.
		LogEntry entry = new LogEntry(this, id, interaction, interaction.getInteractionType().toString(), interactionDate, answer, answer != null ? answer.getFormulaType().toString() : null, effUp);
		this.entries.add( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.ADD, entry );
		
		return entry;
	}
	
	/**
	 * Removes a log entry.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param entry
	 * @throws DatabaseException
	 */
	public synchronized void remove( LogEntry entry ) throws DatabaseException {
		if ( entry.getLog() != this ) {
			throw new DatabaseException("LogEntry doesn't belong to this Log.", null);
		}
		
		// Remove from database.
		SQLDelete sql = this.db.newDelete();
		sql.where(LogTableSchema.ID, entry.getId());
		sql.delete(Schema.maintenanceDatabase.log);
		
		// Remove from list.
		this.entries.remove( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.DELETE, entry );
	}
	
	/**
	 * Removes all log-entries of this user.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @throws DatabaseException
	 */
	public synchronized void clear() throws DatabaseException {
		// Remove all log entries for this user.
		SQLDelete sql = this.db.newDelete();
		sql.where(LogTableSchema.USER_ID, this.userId);
		sql.delete(Schema.maintenanceDatabase.log);
		
		// Notify observers.
		for ( LogEntry entry : this.entries ) {
			this.notifyObservers( Observable.Action.DELETE, entry );
		}
		
		// Clear the list.
		this.entries.clear();
	}
	
	/**
	 * Returns a specific log entry with the given id.
	 * @param id The Id of the log entry that will be returned.
	 * @return A log entry. Cannot be null.
	 * @throws UnknownLogException Throws the Exception if the log entry with the given id does not exist.
	 */
	public synchronized LogEntry get( int id ) throws UnknownLogException {
		for ( LogEntry entry : this.entries ) {
			if (entry.getId() == id) {
				return entry;
			}
		}
		throw new UnknownLogException( id ); 
	}
	
	
	/**
	 * Produces a copy of the log with less information.
	 * Currently used to feed the client with information.
	 * FIXME: Send more data to the client?
	 * @return Copy of the log as a List of IdAndFormula.
	 */
	public synchronized List<IdAndFormula> getCopyAsList() {
		List<IdAndFormula> result = new LinkedList<IdAndFormula>();
		
		Iterator<LogEntry> iter = this.iterator();
		while ( iter.hasNext() ) {
			LogEntry entry = iter.next();
			result.add( new IdAndFormula(entry.getId(), entry.getAnswer()) );
		}
		
		return result;
	}

	@Override
	public synchronized Iterator<LogEntry> iterator() {
		return this.entries.iterator();
	}
	
	/**
	 * Returns the number of log entries.
	 * @return Log count.
	 */
	public int size() {
		return this.entries.size();
	}
	
	/**
	 * Returns whether the log is empty or not.
	 * @return True if the the log has no entries, false otherwise.
	 */
	public boolean isEmpty() {
		return this.entries.isEmpty();
	}
}
