package server.data;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import server.database.schema.TableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.parser.Formula;
import server.util.Observable;
import user.IdAndFormula;
import exception.DatabaseException;
import exception.UnknownIdAndFormulaException;

/**
 * This class is used as parent for all Data-Classes that have only
 * an ID, FORMULA and an optional USER_ID entry.
 * Because of this, the SchemaColumn cannot directly be referenced and
 * Strings are used. This isn't the optimal solution and maybe this
 * class will be splitted for every use case.
 */
public abstract class IdAndFormulaTable extends Observable<IdAndFormulaTable,IdAndFormulaEntry> implements Iterable<IdAndFormulaEntry> {
	
	protected CopyOnWriteArrayList<IdAndFormulaEntry> entries;
	protected SQLDatabase db;
	private TableSchema table;
	private int userId;
	
	/**
	 * Loads a table only containing an ID and a Formula from the database.
	 * The visibility of the constructor is limited to the data package.
	 * @param db DB from which the data will be loaded.
	 * @param table The table used to load the data from.
	 * @throws DatabaseException
	 */
	IdAndFormulaTable( SQLDatabase db, TableSchema table ) {
		this(db, table, -1);
	}
	
	/**
	 * Loads a table only containing an ID and a Formula from the database.
	 * The visibility of the constructor is limited to the data package.
	 * @param db DB from which the data will be loaded.
	 * @param table The table used to load the data from.
	 * @param userId Optional User ID. If the id is not equal to -1 it will be used 
	 */
	IdAndFormulaTable( SQLDatabase db, TableSchema table, int userId ) {
		this.entries = new CopyOnWriteArrayList<IdAndFormulaEntry>();
		this.db = db;
		this.table = table;
		this.userId = userId;
	}

	/**
	 * Writes the changes of a IdAndFormulaEntry to the database and notifies all observers.
	 * This method has the package visibility because only methods in the IdAndFormulaEntry class should call it!
	 * 
	 * @param entry The IdAndFormulaEntry that has changed.
	 * @param column The column which will get a new value.
	 * @param value The new value of the column.
	 * @throws DatabaseException
	 */
	void updateFormulaEntry( IdAndFormulaEntry entry, String value ) throws DatabaseException {
		SQLUpdate sql = this.db.newUpdate();
		sql.set("FORMULA", value);
		sql.where("ID", entry.getId());
		sql.update(this.table);
		
		this.notifyObservers( Observable.Action.MODIFY, entry );
	}
	
	/**
	 * Adds a new entry to the table supported by this IdAndFormulaTable.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param formula
	 * @param type
	 * @param preservation
	 * @return
	 * @throws DatabaseException
	 */
	public synchronized IdAndFormulaEntry add( Formula formula ) throws DatabaseException {
		// Copy all mutable objects.
		formula = new Formula(formula);
		
		// Create new database entry.
		SQLInsert sql = this.db.newInsert();
		sql.setGenerated( "ID" );
		sql.set( "FORMULA", formula.toString() );
		if ( this.userId != -1 ) {
			sql.set( "USER_ID", this.userId );
		}
		
		int num = sql.insert( this.table );
		SQLTable generatedValues = sql.getGeneratedValues();
		if ( num != 1 || generatedValues == null || generatedValues.getRowCount() != 1 ) {
			throw new DatabaseException("Failed to insert entry. Rows inserted: "+num, null );
		}
		
		// Get ID for the entry.
		int id = Integer.valueOf( generatedValues.getFirstRow()[0] );
		
		// Create new LogEntry and add to list.
		IdAndFormulaEntry entry = new IdAndFormulaEntry(this, id, formula);
		this.entries.add( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.ADD, entry );
		
		return entry;
	}
	
	/**
	 * Removes an IdAndFormula entry.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param entry
	 * @throws DatabaseException
	 */
	public synchronized void remove( IdAndFormulaEntry entry ) throws DatabaseException {
		if ( entry.getIdAndFormulaTable() != this ) {
			throw new DatabaseException("IdAndFormulaEntry doesn't belong to this IdAndFormulaTable.", null);
		}
		
		// Remove from database.
		SQLDelete sql = this.db.newDelete();
		sql.where("ID", entry.getId());
		sql.delete(this.table);
		
		// Remove from list.
		this.entries.remove( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.DELETE, entry );
	}
	
	/**
	 * Removes all entries.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @throws DatabaseException
	 */
	public synchronized void clear() throws DatabaseException {
		// Remove all entries.
		SQLDelete sql = this.db.newDelete();
		if ( this.userId != -1 ) {
			sql.where("USER_ID", this.userId);
		}
		sql.delete(this.table);
		
		// Notify observers.
		for ( IdAndFormulaEntry entry : this.entries ) {
			this.notifyObservers( Observable.Action.DELETE, entry );
		}
		
		// Clear the list.
		this.entries.clear();
	}
	
	/**
	 * Returns a specific IdAndFormulaEntry with the given id.
	 * @param id The Id of the IdAndFormulaEntry that will be returned.
	 * @return A IdAndFormulaEntry. Cannot be null.
	 * @throws UnknownIdAndFormulaEntryException Throws the Exception if the IdAndFormulaEntry with the given id does not exist.
	 */
	public synchronized IdAndFormulaEntry get( int id ) throws UnknownIdAndFormulaException {
		for ( IdAndFormulaEntry entry : this.entries ) {
			if (entry.getId() == id) {
				return entry;
			}
		}
		throw new UnknownIdAndFormulaException( id ); 
	}
	
	/**
	 * Produces a copy of the table.
	 * Currently used to feed the client with information.
	 * @return Copy of the table.
	 */
	public synchronized List<IdAndFormula> getCopyAsList() {
		List<IdAndFormula> result = new LinkedList<IdAndFormula>();
		
		for ( IdAndFormulaEntry entry : this.entries ) {
			IdAndFormula item = new IdAndFormula( entry.getId(), entry.getFormula() );
			result.add( item );
		}
		
		return result;
	}
	
	@Override
	public synchronized Iterator<IdAndFormulaEntry> iterator() {
		return this.entries.iterator();
	}
	
	/**
	 * Returns the number of entries.
	 * @return Count.
	 */
	public int size() {
		return this.entries.size();
	}
	
	/**
	 * Returns whether the table is empty or not.
	 * @return True if the the table has no entries, false otherwise.
	 */
	public boolean isEmpty() {
		return this.entries.isEmpty();
	}
}
