package server.data;

import exception.DatabaseException;
import server.parser.Formula;

/**
 * One Id and Formula entry. The get methods will only return immutable
 * objects like Strings or copies of mutable objects like Formulas.
 * This means any changes MUST be done through the set methods to
 * take any effect.
 * 
 * Any call of a set method will lead to a notification for all IdAndFormulaData-observers.
 */
public class IdAndFormulaEntry {

	private IdAndFormulaTable parent;
	private int id;
	private Formula formula;
	
	/**
	 * Creates a new IdAndFormulaEntry. All mutable values (like formulas) must be copied beforehand!
	 * This constructor has the package visibility because it should only be called by methods of the class IdAndFormulaTable.
	 * 
	 * @param parent
	 * @param id
	 * @param formula
	 */
	IdAndFormulaEntry( IdAndFormulaTable parent, int id, Formula formula ) {
		this.parent = parent;
		this.id = id;
		this.formula = formula;
	}
	
	/**
	 * Returns the unique id. The id cannot be changed.
	 * @return
	 */
	public int getId() {
		return this.id;
	}
	
	/**
	 * Returns the a copy of the formula.
	 * Modifying this copy will have no effect on the database.
	 * @return
	 */
	public Formula getFormula() {
		return new Formula( this.formula );
	}
	
	/**
	 * Sets the formula to a new value.
	 * A copy of the given formula will be created. Modifying the formula will have no effect on the database.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param formula
	 * @throws DatabaseException
	 */
	public void setFormula( Formula formula ) throws DatabaseException {
		this.formula = new Formula( formula );
		this.parent.updateFormulaEntry( this, formula.toString() );
	}
	
	/**
	 * Returns the table for that this IdAndFormulaEntry belongs to.
	 * @return
	 */
	public IdAndFormulaTable getIdAndFormulaTable() {
		return this.parent;
	}
}
