package server.data;

import server.database.schema.maintenance.ConfidentialityPolicyTableSchema;
import server.parser.Formula;
import communication.CqeType;
import exception.DatabaseException;

/**
 * One Confidentiality Policy entry. The get methods will only return immutable
 * objects like Strings or copies of mutable objects like Formulas.
 * This means any changes MUST be done through the set methods to
 * take any effect.
 * 
 * Any call of a set method will lead to a notification for all ConfidentialityPolicy-observers.
 */
public class ConfidentialityPolicyEntry {

	private ConfidentialityPolicy confidentialityPolicy;
	private int id;
	private Formula formula;
	private CqeType.PolicyType type;
	private CqeType.PolicyPreservation preservation;
	
	/**
	 * Creates a new ConfidentialityPolicyEntry. All mutable values (like formulas) must be copied beforehand!
	 * This constructor has the package visibility because it should only be called by methods of the class ConfidentialityPolicy.
	 * 
	 * @param confPol
	 * @param id
	 * @param formula
	 * @param type
	 * @param preservation
	 */
	ConfidentialityPolicyEntry( ConfidentialityPolicy confPol, int id, Formula formula, CqeType.PolicyType type, CqeType.PolicyPreservation preservation ) {
		this.confidentialityPolicy = confPol;
		this.id = id;
		this.formula = formula;
		this.type = type;
		this.preservation = preservation;
	}
	
	/**
	 * Returns the unique id. The id cannot be changed.
	 * @return
	 */
	public int getId() {
		return this.id;
	}
	
	/**
	 * Returns the a copy of the formula.
	 * Modifying this copy will have no effect on the confidentiality policy.
	 * @return
	 */
	public Formula getFormula() {
		return new Formula( this.formula );
	}
	
	/**
	 * Sets the formula to a new value.
	 * A copy of the given formula will be created. Modifying the formula will have no effect on the confidentiality policy.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param formula
	 * @throws DatabaseException
	 */
	public void setFormula(Formula formula) throws DatabaseException {
		this.formula = new Formula( formula );
		this.confidentialityPolicy.updateEntry( this, ConfidentialityPolicyTableSchema.FORMULA, this.formula.toString() );
	}
	
	/**
	 * Returns the type.
	 * @return
	 */
	public CqeType.PolicyType getType() {
		return this.type;
	}
	
	/**
	 * Sets the type to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param type
	 * @throws DatabaseException
	 */
	public void setType(CqeType.PolicyType type) throws DatabaseException {
		this.type = type;
		this.confidentialityPolicy.updateEntry( this, ConfidentialityPolicyTableSchema.POLICY_TYPE, this.type.toString() );
	}
	
	/**
	 * Returns the preservation.
	 * @return
	 */
	public CqeType.PolicyPreservation getPreservation() {
		return this.preservation;
	}

	/**
	 * Sets the preservation to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param preservation
	 * @throws DatabaseException
	 */
	public void setPreservation(CqeType.PolicyPreservation preservation) throws DatabaseException {
		this.preservation = preservation;
		this.confidentialityPolicy.updateEntry( this, ConfidentialityPolicyTableSchema.PRESERVATION, this.preservation.toString() );
	}
	
	/**
	 * Returns the confidentiality policy this entry belongs to.
	 * @return
	 */
	public ConfidentialityPolicy getConfidentialityPolicy() {
		return this.confidentialityPolicy;
	}
}
