package server.core.notificationHandler;


import java.io.IOException;
import java.util.EnumSet;

import notification.Notification;
import notification.serverToClient.ExceptionNotification;

import org.apache.log4j.Logger;

import communication.CqeType;

import exception.CqeException;
import exception.PermissionDeniedException;

import server.core.Client;

/**
 * This class must be extended by all classes that want to process a notification.
 * Each class extending this one can only process one specific notification.
 * The type of notification that the class can process must be specified twice:
 * 1. As generic type
 * 2. In the constructor by calling the parent constructor with the corresponding class (e.g. "super(LoginNotification.class);").
 * 
 * A notification will only be processed if the client/user has sufficient permissions.
 * See {@link #NotificationHandler(Class, EnumSet)} for details.
 * 
 * The Handler implements Runnable because they are run in a thread pool.
 * This means you must be aware of concurrent data access and do proper synchronization.
 * 
 * @param <N> Type of the notification that can be processed by the extending class.
 */
public abstract class NotificationHandler<N extends Notification> implements Runnable {
	protected final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.core");
	
	// The client and notification are directly set before the handler will be started.
	private Client client = null;
	private N notification = null;
	// Store the class type of this handler.
	private Class<N> handledClass = null;
	// Needed client permissions to execute this handler.
	private EnumSet<CqeType.Right> neededPermissions;
	
	/**
	 * Creates a new notification handler. Must be called by the subclass (see class comment above). The client needs no special permissions.
	 * @param handledClass Class of the notification that can be handled by the subclass.
	 */
	public NotificationHandler( Class<N> handledClass ) {
		this.client = null;
		this.notification = null;
		this.handledClass = handledClass;
		this.neededPermissions = EnumSet.noneOf( CqeType.Right.class );
	}
	
	/**
	 * Creates a new notification handler. Must be called by the subclass (see class comment above).
	 * @param handledClass Class of the notification that can be handled by the subclass.
	 * @param neededPermissions A set of Rights that are needed by the client/user. A notification will only be processed if the granted permissions contains all rights of the needed permissions.
	 */
	public NotificationHandler( Class<N> handledClass, EnumSet<CqeType.Right> neededPermissions ) {
		this.client = null;
		this.notification = null;
		this.handledClass = handledClass;
		this.neededPermissions = neededPermissions;
	}
	
	/**
	 * Returns the class of the notification that can be processed by this notification handler.
	 * @return Class of the notification that can be handled.
	 */
	public Class<N> getHandledClass() {
		return this.handledClass;
	}
	
	/**
	 * Sets the data for processing. Should directly be called before the execution of this Runnable.
	 * @param client
	 * @param notification
	 */
	@SuppressWarnings("unchecked")
	public void setInput( Client client, Notification notification ) {
		this.client = client;
		this.notification = (N)notification;
	}

	/**
	 * Executes the notification handler. Only do this if you called {@link #setInput(Client, Notification)} beforehand!
	 * Checks if the user is logged in. If he's not and the notification handler isn't ok with that discards the notification and disconnects the client.
	 */
	@Override
	public void run() {
		if ( this.client == null || this.notification == null ) {
			logger.error("Notification handler wasn't set up with the needed data. This shouldn't happen!");
			return;
		}
		
		// Check if the client/user has the needed permissions.
		EnumSet<CqeType.Right> grantedPermissions;
		if ( this.client.getUser() == null ) {
			grantedPermissions = EnumSet.noneOf( CqeType.Right.class );
		} else {
			grantedPermissions = this.client.getUser().getRole().getRights();
			grantedPermissions.add( CqeType.Right.USER_LOGGED_IN );
		}
		
		if ( !grantedPermissions.containsAll(this.neededPermissions) ) {
			logger.debug("Client tried to send notification, but has insufficient permissions.");
			try {
				client.send( new ExceptionNotification(new PermissionDeniedException("Permission denied. Your permissions: "+grantedPermissions.toString()+" Needed Permissions: "+this.neededPermissions.toString(), null), ExceptionNotification.USERTOOSTUPID) );
			} catch (IOException e) {
				logger.error("Unable to send message to the client. Disconnecting.", e);
				this.client.disconnect();
			}
			return;
		}
		
		// Now handle the notification.
		try {
			this.handle( this.client, this.notification );
		} catch ( CqeException e ) {
			logger.warn("Notification handler couldn't perform its duty. Notifying client.", e);
			try {
				this.client.send( new ExceptionNotification(e) );
			} catch (IOException e1) {
				logger.error("Unable to send message to the client. Disconnecting.", e);
				this.client.disconnect();
			}
		} catch ( Throwable e ) {
			logger.error("Notification handler crashed. Disconnecting client.", e);
			this.client.disconnect();
		}
	}
	
	/**
	 * Must be implemented by the subclass. Contains the actual processing of the notification.
	 * 
	 * Throwing exceptions inside your handle() implementation will have the following consequences:
	 * 1. If the exception is a subclass of CqeException, the client will be notified about the exception and can react accordingly.
	 * 2. Otherwise it's assumed that the exception was severe (code errors or runtime problems). The exception will be logged and the client will be disconnected.
	 * 3. Errors (not explicitly thrown) will be handled like case 2.
	 * 
	 * Please make sure that in any case the state of the system is consistent. Don't throw exceptions and leave the system in an undetermined state!
	 * 
	 * @param client Client which send the notification.
	 * @param notification The notification that was received by the server.
	 * @throws Exception Throw CqeExceptions to indicate non-critical errors and any other exception type for critical ones. See method description for details.
	 */
	protected abstract void handle( Client client, N notification ) throws Exception;
}
