package server.censor.signature.experiment;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Set;

import org.apache.log4j.Logger;

import server.database.AppDatabase;
import server.database.DatabaseConfiguration;
import server.database.DatabaseConfigurationList;
import server.database.NewMaintenanceDatabase;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLTable;
import server.parser.Formula;
import server.util.Tuple;
import exception.DatabaseConfigurationException;
import exception.DatabaseException;
import exception.ParserException;


/**
 *  Diese Klasse dient der Ausfuehrung des Experiments zum SignatureCensor.
 *  Das Experiment ist in "Signature-Based Inference-Usability Confinement for Relational 
 *  Databases under Functional and Join Dependencies" von Biskup etc. beschrieben.
 *  Details des Experimentaufbaus koennen der Datei 'Experimentbeschreibung_DBSEC12.txt' 
 *  entnommen werden.
 */
public class ExperimentExecutionDBSEC12 extends ExperimentExecution {

	/** Logger*/
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.censor.signature");
	
	/** Name der zu schuetzenden Relation */
	protected static final String RELATION_NAME = "SIG_KRANKHEIT";
	
	
	public ExperimentExecutionDBSEC12(NewMaintenanceDatabase maintenanceDB, AppDatabase appDB, boolean flexible, int userID, boolean cumulatedStatistics) throws DatabaseException, IOException {
		super(maintenanceDB, appDB, RELATION_NAME, flexible, userID, cumulatedStatistics);
	}
	
	public ExperimentExecutionDBSEC12(Tuple<DatabaseConfiguration, DatabaseConfiguration> dbConfig,	boolean flexible, int userID, boolean cumulatedStatistics) throws DatabaseException, IOException {
		super(dbConfig, RELATION_NAME, flexible, userID, cumulatedStatistics);
	}

	@Override
	protected List<Formula> randomQuerys(int numberOfQueries) throws DatabaseException, ParserException {
		SQLDatabase db = this.appDB.getSQLDatabase();
		
		SQLTable table = db.query( "SELECT DISTINCT * FROM " + this.relationname );
		//Random random = new Random(45681318424841123L); // alter seed
		//Random random = new Random(58471239874841123L); // Nachfolger des alten Seeds (bis 18.05.12)
		Random random = new Random();
		List<Formula> querySequence = new ArrayList<Formula>(numberOfQueries);
		
		Formula[] candidates = new Formula[7];
		candidates[0] = new Formula(relationname.toLowerCase() + "(S, D, P)");
		candidates[1] = new Formula("EXISTS X " + relationname.toLowerCase() + "(X, D, P)");
		candidates[2] = new Formula("EXISTS X " + relationname.toLowerCase() + "(S, X, P)");
		candidates[3] = new Formula("EXISTS X " + relationname.toLowerCase() + "(S, D, X)");
		candidates[4] = new Formula(relationname.toLowerCase() + "(\"extra constant\", D, P)");
		candidates[5] = new Formula(relationname.toLowerCase() + "(S, \"extra constant\", P)");
		candidates[6] = new Formula(relationname.toLowerCase() + "(S, D, \"extra constant\")");
		
		for (int i=0; i < numberOfQueries ;i++) {
			// waehle zufaellig eine der obigen Formel-Moeglichkeiten aus
			Formula formula = new Formula(candidates[random.nextInt(7)]);
			int randomTupel = random.nextInt(table.getRowCount());
			
			// belege freie Variablen mit Konstanten
			Set<String> freeVars = formula.getFreeVariables();
			Map<String, String> freeVarsToConstants = new HashMap<String, String>();
			for( String freeVar : freeVars ) {
				String constant = "";
				if( freeVar.equals("S") )
					constant = table.getRow( randomTupel ).get("SYMPTOM");
				else if( freeVar.equals("D") )
					constant = table.getRow( randomTupel ).get("DIAGNOSIS");
				if( freeVar.equals("P") )
					constant = table.getRow( randomTupel ).get("PATIENT");
				freeVarsToConstants.put(freeVar, constant);
			}
			formula.substituteVariables(freeVarsToConstants);
			querySequence.add(formula);
		}
		
		return querySequence;
	}
	
		
	
	/**
	 * Diese Methode fuehrt das Experiment wiederholt fuer unterschiedliche Instanz-Groessen
	 * (Vervielfaeltigungen der Originalinstanz (s. Paper)) durch.
	 * Die Instanz-Groessen, die Anzahl der Queries fuer die einzelnen Durchlaefe sowie alle 
	 * weiteren Parameter muessen in einer Properties-Datei spezifiziert werden.
	 * Der Pfad zur Properties-Datei muss der Methode als erster und einziger Parameter 
	 * uebergeben werden.
	 * 
	 * @param args	String-Array, das als erstes und einziges Element den Pfad zur Properties-
	 * 				Datei enthalten muss
	 */
	public static void main(String[] args) {
		// erstes Argument muss Pfad zu einer Properties-Datei sein
		if( args.length != 1 ) {
			logger.error("Invalid parameter count. Only one parameter specifying the properties file is allowed.");
			return;
		}
		
		// Lade Properties
		Properties experimentProperties = new java.util.Properties();
		try {
			logger.info("Trying to load properties file " + args[0]);
			InputStream is_properties = ExperimentExecutionDBSEC12.class.getClassLoader().getResourceAsStream(args[0]);
			if( is_properties == null ) {
				logger.error("Error while loading properties file: properties file not found.");
				return;
			}
			experimentProperties.load( is_properties );
			logger.info("Properties file " + args[0] + " successfuly loaded");
		} catch( IOException e ) {
			logger.error("Error while loading properties file: " + e.getMessage());
			return;
		}
		
		Tuple<DatabaseConfiguration, DatabaseConfiguration> dbConfig = null;
		boolean flexible;
		int userID;
		boolean cumulatedStatistics;
		List<Tuple<Integer, Integer>> runs;
		int firstRunID;
		
		
		// Datenbankkonfiguration
		DatabaseConfigurationList dbConfigList = null;
		String prop_DatabaseIdentifier = experimentProperties.getProperty("DatabaseIdentifier");
		if( prop_DatabaseIdentifier == null ) {
			logger.error("Error: Property 'DatabaseIdentifier' missing in properties file.");
			return;
		}
		try {
			dbConfigList = new DatabaseConfigurationList( "db.xml" );
		}
		catch(DatabaseConfigurationException e) {
			logger.error("Error while getting database configuration list.");
			return;
		}
		for ( Tuple<DatabaseConfiguration, DatabaseConfiguration> config : dbConfigList ) {
			if ( config.getFirstElement().getId().equals("MaintainDB: " + prop_DatabaseIdentifier) ) {
				// Diese Konfiguration verwenden
				dbConfig = config;
				break;
			}
		}
		if( dbConfig == null ) {
			logger.error("Error: No database found matching the database identifier " + experimentProperties.getProperty("DatabaseIdentifier"));
			return;
		}
		
		// Ansatz
		String prop_Approach = experimentProperties.getProperty("Approach");
		if( prop_Approach == null ) {
			logger.error("Error: Property 'Approach' missing in properties file.");
			return;
		}
		if( prop_Approach.equals("0") ) {
			flexible = false;
		} else if( prop_Approach.equals("1") ) {
			flexible = true;
		} else {
			logger.error("Error: Invalid approach. Approach must either be 0 (unflexible approach) or 1 (flexible approach).");
			return;
		}
		
		// BenutzerID
		String prop_UserID = experimentProperties.getProperty("UserID");
		if( prop_UserID == null ) {
			logger.error("Error: Property 'UserID' missing in properties file.");
			return;
		}
		try {
			userID = Integer.parseInt( prop_UserID );
			if( userID < 0 ) {
				logger.error("Error: Invalid UserID. UserID must be greater equal than 0.");
				return;
			}
		} catch( NumberFormatException e ) {
			logger.error("Error: Invalid UserID. UserID must be an integer.");
			return;
		}
		
		// Statistik
		String prop_CumulatedStatistics = experimentProperties.getProperty("CummulatedStatistics");
		if( prop_CumulatedStatistics == null ) {
			logger.error("Error: Property 'CumulatedStatistics' missing in properties file.");
			return;
		}
		if( prop_CumulatedStatistics.equals("0") ) {
			cumulatedStatistics = false;
		} else if( prop_CumulatedStatistics.equals("1") ) {
			cumulatedStatistics = true;
		} else {
			logger.error("Error: Invalid value for CumulatedStatistics. CumulatedStatistics must either be 0 (not cummulated) or 1 (cummulated).");
			return;
		}
		
		// Runs
		String prop_Runs = experimentProperties.getProperty("Runs");
		if( prop_Runs == null ) {
			logger.error("Error: Property 'Runs' missing in properties file.");
			return;
		}
		runs = new LinkedList<Tuple<Integer,Integer>>();
		for( String run : prop_Runs.split(",") ) {
			String[] runData = run.split(":");
			if( runData.length != 2 ) {
				logger.error("Error: Invalid run specification. Check your properties file!");
				return;
			}
			
			try {
				int sizeOfInstance = Integer.parseInt( runData[0] );
				if( sizeOfInstance < 0 ) {
					logger.error("Error: Invalid run specification. Size of instance must be greater than 0. Check your properties file!");
					return;
				}
				int numberOfQueries = Integer.parseInt( runData[1] );
				if( numberOfQueries < 0 ) {
					logger.error("Error: Invalid run specification. Number of queries must be greater than 0. Check your properties file!");
					return;
				}
				
				runs.add( new Tuple<Integer, Integer>(sizeOfInstance, numberOfQueries) );
			} catch( NumberFormatException e ) {
				logger.error("Error: Invalid run specification. Size of instance and number of queries must be integers. Check your properties file!");
				return;
			}
		}
		
		// FirstRunID
		String prop_FirstRunID = experimentProperties.getProperty("FirstRunID");
		if( prop_FirstRunID == null ) {
			logger.error("Error: Property 'FirstRunID' missing in properties file.");
			return;
		}
		try {
			firstRunID = Integer.parseInt( prop_FirstRunID );
			if( firstRunID < 0 ) {
				logger.error("Error: Invalid FirstRunID. FirstRunID must be greater equal than 0.");
				return;
			}
		} catch( NumberFormatException e ) {
			logger.error("Error: Invalid FirstRunID. FirstRunID must be an integer.");
			return;
		}
		
				
		// Fuehre Experimente durch
		try {
			ExperimentPreparationDBSEC12 experimentPreparation = new ExperimentPreparationDBSEC12(dbConfig, flexible, userID);
			ExperimentExecutionDBSEC12 experimentExecution = new ExperimentExecutionDBSEC12(dbConfig, flexible, userID, cumulatedStatistics);
			
			int nextRunID = firstRunID;
			int lastSizeOfInstance = -1;
			for( Tuple<Integer, Integer> run : runs ) {
				int sizeOfInstance = run.getFirstElement();
				int numberOfQueries = run.getSecondElement();
				
				// ggfs. Instanz anpassen
				if( sizeOfInstance != lastSizeOfInstance ) {
					// Groesse der Instanz muss veraendert werden
					logger.info("Preparing instance for runs with v=" + sizeOfInstance + ".");
					experimentPreparation.prepareExperiment(sizeOfInstance);
					lastSizeOfInstance = sizeOfInstance;
				}
				
				// Lauf durchfuehren
				logger.info("Performing run " + nextRunID + " with v=" + sizeOfInstance + " and " + numberOfQueries + " queries.");
				experimentExecution.executeExperiment(nextRunID++, numberOfQueries);
			}
		} catch( Exception e ) {
			logger.error("Error while executing experiment: " + e.getMessage());
			e.printStackTrace();
			return;
		}
		
		logger.info("All runs successfuly executed.");
	}
}
