package server.censor.refusal;

import java.util.LinkedList;
import java.util.List;

import server.censor.CensorUtil;
import server.censor.QueryCensor;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.MumFormula;
import server.theoremprover.ProverNine;
import server.theoremprover.TheoremProver;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;

import exception.DatabaseException;
import exception.ProverException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;

public class RefusalCensor extends QueryCensor {

	public RefusalCensor(Server server) {
		super("Refusal", server);
	}
	
	/**
	 * Die Methode implementiert die (dynamische) kontrollierte Anfrageauswertung mit dem Verweigerungs-Zensor bei bekannten potentiellen Geheimnissen
	 * und einer als vollstaendig angenommenen Datenbankinstanz. Falls durch die korrekte Antwort auf die gestellte Anfrage oder deren Negation die
	 * Sicherheitspolitik verletzt wird (ein beliebiges potentielles Geheimnis kann durch log u query oder log u -query erschlossen werden),
	 * wird die Beantwortung der Anfrage verweigert. Ansonsten wird die Anfrage stets korrekt beantwortet.
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 * @throws ProverException 
	 * @throws UnsupportedConfidentialityPolicyException 
	 */
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws DatabaseException, UnsupportedFormulaException, ProverException, UnsupportedConfidentialityPolicyException {
		boolean answer = false;
		boolean impliedByLogPos, impliedByLogNeg = false; //Zur Festellung, ob das Log die Anfrage impliziert
		List<Formula> result = new LinkedList<Formula>();

		List<IdAndFormula> log = client.getUser().getKnowledge().getCopyAsList();
		List<ConfidentialityPolicyItem> confidentialityPolicy = client.getUser().getConfidentialityPolicy().getCopyAsList();

		//positiv
		impliedByLogPos = this.isFormulaImpliedByLog(interaction, log);
		interaction.negate();
		impliedByLogNeg = this.isFormulaImpliedByLog(interaction, log);
		interaction.negate();

		// Nur wenn die Anfrage UND die negierte Anfrage nicht vom log impliziert werden, dann muss
		// die Verletzung der Geheimnisse ueberprueft werden.
		if ( impliedByLogPos == false & impliedByLogNeg == false) {

			answer = CensorUtil.isConfidentialityPolicyViolated( interaction, log, confidentialityPolicy ); //nicht negiert
			logger.debug( "Refusal-Zensor isConfidentialityPolicyViolated() #1: "+ answer );

			if ( answer == true ){ //wenn true, verweigern
				result.add( new MumFormula(interaction) );
				return result;
			}

			interaction.negate();

			answer = CensorUtil.isConfidentialityPolicyViolated(interaction, log, confidentialityPolicy); //negiert	
			logger.debug( "Refusal-Zensor isConfidentialityPolicyViolated() #1: "+ answer );

			interaction.negate(); //zuruecknegieren, um Originalzustand wieder herzustellen

			if ( answer == true ){ //wenn true, verweigern
				result.add( new MumFormula(interaction) );
				return result;
			}

		}

		// Ergebnis ist nicht gefaehrlich => Query auswerten.
		result.add( Formula.createFormula(client.getApplicationDB().evalStarComplete(interaction)) );

		return result;
	}
	
	/**
	 * Konstruiert die Eingabe fuer den Theorembeweiser: Fuer die Formelmenge log u query wird geprueft, ob das log die Anfrage bereits impliziert.
	 * Ist dies der Fall, so wird die Anfrage wahrheitgemaess beantwortet.
	 * Hinweis: Dient der Verwendung in modify
	 * @param query Die vom Benutzer gestellte Anfrage (bzw. deren Negation)
	 * @param log Das Vorwissen des Benutzers
	 * @return true, falls das log die query impliziert; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 * @author benet
	 */
	protected boolean isFormulaImpliedByLog( Formula query, List<IdAndFormula> log ) throws ProverException {
		/*
		 * Eigentlich impliziert die leere Menge jede beliebige Formel, allerdings wird diese Methode 
		 * nur zur Ueberpruefung, ob der Benutzer die Antwort auf seine Anfrage schon weiß, verwendet.
		 * Deshalb wird an dieser Stelle false statt true zurueckgegeben.
		 */
		if ( log.isEmpty() ){
			return false;
		}
		
		List<Formula> sos = new LinkedList<Formula>();

		// Benutzerwissen log Zeile fuer Zeile in Prover9-Eingabe schreiben
		for ( IdAndFormula logentry : log ) {
			sos.add( logentry.getFormula() );
		}
		
		// Pruefe, ob die Implikation log u query |= psi fuer ein psi aus pot_sec wahr ist
		TheoremProver tp = new ProverNine();
		boolean ergebnis = tp.prove( sos, query, true );

		return ergebnis;
	}
}
