package server.censor.precqe;

import java.util.ArrayList;
import java.util.List;

import server.parser.Formula;
import server.parser.node.Node;

/**
 * Baumknoten fuer den preCQE-Algorithmus.
 * Jeder Knoten besteht aus einem generischem Schluesselwert,
 * einer Liste von Constraints und der Anzahl der Luegen.
 *
 * @param <K> Schluesselwert eines Knotens. Muss fuer jeden Knoten eindeutig sein!
 */
public class TreeNode<K> {

    private TreeNode<K> parent;
    private ArrayList<TreeNode<K>> children;
    private K key;
    private List<Formula> constraints;
    private int lieCount;
    
    // Only one Formula can be marked in each node. Save it here
    // so that the marking can be done at the start of ground().
    private Node mark;

    /**
     * Erzeugt einen neuen Knoten ohne Kinder, Elter und Constraints.
     * 
     * @param key Schluesselwert fuer diesen Knoten.
     */
    public TreeNode(K key) {
    	this.parent = null;
    	this.children = new ArrayList<TreeNode<K>>();
        this.key = key;
        this.constraints = null;
        this.mark = null;
    }
    
    /**
     * Erzeugt einen neuen Knoten ohne Elter.
     * 
     * @param key Schluesselwert fuer diesen Knoten.
     * @param constraints Constraints.
     * @param lieCount Anzahl der Luegen.
     */
    public TreeNode(K key, List<Formula> constraints, int lieCount) {
    	this(key);
    	this.constraints = constraints;
    	this.lieCount = lieCount;
    }
    
    /**
     * Gibt den Elter-Knoten zurueck.
     * 
     * @return Elter-Knoten.
     */
    public TreeNode<K> getParent() {
        return this.parent;
    }
    
    /**
     * Setzt einen neuen Elter-Knoten.
     * 
     * @param t Neuer Elter.
     */
    public void setParent(TreeNode<K> parent) {
        this.parent = parent;
    }
    
    public K getKey() {
    	return this.key;
    }

    public void setKey(K i) {
        this.key = i;
    }
    
    public List<Formula> getConstraints() {
    	return this.constraints;
    }
    
    public void setConstraints(List<Formula> value) {
    	this.constraints = value;
    }
    
    public int getLieCount() {
    	return this.lieCount;
    }
    
    public void setLieCount(int lieCount) {
    	this.lieCount = lieCount;
    }
    
    public Node getMark() {
    	return this.mark;
    }
    
    public void setMark( Node mark ) {
    	this.mark = mark;
    }
    
    public TreeNode<K> getChild(K key) {
        for( TreeNode<K> t : this.children ) {
            if( t.getKey().equals(key) ) {
                return t;
            }
        }

        return null;
    }

    public List<TreeNode<K>> getChildren() {
        return this.children;
    }

    public void addChild(TreeNode<K> t) {
        t.parent = this;
        this.children.add(t);
    }

    public void delChild(K key) {
        for(int i = 0; i < children.size(); i++) {
            if( children.get(i).key.equals(key) ) {
                children.remove(i);
                return;
            }
        }
    }

    public void delChildren() {
        this.children.clear();
    }

    public int getChildCount() {
        return this.children.size();
    }
    
    /**
     * Gibt an, ob ein Knoten Geschwister hat (d.h. existierten andere
     * Knoten, die den selben Elter haben wie dieser Knoten).
     * 
     * @return true, falls Geschwister existieren, sonst false.
     */
    public boolean hasSibilings() {
        if(parent == null) {
            return false;
        }
        else {
            return parent.getChildCount() > 1;
        }
    }

    /**
     * Gibt den Pfad vom aktuellen Knoten zur Wurzel mit einer durch
     * Komma separierten Liste an. Die Liste startet mit der Wurzel
     * und endet beim aktuellen Knoten. Die Knoten werden durch ihren
     * eindeutigen Schluesselwert repraesentiert.
     * 
     * @return Pfad von der Wurzel zum aktuellen Knoten.
     */
    public String getPath() {
        if(parent == null) {
            return key.toString();
        }
        else {
            return parent.getPath() + ", " + key.toString();
        }
    }

    /**
     * ???
     * @return
     */
    public String getCrossPath() {
        String s = key.toString();

        if(parent != null) {
            //if(!isLeftChild()) {
            if(getChildCount() == 1) {
                s += ", " + parent.getCrossPath();
            }
        }

        return s;
    }

    /**
     * Gibt an, ob es sich beim aktuellen Knoten um den Wurzelknoten handelt.
     * @return true, falls der aktuelle Knoten kein Elter hat (Wurzel ist), sonst false.
     */
	public boolean isRoot() {
		return this.parent == null;
	}
	
	public String toString() {
		return this.key.toString();
	}
}
