package server.censor.precqe;

import java.util.LinkedList;
import java.util.List;

import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.ProverException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;
import server.censor.QueryCensor;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.node.AtomPredicateNode;
import server.parser.node.Node;


/**
 * Dieser Zensor leitet die Anfragen eines Users an eine inferenzsichere Datenbank um.
 * Tabellen dieser Datenbank haben ein Suffix, das den zugehoerigen User eindeutig identifiziert.
 */
public class PreCQECensor extends QueryCensor {

	public PreCQECensor( Server server ) {
		super("PreCQE", server);
		server.getNotificationHub().register( new PreCQENotificationHandler() );
	}

	protected List<Formula> run( Client client, Formula interaction ) throws DatabaseException, ProverException, UnsupportedFormulaException, DictionaryTooSmallException, UnsupportedConfidentialityPolicyException {
		
		//formula clonen und mit renameRelationnamesToUserSuffix umbenennen
		Formula userFormula = (Formula)interaction.clone();
		this.renameRelationnamesToUserSuffix( String.valueOf(client.getUser().getId()), userFormula );
		
		
		List<Formula> result = new LinkedList<Formula>();
		
		// Eval-Stern ausfuehren
		Formula evalStarQuery = Formula.createFormula( client.getApplicationDB().evalStarComplete( userFormula ) );
		result.add( evalStarQuery );
				
		return result;
	}
	
	
	/**
	 * Benennt die Praedikatenzeichen in der Anfrage des Users um, sodass die Anfrage nicht an die normale
	 * Datenbankinstanz, sondern an die inferenzsichere Instanz des Users gestellt wird. Dazu wird der Name
	 * des Users an die Praedikatenzeichen angehaengt.
	 * @param formula
	 */
	private void renameRelationnamesToUserSuffix( String identifier, Node formula ){
		String suffix = MarkedOracleSQLAppDatabase.userSeparator + identifier + "_b";
		
		// Usernamen als Suffix an das Praedikatenzeichen haengen 
		if (formula.isAtomPredicateNode()){
			AtomPredicateNode atom = (AtomPredicateNode)formula;
			
			atom.setRelationname( atom.getRelationname() + suffix );
		}
		
		// Aktueller Knoten ist kein Praedikatenzeichen? Dann weiter absteigen.
		else{
			for ( Node child : formula.getChildren() ){
				this.renameRelationnamesToUserSuffix( identifier, child );
			}
		}
    }
}
