package server.censor.combined;

import java.util.LinkedList;
import java.util.List;

import server.censor.CensorUtil;
import server.censor.QueryCensor;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.MumFormula;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;

import exception.DatabaseException;
import exception.ProverException;
import exception.UnsupportedFormulaException;


public class CombinedCensor extends QueryCensor {

	public CombinedCensor(Server server) {
		super("Combined", server);
	}
	
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws DatabaseException, UnsupportedFormulaException, ProverException {
		return this.run(client, interaction, client.getUser().getKnowledge().getCopyAsList());
	}

	/** 
	 *  Die Methode implementiert die (dynamische) kontrollierte Anfrageauswertung mit dem kombinierten Zensor bei bekannten potentiellen Geheimnissen
	 *  und einer als vollstaendig angenommenen Datenbankinstanz. Falls durch die korrekte Antwort auf die gestellte Anfrage die Sicherheitspolitik
	 *  nicht verletzt wird, wird die korrekte Antwort zurueck gegeben. Falls durch die korrekte Antwort hingegen die Sicherheitspolitik verletzt
	 *  wird (ein beliebiges potentielles Geheimnis kann erschlossen werden), wird entweder die Negation der korrekten Antwort zurueck gegeben
	 *  oder die Antwort wird verweigert. Dabei wird verweigert, falls auch durch die Negation der korrekten Antwort ein beliebiges potentielles
	 *  Geheimnis erschlossen werden kann. Ansonsten wird gelogen.
	 *  @param query Die an die Datenbank gestellte Anfrage
	 *  @param log   Das aktuelle Log des Benutzers. Parameter ist fuer den OpencombinedCensor erforderlich.
	 *  
	 */
	protected List<Formula> run(Client client, Formula query, List<IdAndFormula> log) throws DatabaseException, UnsupportedFormulaException, ProverException {
		List<Formula> result = new LinkedList<Formula>();
		
		// PotSec-Menge aus dem User-Objekt laden
		List<ConfidentialityPolicyItem> confidentialityPolicy = client.getUser().getConfidentialityPolicy().getCopyAsList();
		
		// Query klonen, da evalStarComplete die uebergebene Formel modifiziert (Originalformel wird fuer Verweigerungsfall benoetigt)
		Formula evalStarQuery = Formula.createFormula(client.getApplicationDB().evalStarComplete(query.clone()));
		
		// Falls ein PotSec durch log u evalStarQuery impliziert wird: Verweigern oder Luegen
		if ( CensorUtil.isConfidentialityPolicyViolated(evalStarQuery, log, confidentialityPolicy) ){
			// Negation bilden, um zu pruefen, ob Luege problematisch ist
			evalStarQuery.negate();
			logger.debug("Kombinierter Zensor: korrekte Antwort ist gefaehrlich!");
			if ( CensorUtil.isConfidentialityPolicyViolated(evalStarQuery, log, confidentialityPolicy) ) {
				// Falls auch durch Negation der korrekten Antwort PotSec impliziert wird: Verweigern
				// Mum-Formel erstellen und zur Antwort hinzufuegen
				result.add( new MumFormula(query) );
				logger.debug("Kombinierter Zensor: auch falsche Antwort ist gefaehrlich --> Verweigern!");
				return result;
			} else {
				// Falls durch Negation der korrekten Antwort KEIN PotSec impliziert wird: Luegen
				// Bereits negierte korrekte Antwort hinzufuegen
				result.add( evalStarQuery );
				logger.debug("Kombinierter Zensor: falsche Antwort ist nicht gefaehrlich --> Luegen!");
				return result;
			}
		} else {
			// KEIN PotSec wird durch log u evalStarQuery impliziert --> korrekte Antwort geben
			result.add( evalStarQuery );
			logger.debug("Kombinierter Zensor: korrekte Antwort ist nicht gefaehrlich!");
			return result;
		}
	}

}
