package server.censor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.SortedSet;

import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.ParserException;
import exception.ProverException;
import exception.UnsupportedFormulaException;
import server.core.Client;
import server.core.Server;
import server.parser.CompletenessSentenceFormula;
import server.parser.Formula;


/**
 * Die Klasse OpenCensor ist eine abstrakte Oberklasse fuer Zensoren, die offene Anfragen auswerten.
 * Sie uebernimmt allgemeine Aufgaben wie das Laden eines zur Anfrage passenden Dictionaries oder das
 * Generieren von CompletenessSentences.
 * 
 * @author bring
 *
 */
public abstract class OpenCensor extends QueryCensor {

	/**
	 * Konstruktur: Erzeugt ein neues Objekt dieses Zensor-Typs
	 */
	public OpenCensor(String censorName, Server server) {
		super(censorName, server);
	}
	
	
	/**
	 * Diese Methode prueft, ob alle weiteren Substitutionen der freien Variablen aus der uebergebenen
	 * Formel ab der Position k in der Diagonalisierungsreihenfolge in der Datenbankinstanz zu false
	 * ausgewertet werden und gibt dies als Ergebnis zurueck.
	 * Es wird also geprueft, ob Complete(phi(x1,...,xn), k) in der Datenbankinstanz gilt.
	 * 
	 * @param client Client mitsammt Datenbanken, User etc.
	 * @param formula offene Anfrage, bzgl. der die Ueberpruefung stattfinden soll
	 * @param k Nummer der Konstanten (>=0) im Dictionary
	 * @return true, wenn alle weiteren Substitutionen in der Datenbankinstanz _nicht_ gelten, sonst false
	 * @throws DatabaseException
	 * @throws UnsupportedFormulaException
	 * @throws DictionaryTooSmallException
	 */
	protected boolean isComplete(Client client, Map<String, List<String>> dictionary, Formula formula, int k) throws DatabaseException, UnsupportedFormulaException, DictionaryTooSmallException {
		// keine offene Anfrage?
		if ( ! formula.isOpenFormula() )
			throw new UnsupportedFormulaException( formula.toString() );
		
		// Dictionary laden, falls noch nicht geschehen
		if( dictionary == null )
			dictionary = this.fillDictionary(client, formula, formula.getFreeVariables());
		
		// CompletenessSentence generieren
		CompletenessSentenceFormula complete = this.generateCompletenessSentence( client, dictionary, formula, k );
		
		// pruefen, ob der generierte CompletenessSentence in der Datenbank gueltig ist
		return client.getApplicationDB().evalComplete( complete );
	}
	
	
	
	/**
	 * Diese Methode generiert eine Formel für den Completeness Sentence, der prüft,
	 * ob alle weiteren Substitutionen der freien Variablen in der offenen Anfrage 
	 * formula nach der Position k im Dictionary zu false ausgewertet werden.
	 * 
	 * @param client Client mitsammt Datenbanken, User etc.
	 * @param formula offene Anfrage, zu der der Completeness Sentence gebildet werden soll
	 * @param k Nummer der Konstanten (>=0) im Dictionary ab der der Completeness Sentence angewendet werden soll
	 * @return Formel des Completeness Sentence
	 * @throws DatabaseException
	 * @throws ParserException
	 * @throws ProverException
	 * @throws UnsupportedFormulaException
	 */
	protected CompletenessSentenceFormula generateCompletenessSentence(Client client, Map<String, List<String>> dictionary, Formula formula, int k) throws DatabaseException, UnsupportedFormulaException, DictionaryTooSmallException {
		// keine offene Anfrage?
		if ( ! formula.isOpenFormula() )
			throw new UnsupportedFormulaException( formula.toString() );
		
		// Dictionary laden, falls noch nicht geschehen
		if( dictionary == null )
			dictionary = this.fillDictionary(client, formula, formula.getFreeVariables());
		
		// freie Variable ermitteln
		Set<String> freeVars = formula.getFreeVariables();
		
		// Alle Substitutionen der freien Variablen ermitteln, die fuer den CompletenessSentence
		// fuer das uebergebene k betrachtet werden muessen
		Set<Map<String, String>> substitutions = new HashSet<Map<String, String>>();
		int[] combination_i;
		for( int i=1; i<=k; ++i ) {
			// i-te Kombination in Substitutions-Array aufnehmen
			Map<String, String> combination = new HashMap<String, String>();
			combination_i = getDiagonalisationElement( freeVars.size(), i );
			int j = 0;
			for( String freeVar : freeVars ) {
				if( dictionary.get(freeVar).size() < combination_i[j] + 1 )
					throw new DictionaryTooSmallException("Dictionary enthaelt zu wenig Konstanten.");
				
				combination.put( freeVar, dictionary.get(freeVar).get(combination_i[j]) );
				++j;
			}
			substitutions.add( combination );
		}

		return new CompletenessSentenceFormula( formula, substitutions );
	}
	
	/**
	 * Diese Methode liest die fuer die Formel formula benoetigten Dictionaries aus der Datenbank.
	 * Dabei werden feur jede freie Variable separate Dictionaries in der als Klassenattribut 
	 * vorhandenen Map dictionary angelegt.
	 * Fuer jede freie Variable werden alle Dictionaries der Attribute von allen Relationen vereinigt, 
	 * in denen die freie Variable vorkommt.
	 * 
	 * @param client Client mitsammt Datenbanken, User etc.
	 * @param formula  Formel bzgl. der das Dictionary aufgebaut werden soll
	 * @param freeVars freie Variablen für die das Dictionary aufgebaut werden sollen
	 * @throws DatabaseException
	 */
	protected Map<String, List<String>> fillDictionary(Client client, Formula formula, Set<String> freeVars) throws DatabaseException {
		Map<String, List<String>> dictionary = new HashMap<String, List<String>>();
		
		for( String freeVar : freeVars ) {
			// neuen Eintrag fuer freeVar in Dictionary erstellen
			dictionary.put(freeVar, new ArrayList<String>());
			
			// temporaeres HashSet um effizient zu ermitteln, ob eine Konstante bereits im Dictionary enthalten ist
			Set<String> entrys = new HashSet<String>();
			
			// Benoetigte Dictionaries ermitteln
			// dafuer benoetigt: Relationennamen Attribute, in denen die freie Variable vorkommt
			Map<String, SortedSet<Integer>> relAtt = formula.getRelationnamesAndAttributesContainingVar(freeVar);
			
			// Dictionaries laden (bei mehreren werden diese zu einem vereinigt)
			// FIXME evtl. Optimierung. Bei AND kann anstatt der Vereinigung auch der Schnitt berechnet werden.
			for( Map.Entry<String, SortedSet<Integer>> entry : relAtt.entrySet() ) {
				// Attributnamen der Relation ermitteln, da Dictionary-Namen darauf basieren
				String[] columnnames = client.getApplicationDB().getAttributeNames(entry.getKey());
				for( Integer attr_index : entry.getValue() ) {
					List<String> dict = client.getMaintenanceDB().getDictionary().getValues(entry.getKey(), columnnames[attr_index]);
					// bereits im Dictionary vorhandene Konstanten nicht nochmals einfuegen
					for(String s : dict) {
						if( ! entrys.contains(s) ) {
							dictionary.get(freeVar).add(s);
							entrys.add(s);
						}
					}
				}
			}
		}
		
		return dictionary;
	}
	
	
	/**
	 * Diese Methode berechnet den Index der letzten Konstante (bzw. Element 
	 * in der Diagonalisierungsreihenfolge bei >1 freien Variable), die die 
	 * Formel phi erfuellt. Die Confidentiality-Policy wird dabei _nicht_ 
	 * beruecksichtigt.
	 * 
	 * @param client Client mitsammt Datenbanken, User etc.
	 * @param phi  Formel bzgl. der die letzte erfuellende Konstante ermittelt werden soll
	 * @return     Index der gesuchten Konstante (bzw. Element in der Diagonalisierungsreihenfolge)
	 * @throws DictionaryTooSmallException
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 */
	protected int getLastConstantSatisfyingPhi(Client client, Map<String, List<String>> dictionary, Formula phi) throws DictionaryTooSmallException, UnsupportedFormulaException, DatabaseException {
//		// lineare Suche
//		int k = 0;		
//		
//		while( ! isComplete(phi, k) ) {
//			++k;
//		}
//		
//		return k;
		
		
		// binaere Suche
		// suche Index i=2^j fuer den isComplete(phi, i) erstmalig gilt
		int i = 1;
		while( ! isComplete(client, dictionary, phi, i) ) {
			i *= 2;
		}
		// k enthaelt den bisher kleinsten bisher gefundenen Wert fuer den isComplete(phi, i) wahr ist
		int k = i;
		
		// fuer i=2^(j-1) gilt isComplete(phi, i) nicht
		// fuer i=2^j gilt isComplete(phi, i)
		// -> binare Suche im Intervall von i/2(=2^(j-1)) bis i-1(=2^j - 1)
		int left = i/2;
		int right = i-1;
		while( left <= right ) {
			int middle = (left + right) / 2;
			if( isComplete(client, dictionary, phi, middle) ) {
				// kleineren Wert fuer k gefunden
				k = middle;
				// links nach noch kleineren Werten weitersuchen
				right = middle - 1;
			}
			else {
				// rechts weitersuchen
				left = middle + 1;
			}
		}
		
		return k;
	}
	
	/**
	 * Diese Methode erstellt eine Map, die als Schluessel die freien Variablen enthaelt und
	 * als Wert jeweils die erforderliche Substitution der freien Variable fuer das k-te 
	 * Element in der Diagonalisierungsreihenfolge.
	 * @param freeVars  Menge von freien Variablen, die substituiert werden sollen
	 * @param k         Index des Elements in der Diagonalisierungsreihenfolge bzgl. der
	 *                  die Variablen substituiert werden sollen
	 * @return Map, mit Variablen als Schluessel und erforderliche Substitution als Wert
	 * @throws DictionaryTooSmallException
	 */
	protected Map<String, String> getFreeVarMapping(Map<String, List<String>> dictionary, Set<String> freeVars, int k) throws DictionaryTooSmallException {
		Map<String, String> mapping = new HashMap<String, String>();
		
		int[] combination_k = getDiagonalisationElement(freeVars.size(), k);
		int j = 0;
		
		for(String freeVar : freeVars) {
			if( dictionary.get(freeVar).size() < combination_k[j] + 1)
				throw new DictionaryTooSmallException("Dictionary enthaelt zu wenig Konstanten.");
			mapping.put( freeVar, dictionary.get(freeVar).get(combination_k[j]) );
			++j;
		}
		
		return mapping;
	}
	
	
	/**
	 * Diese Methode bestimmt das k-te Element in der Aufzaehlungsreihenfolge.
	 * Bei Dimensionen >1 wird dabei ein Diagonalisierungsverfahren eingesetzt.
	 * @param dimension Dimension des betrachteten Raumes
	 * @param k			Index des gesuchten Tupels (Position in der Aufzahelungsreihenfolge)
	 * @return			Tupel an k-ter Stelle in der Aufzaehlungsreihenfolge
	 */
	protected static int[] getDiagonalisationElement(int dimension, int k) {
		// Offset von -1, da Dictinaries bei 0 beginnen
		--k;
		
		int[] tuple;
		
		if(dimension == 1) {
			// einfacher Fall, keine Diagonalisierung erforderlich
			tuple = new int[1];
			tuple[0] = k;
		}
		else {
			// Diagonalisierung mittels des Diagonalisierungsverfahrens von Cantor
			tuple = inverseCantorPairingFunction(dimension, k);
		}
		
		return tuple;
	}
	
	
	/**
	 * Diese Methode implementiert die Umkehrfunktion des Cantor'schen Diagonalisierungsverfahrens.
	 * Sie berechnet das Tupel an k-ter Stelle in der Diagonalisierungsreihenfolge fuer die im 
	 * ersten Parameter angegebene Dimension.
	 * @param dimension Dimension des betrachteten Raumes
	 * @param k         Index des gesuchten Tupels (Position in der Diagonalisierungsreihenfolge)
	 * @return			Tupel an k-ter Stelle der Diagonalisierungsreihenfolge fuer dimension Dimensionen
	 */
	protected static int[] inverseCantorPairingFunction(int dimension, int k) {
		int[] tuple = new int[dimension];
		
		for(int i=dimension; i>1; --i) {
			int[] pair = inverseCantorPairingFunction2Dim(k);
			tuple[i-1] = pair[1];
			k = pair[0];
		}
		tuple[0] = k;
		
		return tuple;
	}
	
	/**
	 * Diese Methode implementiert die Umkehrfunktion des Cantor'schen Diagonalisierungsverfahrens.
	 * Sie berechnet das Tupel an k-ter Stelle in der Diagonalisierungsreihenfolge fuer 2 Dimensionen.
	 * @param k Index des gesuchten Tupels (Position in der Diagonalisierungsreihenfolge)
	 * @return	Tupel an k-ter Stelle der Diagonalisierungsreihenfolge fuer 2 Dimensionen
	 */
	protected static int[] inverseCantorPairingFunction2Dim(int k) {
		int[] tuple = new int[2];
		
		int w = (int)(( Math.sqrt(8*k+1) - 1 ) / 2);
		int t = ( w*w + w ) / 2;

		tuple[1] = k - t;
		tuple[0] = w - tuple[1];
		
		return tuple;
	}
	
	
//	/**
//	 * Methode zum einfachen Auflisten der Diagonalisierungsreihenfolge.
//	 * 
//	 * @param args
//	 */
//	public static void main(String[] args) {
//		int dimension = 2;
//		int from_element = 150;
//		int to_element = 155;
//		
//		int[] combination;
//		for( int i=from_element; i<=to_element; ++i ) {
//			combination = OpenCensor.getDiagonalisationElement(dimension, i);
//			System.out.println("Element " + i + ":\t" + combination[0] + "\t" + combination[1]);
//		}
//	}
}
