package client.core;

import exception.CqeException;
import exception.UnexpectedNotificationException;
import exception.UnknownNotificationException;
import exception.UnknownUserException;

import java.awt.Dimension;
import java.awt.Toolkit;
import java.io.IOException;
import java.net.Socket;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.ExecutionException;

import javax.swing.JFrame;
import javax.swing.SwingWorker;
import javax.swing.UIManager;
import javax.swing.JOptionPane;


import user.ConfidentialityPolicyItem;
import user.IdAndFormula;
import user.Role;
import user.User;

import communication.Communication;
import communication.CqeType;
import communication.SocketCommunication;

import client.gui.ImageDatabase;
import client.gui.ListModel;
import client.gui.automatonmanagement.AutomatonManagementFrame;
import client.gui.login.LoginFrame;
import client.gui.mainwindow.InteractionResultPanel;
import client.gui.mainwindow.MainFrame;
import client.gui.usermanagement.UserManagementFrame;

import notification.*;
import notification.clientToServer.AddUserInformationNotification;
import notification.clientToServer.AddUserNotification;
import notification.clientToServer.ChosenStateNotification;
import notification.clientToServer.DeleteUserInformationNotification;
import notification.clientToServer.DeleteUserNotification;
import notification.clientToServer.EditUserInformationNotification;
import notification.clientToServer.EditUserNotification;
import notification.clientToServer.GetAllRolesNotification;
import notification.clientToServer.GetAllUsersNotification;
import notification.clientToServer.InteractionNotification;
import notification.clientToServer.LoginNotification;
import notification.clientToServer.PreCQENotification;
import notification.serverToClient.AllRolesNotification;
import notification.serverToClient.AllUsersNotification;
import notification.serverToClient.AvailableDatabasesNotification;
import notification.serverToClient.ChooseStateNotification;
import notification.serverToClient.ConfidentialityPolicyNotification;
import notification.serverToClient.ConstraintsNotification;
import notification.serverToClient.ExceptionNotification;
import notification.serverToClient.LogNotification;
import notification.serverToClient.LoginAppNotification;
import notification.serverToClient.OkayNotification;
import notification.serverToClient.PreCQETimeAndNodesNotification;
import notification.serverToClient.PriorAllNotification;
import notification.serverToClient.PriorUserNotification;
import notification.serverToClient.QueryResultNotification;
import notification.serverToClient.ResultNotification;
import notification.serverToClient.StateNotification;
import notification.serverToClient.UpdateResultNotification;

/**
 * GUI-Kontroller: Sammelt alle Aktionen, die durch den Benutzer ueber die
 * GUI ausgeloest wurde und sendet sie an die CQE-Anwendung. Empfaengt auch alle
 * Anweisungen der CQE-Anwendung und zeigt sie dem Benutzer an.
 * @author schalge
 */
public class Client {
    public final static ImageDatabase imgDB = new ImageDatabase();
    public static Client GUI;
    
    private LoginFrame loginFrame = null;
    private MainFrame mainFrame = null;
    private JFrame userManagementFrame = null;
    private JFrame automatonManagementFrame = null;
    public Dimension screenSize = null;
    private int interactionCounter = 1;
    private List<String> availableDatabases = null;
    private ListModel<IdAndFormula> logModel = null;
    private ListModel<ConfidentialityPolicyItem> confidentialityPolicyModel = null;
    private ListModel<IdAndFormula> constraintsModel = null;
    private ListModel<IdAndFormula> priorAllModel = null;
    private ListModel<IdAndFormula> priorUserModel = null;
    private Communication server = null;

    /**
     * Start des Client-Kontrollflusses.
     * @param args
     * @throws Exception
     */
	public static void main ( String args[] ) throws Exception {
		Socket client = new Socket( "localhost", 65000 );
		
		Client.GUI = new Client( client );
	}

    
    /**
     * Startet die GUI und zeigt das Login-Fenster.
     */
    public Client( Socket serverSocket ) throws IOException {
        super();
        
        // Netzwerkverbindung speichern
        server = new SocketCommunication( serverSocket );


        // try to set a nice theme
        try {
            UIManager.setLookAndFeel("com.sun.java.swing.plaf.nimbus.NimbusLookAndFeel");
        } catch ( Exception e) { }

        // set screen size
        this.screenSize = Toolkit.getDefaultToolkit().getScreenSize();

        // Datenbankkonfiguration von der Application empfangen.
        Notification notification = server.recv();
        if ( !(notification instanceof AvailableDatabasesNotification) ) {
        	if ( notification instanceof ExceptionNotification ) {
            	this.showError( (ExceptionNotification)notification );
            } else {
        		this.showError( "Server connection failed.", new UnexpectedNotificationException("Received notification: "+notification.toString()+" - wanted: AvailableDatabasesAppNotification"), ExceptionNotification.CRITICAL );
        	}
        	System.exit( -1 );
        }
        
        AvailableDatabasesNotification dbNotification = (AvailableDatabasesNotification)notification;
        this.availableDatabases = dbNotification.getVector();
        
        // GUI-Sprache ist Englisch
        Locale.setDefault(Locale.ENGLISH);
        
        // Login-Dialog anzeigen.
        this.showLoginWindow();
    }

    /**
     * Zeigt das Login-Fenster an. Wenn ein Hauptfenster offen ist, so wird
     * dieses geschlossen.
     */
    public void showLoginWindow( ) {
        this.hideAutomatonManagementWindow();
        this.hideUserManagementWindow();
        this.hideMainWindow();
        
        this.loginFrame = new LoginFrame( this.availableDatabases );
        this.loginFrame.setVisible( true );
    }
    
    public void hideLoginWindow() {
    	if ( this.loginFrame != null ) {
    		this.loginFrame.dispose();
    	}
    	
    	this.loginFrame = null;
    }

    /**
     * Zeigt das Hauptfenster an. Wenn ein Login-Fenster offen ist, so wird
     * dieses geschlossen.
     */
    public void showMainWindow(String username, EnumSet<CqeType.Right> rights, String censor) {
        this.hideLoginWindow();
        
        this.mainFrame = new MainFrame(username, rights, censor);
        this.mainFrame.setVisible( true );
    }
    
    public void hideMainWindow() {
    	if ( this.mainFrame != null ) {
    		this.mainFrame.dispose();
    	}
    	
    	this.mainFrame = null;
    }

    /**
     * Zeigt das Fenster fuer die Benutzerverwaltung an. Kann nur geoeffnet werden,
     * wenn das Hauptfenster schon angezeigt wird.
     */
    public void showUserManagementWindow() {
        // ggf. ein bereits offenes Fenster schliessen
        this.hideUserManagementWindow();

        if ( this.mainFrame != null ) {
            if ( this.userManagementFrame == null ) {
            	try {
            		List<User> users = this.getAllUsers();
            		List<Role> roles = this.getAllRoles();
            		this.userManagementFrame = new UserManagementFrame(users, roles);
                    this.userManagementFrame.setVisible( true );
            	} catch ( IOException exception ) {
            		exception.printStackTrace();
            	}
            }
        }
    }

    /**
     * Schliesst das Fenster fuer die Benutzerverwaltung.
     */
    public void hideUserManagementWindow() {
        if ( this.userManagementFrame != null ) {
            this.userManagementFrame.dispose();
        }
        this.userManagementFrame = null;
    }
    
    public void showAutomatonManagementWindow() {
        // ggf. ein bereits offenes Fenster schliessen
        this.hideAutomatonManagementWindow();

        if ( this.mainFrame != null ) {
            if ( this.automatonManagementFrame == null ) {
                this.automatonManagementFrame = new AutomatonManagementFrame();
            }
            this.automatonManagementFrame.setVisible( true );
        }
    }
    
    public void hideAutomatonManagementWindow() {
    	if ( this.automatonManagementFrame != null ) {
            this.automatonManagementFrame.dispose();
        }
        this.automatonManagementFrame = null;
    }
    
    /**
     * Bei einem Zustandswechsel koennen prinzipiell mehrere Zielzustaende moeglich sein.
     * Der Benutzer soll ein Fenster mit der Auswahl (Dropdown-Feld mit Beschreibungstext)
     * angezeigt bekommen und den Zustandswechsel bestaetigen.
     */
    public void showStateSelection() {
    	
    }
    
    /**
     * Zeigt eine Exception aus einer ExceptionNotification an.
     * @param notification Nachricht, in der die Exception enthalten ist.
     */
    public void showError( ExceptionNotification notification ) {
    	String msg = null;
    	Exception originalException = null;
    	Exception exception = notification.getException();
    	if ( exception instanceof CqeException ) {
    		msg = notification.getException().getMessage();
    		originalException = ((CqeException)notification.getException()).getOriginalException();
    	} else {
    		msg = "Exception caught";
    		originalException = exception;
    	}
    	this.showError( msg, originalException, notification.getExceptionType() );
    }
    
    /**
     * Zeigt eine Exception an. FIXME: sollte das Programm danach automatisch geschlossen werden?
     * @param exception Die Exception, die dem Benutzer angezeigt werden soll.
     */
    public void showError( String msg, Exception exception, int type ) {
    	String title = "Exception caught";
    	if ( type == ExceptionNotification.CRITICAL ) {
    		title = "Critical "+title;
    	}
    	String error = msg;
    	if ( exception != null ) {
    		error += "\nOriginal Exception:\n"+exception.toString();
    		error += "\nStack Trace:";
    		for ( StackTraceElement ste : exception.getStackTrace() ) {
    			error += "\n"+ste;
    		}
    	}
    	JOptionPane.showMessageDialog( null, error, title, JOptionPane.ERROR_MESSAGE );
    	
    	// Bei kritischem Fehler alle Fenster schliessen. Klickt der Benutzer auf "OK" wird das Programm beendet.
    	if ( type == ExceptionNotification.CRITICAL ) {
    		this.closeAllWindows();
    	}
    }
    
    public void closeAllWindows() {
    	this.hideAutomatonManagementWindow();
    	this.hideUserManagementWindow();
    	this.hideMainWindow();
    	this.hideLoginWindow();
    }

    /**
     * Schickt eine Login-Anfrage an die CQE-Anwendung.
     * @param dbms Datenbank-Management-System welches verwendet werden soll. 
     * @param login Login des Benutzers.
     * @param pw Passwort des benutzers.
     */
    public void login( String dbms, String login, String pw ) throws UnknownNotificationException, IOException {
    	// Logindaten senden
        server.send( new LoginNotification(dbms, login, pw) );
        
        // Antwort verarbeiten
        Notification n = server.recv();
        if ( n instanceof LoginAppNotification == false ) {
        	ExceptionNotification ean = (ExceptionNotification)n;
        	if( ean.getException() instanceof UnknownUserException ) {
        		// Login fehlgeschlagen (Unknwon User)
                this.loginFrame.toggleWrongLogin();
        		return;
        	} else {
        		ean.getException().printStackTrace();
        		throw new UnknownNotificationException( n );
        	}
        }
        
    	LoginAppNotification loginAnswer = (LoginAppNotification)n;
    	if ( !loginAnswer.getSuccessful() ) {
    		// Login fehlgeschlagen
            this.loginFrame.toggleWrongLogin();
    		return;
    	}

        // Log empfangen
        n = server.recv();
        if ( n instanceof LogNotification ) {
        	LogNotification notification = (LogNotification)n;
        	this.logModel = new ListModel<IdAndFormula>( notification.getVector() );
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // ConfidentialityPolicy empfangen
        n = server.recv();
        if ( n instanceof ConfidentialityPolicyNotification ) {
        	ConfidentialityPolicyNotification notification = (ConfidentialityPolicyNotification)n;
        	if ( notification.getVector() != null ) {
        		this.confidentialityPolicyModel = new ListModel<ConfidentialityPolicyItem>( notification.getVector() );
        	} else {
        		this.confidentialityPolicyModel = null;
        	}
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // Constraints empfangen
        n = server.recv();
        if ( n instanceof ConstraintsNotification ) {
        	ConstraintsNotification notification = (ConstraintsNotification)n;
        	this.constraintsModel = new ListModel<IdAndFormula>( notification.getVector() );
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // PriorAll empfangen
        n = server.recv();
        if ( n instanceof PriorAllNotification ) {
        	PriorAllNotification notification = (PriorAllNotification)n;
        	this.priorAllModel = new ListModel<IdAndFormula>( notification.getVector() );
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // PriorUser empfangen
        n = server.recv();
        if ( n instanceof PriorUserNotification ) {
        	PriorUserNotification notification = (PriorUserNotification)n;
        	this.priorUserModel = new ListModel<IdAndFormula>( notification.getVector() );
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // Aktueller Zustand empfangen
        String censor = null;
        n = server.recv();
        if ( n instanceof StateNotification ) {
        	StateNotification notification = (StateNotification)n;
        	censor = notification.getCensor();
        } else {
        	throw new UnknownNotificationException( n );
        }
        
        // Login erfolgreich! Hauptfenster anzeigen und Models setzen
        this.showMainWindow( login, loginAnswer.getRights(), censor );
        
        this.mainFrame.getMainPanel().getTabPanel().getLogPanel().setModel( this.logModel );
        if ( this.confidentialityPolicyModel != null ) {
        	this.mainFrame.getMainPanel().getTabPanel().getConfidentialityPolicyPanel().setModel( this.confidentialityPolicyModel );
        } else {
        	this.mainFrame.getMainPanel().getTabPanel().hideConfidentialityPolicyPanel();
        }
        this.mainFrame.getMainPanel().getTabPanel().getConstraintPanel().setModel( this.constraintsModel );
        this.mainFrame.getMainPanel().getTabPanel().getPriorAllPanel().setModel( this.priorAllModel );
        this.mainFrame.getMainPanel().getTabPanel().getPriorUserPanel().setModel( this.priorUserModel );
    }

    /**
     * Schickt eine Logout-Anweisung an die CQE-Anwendung. Muss immer funktionieren.
     */
    public void logout() {
    	// FIXME: logout notification!
        this.showLoginWindow();
    }

    /**
     * Schickt eine Interaction (Query der Update) an die CQE-Anwendung.
     * @param text Auszuwertende Interaction.
     */
    public void interaction ( final String text ) throws IOException {
    	// Antwort in einem neuen Tab anzeigen
    	final InteractionResultPanel result = new InteractionResultPanel();
    	this.mainFrame.getMainPanel().getTabPanel().addClosableTab( "Interaction #"+this.interactionCounter, result );
    	this.interactionCounter++;
    	
    	// Anfrage wird asynchron gestellt, da ansonsten die GUI blockiert.
    	SwingWorker<Notification, Void> worker = new SwingWorker<Notification, Void>() {
    		// Zwischenspeichern des Ergebnis
    		String censor = "";
    		
			@Override
			protected Notification doInBackground() throws Exception {
		        // Interaction an die Application senden
		        Notification notification = new InteractionNotification( text );
		        server.send( notification );
		        
		        // Entweder es kommt direkt die richtige Loesung oder eine Aufforderung einen Zustand auszuwaehlen
		        notification = server.recv();
		        
		        if ( notification instanceof ExceptionNotification ) {
		        	return notification;
		        }
		        
		        if ( notification instanceof ChooseStateNotification ) {
		        	// DUMMY: immer Zustand 1 waehlen
		        	server.send( new ChosenStateNotification(1) );
		        	
		        	// Jetzt die Antwort empfangen
		        	notification = server.recv();
		        }
		        
		        if ( notification instanceof ResultNotification ) {
		        	ResultNotification resultNotification = (ResultNotification)notification;
		        	
		        	// Als naechstes wird noch der neue Zustand uebermittelt.
		        	notification = server.recv();
		        	if ( !(notification instanceof StateNotification) ) {
		        		throw new IOException( "Falsche Nachricht empfangen. Erwarte StateAppNotification, empfangen: "+notification );
		        	}
		        	StateNotification stateNotification = (StateNotification)notification;
		        	
		        	this.censor = stateNotification.getCensor();
		        	
		        	return resultNotification;
		        }
		        
		        // Irgendeine andere Notification => nicht gut
		        throw new IOException( "Unbekannte Nachricht empfangen: "+notification );
			}
			
			@Override
		    public void done() {
				try {
		            Notification notification = get();
		            
		            if ( notification instanceof ExceptionNotification ) {
			        	showError( (ExceptionNotification)notification );
		            } else if ( notification instanceof ResultNotification ) {			        	
			        	// Zustand und Zensor aktualisieren
			        	mainFrame.getMainTopPanel().setCensor( this.censor );

		            	// Ergebnis anzeigen
			        	if ( notification instanceof QueryResultNotification ) {
				        	QueryResultNotification resultNotification = (QueryResultNotification)notification;
			        		result.showResult( resultNotification.getAnsweredInteraction(), resultNotification.getResult(), resultNotification.getProcessingTime() );
			        	} else {
			        		UpdateResultNotification resultNotification = (UpdateResultNotification)notification;
			        		result.showResult( resultNotification.getAnsweredInteraction(), resultNotification.getResult(), resultNotification.getProcessingTime() );
			        	}
			        	
			        	// Log aktualisieren
			        	for ( IdAndFormula item : ((ResultNotification)notification).getAddToLog() ) {
			        		logModel.add( item );
			        	}
		            }
		        } catch (InterruptedException ignore) {
		        } catch (ExecutionException e) {
					e.printStackTrace();
				}
			}
    		
    	};
    	
    	worker.execute();
    }
    
    /**
     * Fragt eine Liste mit allen Benutzers und den dazugehoerigen Daten wie Log, PotSec, etc. an.
     * @return
     * @throws IOException
     */
    public List<User> getAllUsers() throws IOException {
    	// Anfrage an den Server senden.
    	Notification notification = new GetAllUsersNotification();
        server.send(notification);
        
        // Als Antwort koennen entweder die Benutzer kommen oder eine PermissionDenied-Nachricht.
        notification = server.recv();
        
        if ( notification instanceof ExceptionNotification ) {
        	this.showError( (ExceptionNotification)notification );
        	return null;
        }
        
        if ( notification instanceof AllUsersNotification ) {
        	return ((AllUsersNotification)notification).getVector();
        }
        
        // Irgendeine andere Notification => nicht gut
        throw new IOException( "Received unknown message: "+notification );
    }
    
    /**
     * Fragt eine Liste mit allen Rollen und den dazugehoerigen Rechten an.
     * @return
     * @throws IOException
     */
    public List<Role> getAllRoles() throws IOException {
    	// Anfrage an den Server senden.
    	Notification notification = new GetAllRolesNotification();
        server.send(notification);
        
        // Als Antwort koennen entweder die Benutzer kommen oder eine PermissionDenied-Nachricht.
        notification = server.recv();
        
        if ( notification instanceof ExceptionNotification ) {
        	this.showError( (ExceptionNotification)notification );
        	return null;
        }
        
        if ( notification instanceof AllRolesNotification ) {
        	return ((AllRolesNotification)notification).getVector();
        }
        
        // Irgendeine andere Notification => nicht gut
        throw new IOException( "Received unknown message: "+notification );
    }
    
    /**
     * Sendet eine Aktion an den Server. Bearbeitet auch die Antwort, die nur ein Okay
     * oder Fehler sein kann.
     * @param notification Nachricht, die als Aktion an den Server gesendet wird.
     * @return
     * @throws IOException
     */
    private boolean sendBooleanAction( Notification notification ) throws IOException {
    	if ( this.sendAction(notification) == Boolean.FALSE )
    		return false;
    	else
    		return true;
    }
    
    /**
     * Sendet eine Aktion an den Server. Bearbeitet die Antwort, die nur ein Okay
     * oder ein Fehler sein kann. Gibt falls vorhanden Anhang des Okay zurueck,
     * ansonsten Boolean.TRUE, falls kein Anhang bei Okay existiert oder
     * Boolean.FALSE, falls ein Fehler aufgetreten ist.
     * @param notification
     * @return Anhang der Okay-Nachricht, Boolean.TRUE wenn Okay-Nachricht ohne Anhang oder Boolean.FALSE im Fehlerfall.
     * @throws IOException
     */
    private Object sendAction( Notification notification ) throws IOException {
    	// Anfrage an den Server senden.
    	server.send(notification);
    	
    	// Als Antwort kann entweder ein Ok, eine PermissionDenied-Nachricht oder ein Fehler kommen.
    	notification = server.recv();
        
        if ( notification instanceof OkayNotification ) {
        	OkayNotification okay = (OkayNotification)notification;
        	if ( okay.getAttachement() == null ) {
        		return Boolean.TRUE;
        	} else {
        		return okay.getAttachement();
        	}
        }
        
        if ( notification instanceof ExceptionNotification ) {
        	this.showError( (ExceptionNotification)notification );
        	return Boolean.FALSE;
        }
        
        // Irgendeine andere Notification => nicht gut
        throw new IOException( "Received unknown message: "+notification );
    }
    
    
    public User addUser( String username ) throws IOException {
    	Object attachement = this.sendAction( new AddUserNotification(username) );
    	if ( attachement == Boolean.FALSE ) {
    		return null;
    	} else {
    		return (User)attachement;
    	}
    }
    
    public boolean editUser( int userId, String username, String password, Role role, String censor ) throws IOException {
    	return this.sendBooleanAction( new EditUserNotification(userId, username, password, role, censor) );
    }
    
    public boolean deleteUser( int userID ) throws IOException {
    	return this.sendBooleanAction( new DeleteUserNotification(userID) );
    }
    
    public IdAndFormula addUserInformationEntry( int type, int userId, String formula ) throws IOException {
    	Object attachement = this.sendAction( new AddUserInformationNotification(type, userId, formula) );
    	if ( attachement == Boolean.FALSE ) {
    		return null;
    	} else {
    		return (IdAndFormula)attachement;
    	}
    }
    
    public ConfidentialityPolicyItem addUserInformationConfidentialityPolicyEntry(int userId, ConfidentialityPolicyItem conf_pol_item) throws IOException {
    	CqeType.PolicyType policyType = conf_pol_item.getType();
    	CqeType.PolicyPreservation preservation = conf_pol_item.getPreservation();
    	String formula = conf_pol_item.getFormula().toString();
    	
    	Object attachement = this.sendAction( new AddUserInformationNotification(userId, formula, policyType, preservation) );
    	
    	if ( attachement == Boolean.FALSE ) {
    		return null;
    	} else {
    		return (ConfidentialityPolicyItem)attachement;
    	}
    }
    
    /**
     * Sendet eine Aktion an den Server. Bearbeitet die Antwort, die nur die Anzahl der Luegen der
     * besten Loesung, die Anzahl der erzeugten Knoten und die benoetigte Zeit enthaelt und erzeugt
     * ein Info-Fenster in der GUI, das diese Informationen ausgibt
     * @param user
     * @throws IOException
     */
    public void sendPreCQEAction( User user ) throws IOException {
    	// Anfrage an den Server senden
    	server.send(new PreCQENotification(user, false));
    	
    	// Als Antwort sollte der Server die Anzahl der Luegen, die Anzahl 
    	// der erzeugten Knoten und die fuer die Berechnung benoetigte Zeit senden.
    	Notification answer = server.recv();
    	
    	
    	if ( answer instanceof PreCQETimeAndNodesNotification ) {
    		
    		PreCQETimeAndNodesNotification preCQEanswer = (PreCQETimeAndNodesNotification) answer;
    		int numberOfLies = preCQEanswer.getNumberOfLies();
    		int numberOfNodes = preCQEanswer.getNumberOfNodes();
    		String processingTime = preCQEanswer.getProcessingTime();
    		
    		JOptionPane.showMessageDialog( null, "Number of lies in best solution: " + numberOfLies + "\n" +
    				                             "Number of created Nodes: " + numberOfNodes + "\n" +
    				                             "Processing Time: " + processingTime, 
    				                             "Great Success", 
    				                             JOptionPane.INFORMATION_MESSAGE );
    	}
    	else if ( answer instanceof ExceptionNotification ) {
        	this.showError( (ExceptionNotification)answer );
        }
    	else {
    		// Irgendeine andere Notification => nicht gut
            throw new IOException( "Received unknown message: "+answer );
    	}
    }
    
    public boolean editUserInformationEntry( int userId, int type, int formulaId, String formula ) throws IOException {
    	return this.sendBooleanAction( new EditUserInformationNotification(userId, type, formulaId, formula) );
    }
    
    public boolean editUserInformationConfidentialityPolicyEntry( int userId, int policyId, ConfidentialityPolicyItem conf_pol_item ) throws IOException {
    	CqeType.PolicyType policyType = conf_pol_item.getType();
    	CqeType.PolicyPreservation preservation = conf_pol_item.getPreservation();
    	String formula = conf_pol_item.getFormula().toString();
    	
    	return this.sendBooleanAction( new EditUserInformationNotification(userId, policyId, formula, policyType, preservation) );
    }
    
    public boolean deleteUserInformationEntry( int type, int formulaId, int userId ) throws IOException {
    	return this.sendBooleanAction( new DeleteUserInformationNotification(type, formulaId, userId) );
    }
    
    public boolean cleanupPreCQE( User user ) throws IOException {
    	return this.sendBooleanAction( new PreCQENotification(user, true) );
    }
}
