package user;

import java.io.Serializable;
import java.util.List;
import java.util.Vector;


/**
 * Diese Klasse dient zur Verwaltung eines Benutzers in Form eines User-Objekts. Dazu werden zum einen Methoden
 * zur Verwaltung der Daten des Benutzers (Benutzername, Log, ConfidentialityPolicy, Rolle, ...) bereitgestellt
 * und zum anderen Methoden zur kontrollierten Anfrageauswertung im Kontext dieses Benutzers.
 */
public class User implements Serializable {
	private static final long serialVersionUID = 5747032554049374848L;

	/**
	 * Id des Benutzers.
	 */
	private int userId;
	
	/**
	 * Name des Benutzers.
	 */
	private String username;
	
	/**
	 * Passwort des Benutzers. Wird in der Regel nicht gefuellt und bleibt aus
	 * Sicherheitsgruenden null. Nur wenn das Passwort eines Benutzers im
	 * Client geaendert werden soll, wird dieses Feld verwendet.
	 */
	private String password;
	
	/**
	 * Die aktuelle Rolle dieses Benutzers (Admin, StandardUser, ...)
	 */
	private Role role;
	
	/**
	 * FIXME: nur uebergangsweise hier
	 * Aktueller Censor des Benuters.
	 */
	private String censor;

	/**
	 * Die ConfidentialityPolicy dieses Benutzers.
	 */
	private ConfidentialityPolicy confidentialityPolicy;
	
	/**
	 * Enthaelt die Antworten auf die vom Benutzer gestellten Anfragen.
	 * Ist Teilmenge von log.
	 */
	private List<IdAndFormula> originalLog;
	
	/**
	 * Das Vorwissen dieses Benutzers.
	 */
	private List<IdAndFormula> priorSingleUser;
	
	/**
	 * Schema-Constraints der Datenbank. Sollte niemals geaendert werden,
	 * solange User-Instanzen existieren.
	 */
	private final List<IdAndFormula> schemaConstraints;
	
	/**
	 * Vorwissen, das fuer alle Benutzer identisch ist. Sollte niemals
	 * geaendert werden, solange User-Instanzen existieren.
	 */
	private final List<IdAndFormula> priorAll;

	
	/**
	 * Konstruktor, der ein neues User-Objekt auf Basis der uebergebenen Werte erzeugt. Im Rahmen dieses Konstruktors werden
	 * auch die Objekte zur Verwaltung des Logs und der ConfidentialityPolicy des Users erzeugt. Der aktuelle Censor kann
	 * anhand des aktuellen Automaten-Zustands (Parameter historyState) gesetzt werden (zumindest falls 1 zu 1 Beziehung zwischen
	 * Zustaenden und Censoren existiert).
	 *
	 * @param username Der Benutzername des zu erstellenden Users.
	 *
	 * @param role Die Rolle (vom Typ UserRole) des zu erstellenden Users.
	 * @param policy Die ConfidentialityPolicy des zu erstellenden Users in Form eines Vectors von Formulas.
	 * @param log Das Log des zu erstellenden Users in Form eines Vectors von Formulas.
	 */
	public User( int userId, String username, Role role, List<ConfidentialityPolicyItem> policy, List<IdAndFormula> log, String censor, List<IdAndFormula> schemaConstraints, List<IdAndFormula> priorAll, List<IdAndFormula> priorSingleUser ) {
		this.userId = userId;
		this.username = username;
		this.password = null;		// Standardmaessig kein Passwort laden. Das Passwort kann nur ueber setPassword() gesetzt werden.
		this.role = role;
		this.confidentialityPolicy = new ConfidentialityPolicy( policy );
		this.originalLog = new Vector<IdAndFormula>( log );	// Es wird automatisch eine Kopie der Liste erstellt.
		this.censor = censor;
		this.priorAll = priorAll;
		this.priorSingleUser = priorSingleUser;
		this.schemaConstraints = schemaConstraints;
	}
	
	/**
	 * Gibt die eindeutige Identifikation des Benutzers zurueck.
	 * 
	 * @return ID des Benutzers.
	 */
	public int getUserId(){
		return this.userId;
	}
	
	/**
	 * Gibt den Benutzernamen zurueck.
	 * 
	 * @return Der Benutzername.
	 */
	public String getUsername() {
		return this.username;
	}
	
	/**
	 * Setzt einen neuen Benutzernamen.
	 * Die Aenderung wird nicht in die Datenbank uebertragen.
	 * 
	 * @param username Neuer Benutzername.
	 */
	public void setUsername( String username ) {
		this.username = username;
	}
	
	/**
	 * Gibt das Passwort des Benutzers zurueck.
	 * 
	 * @return Aktuelles Passwort des Benutzers.
	 */
	public String getPassword() {
		return this.password;
	}
	
	/**
	 * Setzt ein neues Passwort fuer den Benutzer.
	 * Die Aenderung wird nicht in die Datenbank uebertragen.
	 * 
	 * @param password Neues Passwort des Benutzers.
	 */
	public void setPassword( String password ) {
		this.password = password;
	}
	
	/**
	 * Gibt die Rolle des Benutzers zurueck,
	 * durch die der Benutzer Zugriffsrechte erlangt.
	 * 
	 * @return Aktuelle Rolle des Benutzers.
	 */
	public Role getRole() {
		return this.role;
	}
	
	/**
	 * Setzt eine neue Rolle fuer den Benutzer.
	 * Die Aenderung wird nicht in die Datenbank uebertragen.
	 * 
	 * @param role Neue Rolle des Benutzers.
	 */
	public void setRole( Role role ) {
		this.role = role;
	}

	/**
	 * Censor des Benutzers.
	 * 
	 * @deprecated Wird im neuen Automatenmodell nicht mehr verwendet!
	 * 
	 * @return Aktueller Zustand des Benutzers.
	 */
	public String getCensor() {
		return this.censor;
	}
	/**
	 * @deprecated
	 * @param censor
	 */
	public void setCensor( String censor ) {
		this.censor = censor;
	}
	
	/**
	 * Gibt das Log (Vorwissen + durch Anfragen bereits erlangtes Wissen)
	 * dieses Benutzers zurueck. Modifikation der zurueckgegebenen Liste
	 * veraendert NICHT das intern gespeicherte Log! Dazu muss das
	 * originalLog veraendert werden (siehe getOriginalLog()).
	 * Vorwissen = Datenbank-Constraints + priorAll + priorSingleUser
	 * 
	 * @return Log des Benutzers.
	 */
	public Vector<IdAndFormula> getLog() {
		Vector<IdAndFormula> log = new Vector<IdAndFormula>();
		
		log.addAll( this.schemaConstraints ); // Datenbank-Constraints werden in das Log eingefuegt.
		log.addAll( this.priorAll ); // Das Allgemeinwissen wird in das Log eingefuegt.
		log.addAll( this.priorSingleUser ); // Das Vorwissen des einzelnen Benutzers wird in das Log eingefuegt.
		log.addAll( this.getOriginalLog() ); // Wissen aus den Anfragen.
		
		return log;
	}
	
	/**
	 * Gibt das Wissen, was der Benutzer durch Anfragen erlangt
	 * hat, zurueck. Enthaelt NICHT das Vorwissen des Benutzers.
	 * Ist Teilmenge von getLog().
	 * Aenderung der zurueckgegebenen Liste veraendert die interne
	 * Repraesentation. Datenbankeintraege werden allerdings nicht
	 * veraendert.
	 * 
	 * @return Log des Benutzers ohne Vorwissen.
	 */
	public List<IdAndFormula> getOriginalLog() {
		return this.originalLog;
	}
	
	/**
	 * Gibt die Sicherheitspolitik des Benutzers zurueck.
	 * 
	 * @return Sicherheitspolitik des Benutzers.
	 */
	public ConfidentialityPolicy getConfidentialityPolicy() {
		return this.confidentialityPolicy;
	}

	/**
	 * Gibt das Vorwissen des Benutzers zurueck.
	 * 
	 * @return Vorwissen des Benutzers.
	 */
	public List<IdAndFormula> getPrior() {
		return this.priorSingleUser;
	}
	
	/**
	 * Gibt die Schema-Constraints zurueck. Ist fuer alle Benutzer gleich.
	 * 
	 * @return Schema-Constraints der zum Benutzer gehoerenden AppDB.
	 */
	public List<IdAndFormula> getSchemaConstraints() {
		return this.schemaConstraints;
	}
}
