package server.theoremprover;

import java.util.List;

import server.parser.node.Node;
import exception.ProverException;

/**
 * Interface fuer alle Theorembeweiser, die eine logische Implikation |= auswerten.
 */
public abstract class TheoremProver {

	/**
	 * Testet jeweils eine logische Implikation A |= B auf ihren Wahrheitsgehalt.
	 * Dazu wird ein externer Theorembeweiser aufgerufen und dessen Ausgabe bewertet.
	 * 
	 * @param premise Praemisse der Implikation (linke Seite des Zeichens)
	 * @param conclusion Konklusion der Implikation (rechte Seite des Zeichens)
	 * @param uniqueNameAssumption true, falls von einer Unique Name Assumption ausgegangen werden soll, sonst false.
	 * @return true, falls die Praemisse die Konklusion impliziert, ansonsten false.
	 * @throws ProverException Diese Exception wird durch einen Fehler innerhalb des Provers ausgeloest.
	 */
	public boolean prove( List<? extends Node> premise, Node conclusion, boolean uniqueNameAssumption ) throws ProverException {
		ProverResult proverResult = this.proveWithResult(premise, conclusion, uniqueNameAssumption);
		return proverResult.getResult();
	}
	
	public abstract ProverResult proveWithResult( List<? extends Node> premise, Node conclusion, boolean uniqueNameAssumption ) throws ProverException;
	
	/**
	 * Testet logische Implikationen A |= B_i auf ihren Wahrheitsgehalt.
	 * Dazu wird ein externer Theorembeweiser aufgerufen und dessen Ausgabe bewertet.
	 * 
	 * @param premise Praemisse der Implikationen (linke Seite des Zeichens)
	 * @param conclusions Konklusionen der Implikation (rechte Seite des Zeichens). Fuer jedes Element der Liste wird ein Theorembeweis durchgefuehrt.
	 * @param uniqueNameAssumption true, falls von einer Unique Name Assumption ausgegangen werden soll, sonst false.
	 * @return true, falls die Praemisse mindestens eine Konklusion impliziert, ansonsten false.
	 * @throws ProverException Diese Exception wird durch einen Fehler innerhalb des Provers ausgeloest.
	 */
	public boolean prove( List<? extends Node> premise, List<? extends Node> conclusions, boolean uniqueNameAssumption ) throws ProverException {
		List<ProverResult> proverResults = this.proveWithResult(premise, conclusions, uniqueNameAssumption);
		for( ProverResult proverResult : proverResults )
			if( proverResult.getResult() )
				return true;
		
		return false;
	}
	
	public abstract List<ProverResult> proveWithResult( List<? extends Node> premise, List<? extends Node> conclusions, boolean uniqueNameAssumption ) throws ProverException;
	
	/**
	 * Testet jeweils eine logische Implikation A |= B auf ihren Wahrheitsgehalt.
	 * Dazu wird ein externer Theorembeweiser aufgerufen und dessen Ausgabe bewertet.
	 * Die Eingaben muessen in der jeweiligen Syntax des Theorembeweisers sein!
	 * Verwendet keine Unique Name Assumption.
	 * 
	 * @param premise Praemisse der Implikation (linke Seite des Zeichens)
	 * @param conclusion Konklusion der Implikation (rechte Seite des Zeichens)
	 * @return true, falls die Praemisse die Konklusion impliziert, ansonsten false.
	 * @throws ProverException
	 * @see #prove(List, Node, boolean)
	 */
	public boolean prove( String premise, String conclusion ) throws ProverException {
		ProverResult proverResult = this.proveWithResult(premise, conclusion);
		return proverResult.getResult();
	}
	
	public abstract ProverResult proveWithResult( String premise, String conclusion ) throws ProverException;
}
