package server.parser.languagetypes;

import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;

import server.parser.Formula;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.QuantifiedNode;

/**
 * Diese Klasse wertet die vorkommenden Knotentypen aus. Daraus
 * ergeben sich direkt eine große Anzahl Sprachableitungen. So sind
 * beispielsweise R-sentences aus atomic und !hasVars aufgebaut.
 * 
 * Geparste Typen: "atomic", "hasVars", "hasForallQuantor", "hasExistsQuantor",
 * "hasNegation", "hasConjunction", "hasDisjunction".
 */
public class BaseLTParser extends LanguageTypeParser {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.parser");

	private static final int ATOMIC = 0;
	private static final int HAS_VARS = 1;
	private static final int HAS_FORALL = 2;
	private static final int HAS_EXISTS = 3;
	private static final int HAS_NEGATION = 4;
	private static final int HAS_CONJUNCTION = 5;
	private static final int HAS_DISJUNCTION = 6;
	private static final int EXISTENTIAL_R_SENTENCE = 7;
	
	@Override
	public Set<String> parseLanguageTypes( Node tree ) {
		HashSet<String> types = new HashSet<String>();
		
		// Fuer die Belegung des Arrays bitte Anmerkung der Methode checkSubtree() lesen.
		boolean[] nodeTypes = { true, false, false, false, false, false, false, true };
		this.checkSubtree( tree, nodeTypes );
		
		// Jetzt die Boolean-Werte in Strings umwandeln.
		
		if ( nodeTypes[ATOMIC] )					{ types.add( "atomic" );					}
		if ( nodeTypes[HAS_VARS] )					{ types.add( "hasVars" );					}
		if ( nodeTypes[HAS_FORALL] )				{ types.add( "hasForallQuantor" );			}
		if ( nodeTypes[HAS_EXISTS] )				{ types.add( "hasExistsQuantor" );			}
		if ( nodeTypes[HAS_NEGATION] )				{ types.add( "hasNegation" );				}
		if ( nodeTypes[HAS_CONJUNCTION] )			{ types.add( "hasConjunction" );			}
		if ( nodeTypes[HAS_DISJUNCTION] )			{ types.add( "hasDisjunction" );			}
		if ( nodeTypes[EXISTENTIAL_R_SENTENCE] )	{ types.add( "existential_r_sentence" );	}
		
		return types;
	}
	
	/**
	 * Die Idee dieser Methode ist, dass bei Vorkommen eines bestimmten
	 * Knotentyps im Array nodeTypes der Wert von false auf true
	 * gesetzt wird. Dies wird rekursiv für den gesamten Baum fortgesetzt.
	 * Am Ende enthaelt das Array dann die gewuenschte Information.
	 * 
	 * Arrays werden anders als primitive Datentypen ueber call-by-reference
	 * in Java uebergeben. Dadurch sind die Aenderungen aus einer Rekursionsebene
	 * fuer die darueberliegenden Ebenen sichtbar.
	 * 
	 * Das als zweiter Parameter uebergebene Array muss die Groesse 8 haben und sich
	 * semantisch an die Definition der Klassenvariablen (s.o.) halten. Damit
	 * ein sinnvolles Ergebnis herauskommt muss der Index 0 (also ATOMIC) sowie der Index 7 
	 * (EXISTENTIAL_R_SENTENCE) mit true und der Rest des Arrays mit false belegt sein.
	 * 
	 * @param subtree Aktuell zu untersuchender Teilbaum.
	 * @param nodeTypes Boolean-Array der Groesse 8. Index 0 und 7 MUSS mit true belegt sein, der rest mit false. 
	 */
	private void checkSubtree( Node subtree, boolean[] nodeTypes ) {
		if( subtree.isRootNode() ) {
			if( subtree.isOpenFormula() ) {
				nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
			}
			
			checkSubtree( ((Formula)subtree).getRootChild(), nodeTypes );
			return;
		}
		
		// Abbruchbedingung der Rekursion: wir sind an einem Atom angekommen.
		if ( subtree.isAtomPredicateNode() ) {
			for ( Node child : subtree.getChildren() ) {
				if ( child.isVariableNode() ) {
					nodeTypes[HAS_VARS] = true;
				}
			}
			
			return;
		}
		if ( subtree.isEqualityNode() ) {
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
		}
		
		// Bei Negation vermerken und auch absteigen.
		if ( subtree.isNegationNode() ) {
			nodeTypes[ATOMIC] = false;
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
			nodeTypes[HAS_NEGATION] = true;
			
			checkSubtree( ((NegationNode)subtree).getNegatedFormula(), nodeTypes );
			return;
		}
		
		// Quantor gefunden. Zuerst feststellen welcher und danach die quantifizierte Formel untersuchen.
		if ( subtree.isQuantifiedNode() ) {
			nodeTypes[ATOMIC] = false;
			
			if ( subtree.isExistentialQuantifiedNode() ) {
				nodeTypes[HAS_EXISTS] = true;
			} else {
				nodeTypes[HAS_FORALL] = true;
				nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
			}
			
			checkSubtree( ((QuantifiedNode)subtree).getQuantifiedFormula(), nodeTypes );
			return;
		}
		
		// Bei Konjunktion und Disjunktion hat der Baum mehrere Kinder, also fuer jede einzeln
		// die Typen feststellen.
		if ( subtree.isConjunctorNode() ) {
			nodeTypes[ATOMIC] = false;
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
			nodeTypes[HAS_CONJUNCTION] = true;
			
			for ( Node child : subtree.getChildren() ) {
				checkSubtree( child, nodeTypes );
			}
			
			return;
		}
		
		if ( subtree.isDisjunctorNode() ) {
			nodeTypes[ATOMIC] = false;
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
			nodeTypes[HAS_DISJUNCTION] = true;
			
			for ( Node child : subtree.getChildren() ) {
				checkSubtree( child, nodeTypes );
			}
			
			return;
		}
		
		if( subtree.isEquivalenceNode() ) {
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
		}
		
		if( subtree.isImplicationNode() ) {
			nodeTypes[EXISTENTIAL_R_SENTENCE] = false;
		}
		
		// Wenn wir an dieser Stelle sind, wurde ein Knotentyp gefunden, der uns nicht interessiert.
		logger.debug( "ignoring node of type " + subtree.getClass().getName() );
	}

}
