package server.parser;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import server.parser.node.ExistentialQuantifiedNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.UniversalQuantifiedNode;
import exception.UnsupportedFormulaException;

public class FormulaToSRNF {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.parser");

	/**
	 * Wandelt eine Formel in SRNF um. Die Umwandlung erfolgt nach dem in Kapitel 5.4 S. 83
	 * "Foundations of Databases" von Abiteboul beschriebenen Algorithmus.
	 * @param formula Diese Formel wird in SRNF umgewandelt. Vorher wird keine Kopie erstellt.
	 * @return Gleiche Formel in SRNF. Vorher wurde eine Kopie erzeugt, die Eingabe wird aber nicht veraendert.
	 * @throws IllegalFormulaOperationException 
	 */
	public static void srnf(Node formula) {
		// Ersetze die Variablen so, dass kein Variablenname sowohl eine freie als auch
		// eine gebundene Variable bezeichnet und keine unterschiedlichen Quantoren die
		// gleiche Variable binden.
		FormulaUtil.replaceAllQuantifiedVariables( formula );

		// Ersetze alle Allquantoren durch Existenzquantoren (FORALL X phi = NOT EXISTS X NOT phi).
		FormulaToSRNF.removeUniversalQuantifier(formula);

		// Ersetze alle Äquivalenzen durch Konjunktionen von Disjunktionen. 
		formula.transformEquivalence();

		// Ersetze alle Implikationen durch Disjunktionen.
		formula.transformImplication();

		// Negationen vor Klammern in die Klammer ziehen (deMorgan) und doppelte Negationen entfernen.
		try {
			formula.pushNegationsSRNF();
		} catch (UnsupportedFormulaException e) {
			// Die Exception tritt nur auf, wenn Aequivalenzen oder Implikationen in der Formel
			// enthalten sind, die bereits entfernt worden sind. Deshalb hier abbrechen.
			logger.fatal("Impossible error occured, this is a bug.");
			e.printStackTrace();
		}
		
		// Flatten der Formula: Jedes AND ,OR und EXISTS darf kein Kind mit dem gleichen Typ haben.
		formula.flatten();
	}
	
	/**
	 * Durchsucht rekursiv die Formel nach Vorkommen von Allquantoren und ersetzt diese
	 * durch Existenzquantoren.
	 * @param formula In dieser Formel werden alle Vorkommen von Allquantoren durch Existenzquantoren ersetzt. Dabei wird die Formel veraendert (ggf. Kopie anlegen)!
	 * @return Der Rueckgabewert gibt die uebergebene Formel ohne Allquantoren zurueck. Wird auch fuer den rekursiven Abstieg verwendet.
	 */
	private static Node removeUniversalQuantifier(Node formula) {
		if (formula.isUniversalQuantifiedNode()) {
			Node parent = formula.getParent();
			Node newChild = universal2existence( (UniversalQuantifiedNode)formula );
			// FIXME ist das hier ueberhaupt noch noetig, weil wir im else-Fall unten eh eine neue
			// Liste von Elternknoten aufbaun?
			parent.replaceChild(formula, newChild);
			return removeUniversalQuantifier(newChild);
		} else if ( formula.isAtomPredicateNode() ) {
			// Abbruchbedingung der Rekursion: Wir sind an einem Atom. Darunter kann kein FORALL kommen.
			return formula;
		} else {
			// Iteriere über alle Kinder und baue eine neue Liste mit Kindern auf.
			List<Node> children = new ArrayList<Node>( formula.getChildren() );
			List<Node> newChildren = new ArrayList<Node>();
			
			for ( Node child : children ) {
				newChildren.add( removeUniversalQuantifier(child) );
			}
			formula.replaceChildren( newChildren );
			
			return formula;
		}
	}
	
	/**
	 * Ersetze "FORALL X phi" durch "NOT EXISTS X NOT phi".
	 * @param formula Allquantor, der ersetzt werden soll. Die formula wird dabei veraendert (ggf. Kopie anlegen).
	 * @return Eine Formel ohne Allquantor, dafuer mit Existenzquantor und zwei NOT.
	 */
	private static Node universal2existence(UniversalQuantifiedNode formula) {
		Node phi = formula.getQuantifiedFormula();
		Node notphi = new NegationNode(phi);
		Node existsNode = new ExistentialQuantifiedNode(formula.getVariableNodes(), notphi);
		
		return new NegationNode(existsNode);
	}
}
