package server.parser;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import exception.UnsupportedFormulaException;

import server.parser.node.DisjunctorNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.QuantifiedNode;
import server.parser.node.UniversalQuantifiedNode;

public class FormulaToPLNF {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.parser");

	/**
	 * Wandelt eine Formel in prenex-literal normal form (PLNF) um. Die Quantoren stehen anschließend
	 * vorne und werden "prenex" genannt. Die quantorenfreie Formel heißt matrix. In der matrix stehen
	 * alle Literale direkt vor den Atomen. 
	 * @param formula
	 */
	public static void plnf( Formula formula ) {		
		// Ersetze die Variablen so, dass kein Variablenname sowohl eine freie als auch
		// eine gebundene Variable bezeichnet und keine unterschiedlichen Quantoren die
		// gleiche Variable binden.
		FormulaUtil.replaceAllQuantifiedVariables(formula);

		// Ersetze alle Äquivalenzen durch Konjunktionen von Disjunktionen. 
		formula.transformEquivalence();

		// Ersetze alle Implikationen durch Disjunktionen.
		formula.transformImplication();

		// Negationen vor Klammern in die Klammer ziehen (deMorgan) und doppelte Negationen entfernen.
		try {
			formula.pushNegations();
		} catch (UnsupportedFormulaException e) {
			// Die Exception tritt nur auf, wenn Aequivalenzen oder Implikationen in der Formel
			// enthalten sind, die bereits entfernt worden sind. Deshalb hier abbrechen.
			logger.fatal("Impossible error occured, this is a bug.");
			e.printStackTrace();
			return;
		}
		
		// Quantoren als Prenex an den Anfang der Formel verschieben.
		FormulaToPLNF.calculatePrenex(formula);
		
		// Flatten der Formula: Jedes AND ,OR und EXISTS darf kein Kind mit dem gleichen Typ haben.
		formula.flatten();
	}
	
	/**
	 * Verschiebt die Quantoren einer Formel an den Anfang. Beachtet dabei die Geltungsbereiche
	 * von Variablen NICHT! => Vorher muss Variablenumbennung durchgefuehrt werden.
	 * Die uebergebene Formel darf ausserdem keine Implikation (IMPL, =>) und Aequivalenz (EQUIV, <=>) enthalten.
	 * @param formula Formel mit Disjunktion, Konjunktion, Negation und Quantoren.
	 */
	private static void calculatePrenex( Formula formula ) {
		// Filtere alle Quantoren aus dem Syntaxbaum und gebe sie in richtiger Reihenfolge zurueck.
		QuantifiedNode prenex = calculatePrenex( formula.getRootChild() );
		
		// Keine Quantoren vorhanden? => fertig.
		if ( prenex == null ) {
			return;
		}
		
		Node rest = formula.getRootChild();
		
		// Umgewandelte Formel mit prenex unter den Wurzelknoten haengen.
		// Muss vor anhaengen der restlichen Formel gemacht werden, da ansonsten der Parent-Zeiger
		// wieder geloescht wird.
		formula.setRootChild( prenex );
		
		// Der formula-Baum muss nun an den prenex-Baum mit den Quantoren gehängt werden.
		QuantifiedNode child = prenex;
		
		// In der Quantorliste absteigen bis in den untersten Knoten.
		while ( child.getChildrenNumber() == 1 ) {
			child = (QuantifiedNode) child.getQuantifiedFormula();
		}
		
		// Prenex und restliche Formel zusammenbaun.
		child.setQuantifiedFormula( rest );
	}
	
	/**
	 * Bekommt eine Node formula und veraendert den Baum, der formula repraesentiert. 
	 * Alle Quantorenknoten werden direkt rausgelöscht und ein neuer Baum, der nur die Quantoren
	 * enthält, wird aufgebaut. Die Quantoren sind dabei in der Reihenfolge untereinander gehängt,
	 * in der sie in formula vorkommen.
	 * @param formula
	 * @return
	 */
	private static QuantifiedNode calculatePrenex( Node formula ){
		
		if ( formula.isRootNode() ) {
			return calculatePrenex(((Formula)formula).getRootChild());
		}
		
		if ( formula.isAtomPredicateNode() || formula.isEqualityNode() ){
			return null;
		}
		
		if ( formula.isNegationNode() ){
			NegationNode neg = (NegationNode) formula;
			Node next = neg.getNegatedFormula();
			
			return calculatePrenex(next);
			
			//TODO: Variablenumbenennung! 
		}
		/*
		 * Falls ein Knoten ein ConnectorNode ist, müssen wir für alle Kinder des Knotens
		 * das Prenex berechnen. Dafür nehmen wir die Kinder als Iterator. Ist ein Kind aber ein
		 * QuantorNode so wird der Baum verändert und somit auch die Kinder des ConjunctorNode, die
		 * als Iterator dienen. Das Funktioniert so nicht.
		 * Darum wird zunächst eine Kopie der Kinder angelegt, die verändert werden kann.
		 */
		if ( formula.isConnectorNode() ){
			// Tail zeigt immer auf das letzte Element der Liste mit Quantoren.
			QuantifiedNode tail = null;
			// Head zeigt auf das erste Element der Liste mit Quantoren.
			QuantifiedNode head = null;
			
			// Kopie der Kinder anlegen.
			List<Node> children = formula.getChildren(); //Iterator, der nicht verändert werden darf
			//formula.replaceChildren((List<Node>)CommonUtil.cloneCollection(children)); // Kindliste wird kopiert und kann jetzt verändert werden
			formula.replaceChildren( new ArrayList<Node>(children) );
			
			for (Node child: children) {
				QuantifiedNode childHead = calculatePrenex( child );
				QuantifiedNode childTail = childHead;
				if ( childHead != null ) {
					childHead.setParent(tail);
					// Zum letzten Element laufen.
					while ( childTail.getQuantifiedFormula() != null ) {
						childTail = (QuantifiedNode)childTail.getQuantifiedFormula();
					}
				}

				// Wenn es eine Liste mit Quantoren gibt, dann anhaengen. Ansonsten head auf Ergebnis setzen.
				if ( tail != null ) {
					tail.setQuantifiedFormula( childHead );
				} else {
					head = childHead;
				}
				
				// Neues Ende der Quantorliste setzen.
				if ( childTail != null ) {
					tail = childTail;
				}
			}
			return head;
		}
		
		/*Hier stehen Quantoren vorne. Der formula-Baum muss veraendert werden.*/
		if ( formula.isQuantifiedNode() ){
			QuantifiedNode quantor = (QuantifiedNode) formula;
			
			/*Formel hinter dem Quantor in next speichern*/
			Node next = quantor.getQuantifiedFormula();
			quantor.setQuantifiedFormula(null);
			
			/*QuantifiedNode quantor rausschneiden*/
			Node parent = quantor.getParent();
			parent.removeChild( quantor );
			parent.addChild( next );
			
			// Rekursiver Aufruf.
			QuantifiedNode childQuantors = calculatePrenex( next );
			if ( childQuantors != null ) {
				quantor.setQuantifiedFormula( childQuantors );
			}
			
			return quantor;
		}
		
		return null;
	}
	
	
	
	
	
	/**
	 * Ueberprueft, ob eine Formel ein Denial Constraint ist. Denial Constraints sind allquantifizierte
	 * Disjunktionen von negativen Literalen. Der zu ueberpruefende Syntaxbaum ist also eindeutig
	 * UniversalQuantifiedNode - DisjunctorNode - NegationNode - AtomPredicateNode.
	 * Die Formel muss in PLNF gegeben sein, damit diese Ueberprüfung einfach geht. 
	 * @param formula in PLNF
	 * @return boolean
	 */
	public static boolean isDenialConstraint( Formula root ) {
		plnf(root);
		Node formula = root.getRootChild();
		
		if ( ! formula.isUniversalQuantifiedNode() ) {
			return false;
		}
		  
		UniversalQuantifiedNode forall = (UniversalQuantifiedNode) formula;
		     
		Node next = forall.getQuantifiedFormula();

		if ( ! formula.isDisjunctorNode() ) {
			return false;
		}

		DisjunctorNode or = (DisjunctorNode) next;
		
		List<Node> children = or.getOperands();
		
		for ( Node child : children ){
			if ( !child.isNegationNode() ){
				return false;
			}
			NegationNode neg = (NegationNode) child;
			Node negChild = neg.getNegatedFormula();
			
			if( !negChild.isAtomPredicateNode() ){
				return false;
			}
			continue;	
		}
		//FIXME: ist die Formel geschlossen?
		
		return true;
	}
	
	
}
