package server.parser;

import java.util.Collection;
import java.util.Set;
import java.util.TreeSet;

import communication.CqeType;

import server.parser.Formatter.FormulaFormatter;
import server.parser.languagetypes.BaseLTParser;
import server.parser.languagetypes.LanguageTypeParser;
import server.parser.languagetypes.OpenClosedLTParser;
import server.parser.node.NegationNode;
import server.parser.node.Node;

import exception.IllegalFormulaOperationException;
import exception.ParserException;
import exception.UnsupportedFormulaException;

/**
 * Objekte des Typs Formula dienen im Programm als interen Datenstruktur zur Repraesentation von Formeln
 * und enthalten des Weiteren alle notwendigen Methoden zur Konvertierung (z.B. Umwandlung zwischen String-
 * und Formula-Objekt), Analyse (z.B. Erkennen, ob Formel zu eingeschraenkter Teilsprache gehoert) und
 * Modifikation (z.B. Negieren) von Formeln.
 *
 */
public class Formula extends Node {
	private static final long serialVersionUID = -2385444710280308194L;
	
	/**
	 * Typ der Formula. Entweder normale Formel oder Vollstaendigkeitssatz.
	 * Beim Vollstaendigkeitssatz wird zwischen Vollstaendigkeitssaetzen mit
	 * und ohne auszuschliessende Konstanten unterschieden.
	 */
	public enum FormulaType {
		STANDARD_FORMULA,
		CS_WITHOUT_RULEOUT_COMBINATIONS,
		CS_WITH_RULEOUT_COMBINATIONS,
	}
	protected FormulaType formulaType;
		
	/**
	 * Die Menge der verwendeten LanguageType-Parser. Fuer jeden Parser wird genau
	 * eine Instanz erzeugt (sind Klassenattribute und damit nicht von Instanzen
	 * der Klasse Formula abhaengig).
	 */
	private static final LanguageTypeParser languageTypeParsers[] = { new OpenClosedLTParser(), new BaseLTParser() };
		
	/**
	 * Interaktionstyp dieser Formel. NONE, falls diese Formel nicht fuer eine Interaktion verwendet wurde (z.B. Log).
	 */
	private CqeType.InteractionType interactionType;
	
	
	
	
	/**
	 * Dieser Konstruktor erzeugt eine leere Formel und darf nur von der Klasse MumFormula verwendet werden!
	 */
	protected Formula() {
		this.interactionType = CqeType.InteractionType.NONE;
		this.formulaType = FormulaType.STANDARD_FORMULA;
	}

	/**
	 * Dieser Konstruktor bekommt eine Formel in Form eines Strings uebergeben und konvertiert diese (durch Einparsen) in
	 * die intern verwendete Datenstruktur in Form eines Syntaxbaums. Durch die Anforderung, dass dem Konstruktor gleich
	 * eine Formel (als String) uebergeben werden muss, wird sichergestellt, dass nur Formel-Objekte entstehen koennen,
	 * die korrekt initialisiert sind (d.h. einen Syntaxbaum enthalten, der Element der Sprache ist).
	 * @param formula Die Formel (in Form eines Strings), die in die interne Datenstruktur eingelesen werden soll.
	 */
	public Formula(String formula) throws ParserException {
		this.formulaType = FormulaType.STANDARD_FORMULA;
		this.parseFormula(formula);
	}
	
	/**
	 * Erstellt ein neues Formula-Objekt mit dem durch root gegebenen Syntaxbaum.
	 * 
	 * 
	 * @param root Formel, um die ein Formula-Objekt (=Wurzelknoten) gewrappt wird
	 */
	public Formula( Node root ) {
		// Teilformel ohne Wurzelknoten.
		this.setFormulaType( FormulaType.STANDARD_FORMULA );
		this.setInteractionType( CqeType.InteractionType.NONE );
		this.setRootChild( root );
	}
	
	/**
	 * Copy-Konstruktor.
	 * Erstellt eine exakte (tiefe) Kopie des als Parameter übergebenen Formula-Objekts.
	 * 
	 * @param formula Formel, die kopiert werden soll.
	 */
	public Formula( Formula formula ) {
		// Vollstaendige Formel inkl. Wurzelknoten.
		this.setFormulaType( formula.getFormulaType() );
		this.setInteractionType( formula.getInteractionType() );
		this.setRootChild( formula.getRootChild().clone() );
	}

	/**
	 * Liefert null zurueck.
	 * @return null
	 */
	@Override
	public Node getParent() {
		return null;
	}
	
	/**
	 * Ungueltige Operation. Wirft eine Exeption.
	 * @throws IllegalFormulaOperationException
	 */
	@Override
	public void setParent(Node parent) throws IllegalFormulaOperationException {
		throw new IllegalFormulaOperationException("Formula objects can not have parents.", this);
	}
	
	/**
	 * Negiert das aktuelle Formula-Objekt. Dazu wird der Syntaxbaum direkt manipuliert.
	 */
	public void negate() {
		Node rootChild = this.getRootChild();
		
		if ( rootChild.isNegationNode() ) {
			// Formel ist bereits negiert => Negation aufloesen.
			this.setRootChild( ((NegationNode)rootChild).getNegatedFormula() );
		} else {
			// Formel noch nicht negiert. Negation hinzufuegen.
			this.setRootChild( new NegationNode(rootChild) );
		}
	}
	
	/**
	 * Negiert jedes Vorkommen eines uebergebenen Literals.
	 * 
	 * Veraendert den zugrundeliegenden Syntaxbaum!
	 * 
	 * @param literal Literal, dessen beinhaltetes Atom im Syntaxbaum nergiert werden soll.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn der uebergebene Knoten kein Literal ist.
	 */
	public void neg( Node literal ) throws UnsupportedFormulaException {
		super.negateLiteral( literal );
	}
	
	/**
	 * Negiert nacheinander jedes Vorkommen eines der uebergebenen Literale.
	 * 
	 * Veraendert den zugrundeliegenden Syntaxbaum!
	 * 
	 * @param literals Liste von Literal, deren beinhalteten Atome im Syntaxbaum nergiert werden soll.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn der uebergebene Knoten kein Literal ist.
	 */
	public void neg( Collection<? extends Node> literals ) throws UnsupportedFormulaException {
		for ( Node literal : literals ) {
			super.negateLiteral( literal );
		}
	}

	/**
	 * Gibt die Menge aller LanguageTypes zurueck, die auf diese Formula zutreffen. Dafuer wird
	 * iterativ jede LanguageType-Parser aufgerufen und das Ergebnis zur Menge hinzugefuegt.
	 * Damit haengen die moeglichen LanguageType-Strings von den einzelnen LanguageType-Parser
	 * ab.
	 * 
	 * @return Alle LanguageTypes, die auf diese Formula zutreffen.
	 */
	public Set<String> getLanguageTypes() {
		Set<String> ltypes = new TreeSet<String>();

		// Fuer Updates gibt es keine wirklich unterschiedlichen Sprachen, deshalb
		// sind sie hier fest kodiert.
		if ( this.getInteractionType() == CqeType.InteractionType.PROVIDER_UPDATE ||
			 this.getInteractionType() == CqeType.InteractionType.PROVIDER_UPDATE_TRANSACTION ) {
			ltypes.add("admin_update");
			return ltypes;
		}
		
		if ( this.getInteractionType() == CqeType.InteractionType.VIEW_UPDATE ||
			 this.getInteractionType() == CqeType.InteractionType.VIEW_UPDATE_TRANSACTION ) {
			ltypes.add("view_update");
			return ltypes;
		}
		
		// Der Dummy-Typ all wird von jeder Formula erfuellt und ist deshalb immer in der
		// Menge enthalten.
		ltypes.add("all");
		
		// Jetzt die eigentlichen LanguageTypes hinzufuegen.
		for ( LanguageTypeParser parser : Formula.languageTypeParsers ) {
			ltypes.addAll( parser.parseLanguageTypes( this ) );
		}
		
		return ltypes;
	}
	
	public CqeType.InteractionType getInteractionType() {
		return this.interactionType;
	}
	
	public void setInteractionType( CqeType.InteractionType type ) {
		this.interactionType = type;
	}
	
	public FormulaType getFormulaType() {
		return this.formulaType;
	}
	
	public void setFormulaType( FormulaType type ) {
		this.formulaType = type;
	}
	
	/**
	 * Einparsen der Formel in einen Syntaxbaum und Umwandlung dieses Typs eines Syntaxbaums in einen (evtl. syntaktisch anderen)
	 * Syntaxbaum, der zur internen Repraesentation einer Formel in der Formula-Klasse verwendet wird.
	 * @param formulaString Die Formel (in Form eines Strings), die eingelesen werden soll.
	 * @return Die eingelesene Formel in Form eines Syntaxbaums des Typs, der in der Formula-Klasse verwendet wird.
	 */
	private void parseFormula(String formulaString) throws ParserException {
		Parser parser = new Parser( formulaString );
		try {
			parser.parseInteraction( this );
		} catch (ParseException parseException) {
			throw new ParserException(parseException);
		}
	}
	
	/**
	 * Setzt das einzige Kind der RootNode und damit den Inhalt der durch diesen Baum
	 * repraesentierten Formel neu.
	 * @param child Node, die als neues Kind (und neuer Inhalt der Formel) gesetzt werden soll.
	 */
	public void setRootChild( Node child ) {
		Node oldRoot = this.getRootChild();
		if ( oldRoot == null ) {
			this.addChild( child );
		} else {
			this.replaceChild( oldRoot, child );
		}
	}
	
	/**
	 * Gibt das einzige Kind der Formula zurueck, das im Syntaxbaum die erste
	 * "echte" Node ist.
	 * @return Erstes "echtes" Kind im Syntaxbaum.
	 */
	public Node getRootChild() {
		if ( this.getChildrenNumber() > 0 ) {
			return this.getChild(0);
		} else {
			return null;
		}
	}
	
	@Override
	public boolean isRootNode() {
		return true;
	}

	@Override
	public String toString(FormulaFormatter formulaformatter) {
		return this.getRootChild().toString(formulaformatter);
	}
	
	@Override
	public Set<String> rr() {
		return this.getRootChild().rr();
	}
	
	@Override
	public void pushNegations() throws UnsupportedFormulaException {
		this.getRootChild().pushNegations();
	}
	
	@Override
	public void pushNegationsSRNF() throws UnsupportedFormulaException {
		this.getRootChild().pushNegationsSRNF();
	}
	
	/**
	 * Erstellt eine tiefe Kopie der aktuellen Formel. Alternativ kann auch der
	 * Copy-Konstruktor verwendet werden.
	 * @return Tiefe Kopie der aktuellen Formel.
	 */
	@Override
    public Formula clone() {
    	Formula copy = (Formula)super.clone();
    	copy.setInteractionType( this.getInteractionType() );
    	
    	return copy;
	}
	
	/**
	 * Diese Methode akzeptiert einen beliebigen Knoten des Syntaxbaums als Parameter und haengt
	 * als Wurzel darueber ein neues Formula-Objekt, es sei denn der Knoten selbst ist eine Formula.
	 * In diesem Fall wird lediglich die Formula zurueckgegeben.
	 * 
	 * Achtung: Der urspruengliche Syntaxbaum wird manipuliert und wenn z.B. nur eine Teilformel uebergeben
	 * wurde, wird die urspruengliche Gesamtformel zerstoert.
	 * @param formula Vollstaendiger Syntaxbaum (Formula-Objekt) oder nur ein Teilbaum (beliebige andere Node).
	 * @return Vollstaendiger Syntaxbaum, mit dem Inhalt des Parameters.
	 */
	public static Formula createFormula( Node formula ) {
		if ( formula.isRootNode() ) {
			return (Formula)formula;
		} else {
			return new Formula( formula );
		}
	}
}
