package server.database.sql;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

import exception.DatabaseException;

import server.database.schema.SchemaColumn;
import server.database.schema.TableSchema;
import server.util.Tuple;

/**
 * Resembles an SQL INSERT statement.
 * Please see the note for method chaining in the description
 * of SQLBaseExpression.
 *
 * @see SQLBaseExpression
 */
public class SQLInsert extends SQLBaseExpression<SQLInsert> {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");

	private List<Tuple<String, String>> columnValues;
	private List<String> generatedColumns;
	private SQLTable generatedValues;
	private List<Object> setBindVariables;
	
	/**
	 * Creates a new SQL INSERT expression.
	 * @param db Database connection. Must already be established and usable.
	 */
	public SQLInsert( SQLDatabase db ) {
		super( db );
		
		this.columnValues = new LinkedList<Tuple<String, String>>();
		this.generatedColumns = new LinkedList<String>();
		this.generatedValues = null;
		this.setBindVariables = new LinkedList<Object>();
	}	
	
	/**
	 * Returns a list of columns and their values that would be used upon insert.
	 * @return List of tuples containing a column and the assigned value.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 */
	public List<Tuple<String, String>> getColumnValues() {
		return this.columnValues;
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(SchemaColumn column, String value) {
		return this.set( column, value, false );
	}
	
	/**
	 * Sets the value of a specific column to a number.
	 * @param column Column that will be assigned with a value.
	 * @param number The numeric value the column should have.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(SchemaColumn column, int number) {
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @param noprime If true '' around the value will be omitted, otherwise they will be automatically added.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(SchemaColumn column, String value, boolean noprime) {
		return this.set( column.name, value, noprime );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(String column, String value) {
		return this.set( column, value, false );
	}
	
	/**
	 * Sets the value of a specific column to a number.
	 * @param column Column that will be assigned with a value.
	 * @param number The numeric value the column should have.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #set(String, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(String column, int number) {
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Sets the value of a specific column.
	 * @param column Column that will be assigned with a value.
	 * @param value The value the column should have.
	 * @param noprime If true '' around the value will be omitted, otherwise they will be automatically added.
	 * @return Current SQLInsert object. Allows method chaining.
	 * @see #set(SchemaColumn, int)
	 * @see #set(SchemaColumn, String)
	 * @see #set(String, int)
	 * @see #set(String, String)
	 * @see #set(SchemaColumn, String, boolean)
	 * @see #getColumnValues()
	 */
	public SQLInsert set(String column, String value, boolean noprime) {
		// Every value can be replaced by bind variables. Prepare this here.
		this.setBindVariables.add( value );
		
		// Add '' if necessary.
		if ( !noprime ) {
			value = "'"+value+"'";
		}

		this.columnValues.add( new Tuple<String, String>(column, value) );
		
		return this;
	}
	
	/**
	 * Returns a list of columns whose values are generated by the database.
	 * The generated values can be retrieved with the method {@link SQLDatabase#getGeneratedValues()}.
	 * @return The list of columns whose values are generated by the database.
	 * @see #setGenerated(String)
	 * @see SQLDatabase#getGeneratedValues()
	 */
	public List<String> getGeneratedColumns() {
		return this.generatedColumns;
	}
	
	/**
	 * Marks a column as one whose values are generated by the database.
	 * The generated values can be retrieved with the method {@link SQLDatabase#getGeneratedValues()}.
	 * @param column A column whose values are generated by the database.
	 * @see #getGeneratedColumns()
	 * @see #setGenerated(SchemaColumn)
	 * @see SQLDatabase#getGeneratedValues()
	 */
	public void setGenerated( String column ) {
		this.generatedColumns.add( column );
	}
	
	/**
	 * Marks a column as one whose values are generated by the database.
	 * The generated values can be retrieved with the method {@link SQLDatabase#getGeneratedValues()}.
	 * @param column A column whose values are generated by the database.
	 * @see #getGeneratedColumns()
	 * @see #setGenerated(String)
	 * @see SQLDatabase#getGeneratedValues()
	 */
	public void setGenerated( SchemaColumn column ) {
		this.generatedColumns.add( column.name );
	}
	
	/**
	 * Returns the values of the generated columns, previously specified with {@link #set(SchemaColumn, String)}.
	 * @return A result set that contains only the values of the generated columns.
	 */
	public SQLTable getGeneratedValues() {
		return this.generatedValues;
	}
	
	
	
	@Override
	protected String toSQL( boolean bindVars ) throws DatabaseException {
		String columns = "";
		String values = "";		
		String sep = "";
		
		if ( this.getTable() == null || this.columnValues.isEmpty() ) {
			throw new DatabaseException("No table for INSERT specified or empty INSERT statement.", null);
		}
		
		for ( Tuple<String, String> columnValue : this.columnValues ) {
			columns += sep + columnValue.getFirstElement();
			if ( bindVars ) {
				values += sep + "?";
			} else {
				values += sep + columnValue.getSecondElement();
			}
			
			sep = ",";
		}
				
		return "INSERT INTO " + this.getTable() + "(" + columns + ") VALUES (" + values + ")";
	}
	
	
	/**
	 * Executes the INSERT statement and returns the number of rows inserted.
	 * @return Number of rows inserted.
	 * @throws DatabaseException
	 * @see #insert(String)
	 * @see #insert(TableSchema)
	 */
	public int insert() throws DatabaseException {
		int result = -1;
		
		// Create SQL statement.
		String sql = this.toSQL( true );
		
		logger.debug("SQL INSERT: " + sql);
		logger.debug("Bind Values: "+this.setBindVariables);
		
		// Replace variables.
		PreparedStatement pstmt = null;
		int generatedColumnSize = this.getGeneratedColumns().size();
		if ( generatedColumnSize > 0 ) {
			String[] generatedColumnList = this.getGeneratedColumns().toArray( new String[generatedColumnSize] );
			pstmt = this.getStatement( sql, this.setBindVariables, generatedColumnList );
		} else {
			pstmt = this.getStatement( sql, this.setBindVariables );
		}
		 
		// Execute SQL statement.
		try {
			result = pstmt.executeUpdate();
			
			// Save the generated values for retrieval via getGeneratedValues().
			if ( generatedColumnSize > 0 ) {
				this.generatedValues = new SQLTable( pstmt.getGeneratedKeys() );
			} else {
				this.generatedValues = new SQLTable( null );
			}
			
			pstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error in the current SQL INSERT statement.", exception );
		}
		
		return result;
	}
	
	/**
	 * Executes the INSERT statement and returns the number of rows inserted.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)} is overridden with the specified value.
	 * Try to use {@link #insert(TableSchema)} if possible.
	 * @param table Table in which the data will be inserted.
	 * @return Number of rows inserted.
	 * @throws DatabaseException
	 * @see #insert()
	 * @see #insert(TableSchema)
	 */
	public int insert( String table ) throws DatabaseException {
		this.table( table );
		return this.insert();
	}
	
	/**
	 * Executes the INSERT statement and returns the number of rows inserted.
	 * Any name set by {@link #table(String)} or {@link #table(TableSchema)} is overridden with the specified value.
	 * @param table Table in which the data will be inserted.
	 * @return Number of rows inserted.
	 * @throws DatabaseException
	 * @see #insert()
	 * @see #insert(String)
	 */
	public int insert( TableSchema table ) throws DatabaseException {
		this.table( table.name );
		return this.insert();
	}
}
