package server.database.schema.maintenance;

import server.data.Role;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.DatabaseSchema;
import server.database.schema.Schema;
import server.database.schema.SchemaColumn;
import server.database.schema.TableSchema;
import server.database.schema.app.DictTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLInsert;
import exception.DatabaseException;

public class MaintenanceDatabaseSchema extends DatabaseSchema {

	// Maintenance Tables
	public final TableSchema priorAll;
	public final TableSchema constraints;
	public final TableSchema right;
	public final TableSchema role;
	public final TableSchema role_right;
	public final TableSchema cqeUser;
	public final TableSchema confidentialityPolicy;
	public final TableSchema priorUser;
	public final TableSchema log;
	public final TableSchema dictionaries;
	public final TableSchema signaturesFlexiblePotSecAV;
	public final TableSchema signaturesFlexiblePotSec;
	public final TableSchema signaturesFlexibleQuery;
	public final TableSchema signaturesFlexibleQueryAV;
	public final TableSchema signaturesFlexibleSignatures;
	public final TableSchema signaturesFlexibleSignaturesP;
	public final TableSchema signaturesFlexibleSignaturesPAV;
	public final TableSchema signaturesUnflexiblePotSecKrankheit;
	public final TableSchema signaturesUnflexibleQueryKrankheit;
	public final TableSchema signaturesUnflexibleSignaturesKrankheit;
	// AppDB Tables (intentionally left blank)
	
	public MaintenanceDatabaseSchema() throws DatabaseException {
		this("Maintenance Database Schema");
	}
	
	public MaintenanceDatabaseSchema(String name) throws DatabaseException {
		super(name);
				
		// Knowledge of all users.
		this.priorAll = new PriorAllTableSchema();
		this.constraints = new ConstraintsTableSchema();
		
		// Tables for access rights.
		this.right = new RightTableSchema();
		this.role = new RoleTableSchema();
		this.role_right = new RoleRightTableSchema();
		
		// User specific data.
		this.cqeUser = new UserTableSchema();
		this.confidentialityPolicy = new ConfidentialityPolicyTableSchema();
		this.priorUser = new PriorUserTableSchema();
		this.log = new LogTableSchema();
		
		// Dictionaries for open censors that need more domain values.
		this.dictionaries = new DictionariesTableSchema();
		
		// Tables for the signature censor including the ones of the signature censor experiment
		this.signaturesFlexiblePotSecAV = new SignaturesFlexiblePotSecAVTableSchema();
		this.signaturesFlexiblePotSec = new SignaturesFlexiblePotSecTableSchema();
		this.signaturesFlexibleQuery = new SignaturesFlexibleQuerySchema();
		this.signaturesFlexibleQueryAV = new SignaturesFlexibleQueryAVSchema();
		this.signaturesFlexibleSignatures = new SignaturesFlexibleSignaturesSchema();
		this.signaturesFlexibleSignaturesP = new SignaturesFlexibleSignaturesPSchema();
		this.signaturesFlexibleSignaturesPAV = new SignaturesFlexibleSignaturesPAVSchema();
		this.signaturesUnflexiblePotSecKrankheit = new SignaturesUnflexiblePotSecKrankheitSchema();
		this.signaturesUnflexibleQueryKrankheit = new SignaturesUnflexibleQueryKrankheitSchema();
		this.signaturesUnflexibleSignaturesKrankheit = new SignaturesUnflexibleSignaturesKrankheitSchema();
		
		// Add all tables to one array so that it can be iterated in the correct order easily.
		this.maintenanceTables.add( this.priorAll );
		this.maintenanceTables.add( this.constraints );
		this.maintenanceTables.add( this.right );
		this.maintenanceTables.add( this.role );
		this.maintenanceTables.add( this.role_right );
		this.maintenanceTables.add( this.cqeUser );
		this.maintenanceTables.add( this.confidentialityPolicy );
		this.maintenanceTables.add( this.priorUser );
		this.maintenanceTables.add( this.log );
		this.maintenanceTables.add( this.dictionaries );
		this.maintenanceTables.add( this.signaturesFlexiblePotSecAV );
		this.maintenanceTables.add( this.signaturesFlexiblePotSec );
		this.maintenanceTables.add( this.signaturesFlexibleQuery );
		this.maintenanceTables.add( this.signaturesFlexibleQueryAV );
		this.maintenanceTables.add( this.signaturesFlexibleSignatures );
		this.maintenanceTables.add( this.signaturesFlexibleSignaturesP );
		this.maintenanceTables.add( this.signaturesFlexibleSignaturesPAV );
		this.maintenanceTables.add( this.signaturesUnflexiblePotSecKrankheit );
		this.maintenanceTables.add( this.signaturesUnflexibleQueryKrankheit );
		this.maintenanceTables.add( this.signaturesUnflexibleSignaturesKrankheit );
	}
	

	
	
	public void createDefaultAccessRights(NewMaintenanceDatabase mdb) throws DatabaseException {
		String[] rights = { "SECURE_INTERACTION", "UNSECURE_INTERACTION",
				"MANAGE_USERS", "MANAGE_CONSTRAINTS", "MANAGE_AUTOMATON", "VIEW_CONFIDENTIALITY_POLICY"
		};
		String[] roles = { "Administrator", "User (known)", "User (unknown)" };
		SQLDatabase db = mdb.getDb();
		SQLInsert sql;
		
		// Create rights.
		for ( String right : rights ) {
			sql = db.newInsert();
			sql.set( RightTableSchema.NAME, right );
			sql.insert(Schema.maintenanceDatabase.right);
		}
		
		// Create roles.
		for ( String role : roles ) {
			sql = db.newInsert();
			sql.set( RoleTableSchema.NAME, role );
			sql.insert(Schema.maintenanceDatabase.role);
		}
		
		// Give the roles some meaning.
		// 1. Admin has every access right.
		for ( int i = 0; i < rights.length; i++ ) {
			sql = db.newInsert();
			sql.set(RoleRightTableSchema.ROLE_ID, 0).set(RoleRightTableSchema.RIGHT_ID, i);
			sql.insert(Schema.maintenanceDatabase.role_right);
		}
		// User (known) can view the confidentiality policy and can use secure interactions.
		sql = db.newInsert();
		sql.set(RoleRightTableSchema.ROLE_ID, 1).set(RoleRightTableSchema.RIGHT_ID, 0);
		sql.insert(Schema.maintenanceDatabase.role_right);
		
		sql = db.newInsert();
		sql.set(RoleRightTableSchema.ROLE_ID, 1).set(RoleRightTableSchema.RIGHT_ID, 5);
		sql.insert(Schema.maintenanceDatabase.role_right);
		// User (unknown) can just use secure interactions.
		sql = db.newInsert();
		sql.set(RoleRightTableSchema.ROLE_ID, 2).set(RoleRightTableSchema.RIGHT_ID, 0);
		sql.insert(Schema.maintenanceDatabase.role_right);
	}
	
	public void createDefaultUsers(NewMaintenanceDatabase mdb) throws DatabaseException {
		String[] censors = { "Lying", "Refusal", "Combined", "OpenLying", "OpenRefusal", "OpenCombined",
			"FlexibleStaticSignature", "UnflexibleStaticSignature", "LyingUpdate", "RefusalUpdate", "AccessControl", "DirectAccess",
			"RefusalSecreciesUnknown", "RefusalSecreciesKnown", "LyingSecreciesUnknown", "CombinedSecreciesKnown", 
			"RefusalUnknown", "LyingUnknown", "CombinedUnknown", "FPOptimized",
			"PreCQE", "PreCQEUpdate"
		};
		
		// Create superuser.
		this.createAdminUser(mdb);

		//load the standard role for all users
		Role role = Role.load(mdb.getDb(), 1);

		// Create one-censor users.
		for ( String censor : censors ) {
			User tmpUser = mdb.getUserManagement().add(censor.toLowerCase());
			tmpUser.setPassword(censor.toLowerCase());
			tmpUser.setCensor(censor);
			tmpUser.setRole(role);
		}
	}
	
	/**
	 * Creates a root user (admin)
	 * @param mdb the maintenaceDatabase to work on
	 * @return the id of the root user
	 * @throws DatabaseException
	 */
	public int createAdminUser(NewMaintenanceDatabase mdb) throws DatabaseException {
		User root = mdb.getUserManagement().add("root");
		root.setPassword("root");
		return root.getId();
	}
	
	public int createTestUser(NewMaintenanceDatabase mdb, int roleId, String censor) throws DatabaseException {
		User test = mdb.getUserManagement().add("test");
		test.setPassword("test");
		test.setCensor(censor);
		test.setRole(Role.load(mdb.getDb(), roleId));
		return test.getId();
	}

	
	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase mdb) throws DatabaseException {
		this.createDefaultAccessRights(mdb);
		this.createDefaultUsers(mdb);
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
	}
	
	/**
	 * Links a dictionary to a column in a specific relation. One dictionary
	 * can be linked tu multiple columns (just call the method more than once).
	 * 
	 * @param mdb
	 * @param dict
	 * @param column
	 * @throws DatabaseException
	 */
	public void linkDictionary(NewMaintenanceDatabase mdb, DictTableSchema dict, SchemaColumn column) throws DatabaseException {
		SQLInsert sql = mdb.getDb().newInsert();
		
		sql.set( DictionariesTableSchema.DICT_TABLE, dict.name.toUpperCase() );
		sql.set( DictionariesTableSchema.RELATION, column.getParent().name.toUpperCase() );
		sql.set( DictionariesTableSchema.ATTRIBUTE, column.name.toUpperCase() );
		sql.insert( this.dictionaries );
	}
}
