package server.database.schema.app;

import server.data.Role;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.schema.maintenance.SignaturesExperimentEvaluationSchema;
import server.database.schema.maintenance.SignaturesExperimentEvaluationCumulatedSchema;
import server.database.sql.SQLDatabase;
import exception.DatabaseException;

/**
 * Datenbank-Schema fuer den SignatureCensor und der krankheit-Tabelle.
 * 
 * Achtung: Fuer eine korrekte Funktionsweise muss zusaetzlich zur Generierung
 * dieses Schemas auch die sql-Datei 'staticSignatureCensorExperimentCreate.sql', 
 * die u.a. die Trigger fuer den SignatureCensor enthaelt, in der Datenbank
 * ausgefuehrt werden.
 */
public class SignatureExperimentSchema extends MaintenanceDatabaseSchema {
	
	public final SignaturesExperimentEvaluationSchema experimentEval;
	public final SignaturesExperimentEvaluationCumulatedSchema experimentEvalCumulated;
	public final KrankheitTableSchema krankheit;
	
	public SignatureExperimentSchema() throws DatabaseException {
		this("Signature Experiment Schema");
	}
	
	public SignatureExperimentSchema(String name) throws DatabaseException {
		super(name);
		
		// tables for the MaintainDB
		this.experimentEval = new SignaturesExperimentEvaluationSchema();
		this.experimentEvalCumulated = new SignaturesExperimentEvaluationCumulatedSchema();
		this.maintenanceTables.add( this.experimentEval );
		this.maintenanceTables.add( this.experimentEvalCumulated );
		
		// tables for the AppDB
		this.krankheit = new KrankheitTableSchema("SIG_KRANKHEIT");
		this.appTables.add( this.krankheit );
	}

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		// Create default access rights
		super.createDefaultAccessRights(db);
		
		String[] censors = { "UnflexibleStaticSignature", "FlexibleStaticSignature"};
		
		Role stdRole = Role.load(db.getDb(), 1);
		for( String censor : censors ) {
			User tmpUser = db.getUserManagement().add(censor.toLowerCase());
			tmpUser.setPassword(censor.toLowerCase());
			tmpUser.setCensor(censor);
			tmpUser.setRole(stdRole);
		}
	}
	
	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillKrankheit(db);
	}
	
	private void fillKrankheit(SQLDatabase db) throws DatabaseException {
		// no content needed
	}
}
