package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.DiagnosisDictionary;
import server.database.schema.app.dict.EffectsDictionary;
import server.database.schema.app.dict.NameDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class PreCQESecondExample extends MaintenanceDatabaseSchema {

	public final IllTableSchema ill;
	public final EffectsTableSchema effects;
	
	public final NameDictionary nameDict;
	public final DiagnosisDictionary diagnosisDict;
	public final EffectsDictionary effectsDict;
	
	public PreCQESecondExample() throws DatabaseException {
		super("PreCQE Second Example");
		
		this.ill = new IllTableSchema();
		this.effects = new EffectsTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.ill );
		this.appTables.add( this.effects );
		
		this.nameDict = new NameDictionary();
		this.diagnosisDict = new DiagnosisDictionary();
		this.effectsDict = new EffectsDictionary();
		
		this.maintenanceTables.add( this.nameDict );
		this.maintenanceTables.add( this.diagnosisDict );
		this.maintenanceTables.add( this.effectsDict );
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillEffects(db);
	}
	
	public void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "Cancer" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "BrokenLeg" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" );
		sql.set( IllTableSchema.DIAGNOSIS, "BrokenArm" );
		db.insert(this.ill);
	}
	
	public void fillEffects(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( EffectsTableSchema.NAME, "Mary" );
		sql.set( EffectsTableSchema.EFFECT, "LowWorkCpcy" );
		db.insert(this.effects);
		
		sql = db.sql();
		sql.set( EffectsTableSchema.NAME, "Mary" );
		sql.set( EffectsTableSchema.EFFECT, "HighCosts" );
		db.insert(this.effects);
		
		sql = db.sql();
		sql.set( EffectsTableSchema.NAME, "Lisa" );
		sql.set( EffectsTableSchema.EFFECT, "LowWorkCpcy" );
		db.insert(this.effects);
	}

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		super.createAdminUser(db);
		int userId = super.createTestUser(db, 1, "PreCQE");
		try {
			this.fillPrior(db.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(db.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User does not exists, check your code.", e);
		}
		super.linkDictionary( db, this.nameDict, IllTableSchema.NAME );
		super.linkDictionary( db, this.diagnosisDict, IllTableSchema.DIAGNOSIS );
		super.linkDictionary( db, this.nameDict, EffectsTableSchema.NAME );
		super.linkDictionary( db, this.effectsDict, EffectsTableSchema.EFFECT );
	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("FORALL X (ill(X,\"BrokenArm\") IMPL effects(X,\"LowWorkCpcy\"))"));
			prior.add(new Formula("FORALL X (ill(X,\"Cancer\") IMPL (effects(X,\"LowWorkCpcy\") AND effects(X,\"HighCosts\")))"));
			prior.add(new Formula("FORALL X (ill(X,\"BrokenLeg\") IMPL effects(X,\"HighCosts\"))"));
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("EXISTS X (effects(X,\"HighCosts\"))"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}
