package server.database.schema.app;

import static org.junit.Assert.fail;
import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.DiagnosisDictionary;
import server.database.schema.app.dict.NameDictionary;
import server.database.schema.app.dict.PrescriptionDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class IllTreatExample extends MaintenanceDatabaseSchema {
	
	public final IllTableSchema ill;
	public final TreatTableSchema treat;
	private int LYING_ID = 1;
	private int REFUSAL_ID = 2;
	private int COMBINED_ID = 3;
	public final NameDictionary nameDict;
	public final DiagnosisDictionary diagnosisDict;
	public final PrescriptionDictionary prescriptionDict;

	public IllTreatExample() throws DatabaseException {
		super("Ill Treat Example");
		
		this.ill = new IllTableSchema();
		this.treat = new TreatTableSchema();
		
		this.appTables.add(this.ill);
		this.appTables.add(this.treat);
		
		this.nameDict = new NameDictionary();
		this.diagnosisDict = new DiagnosisDictionary();
		this.prescriptionDict = new PrescriptionDictionary();
		
		this.maintenanceTables.add( this.nameDict );
		this.maintenanceTables.add( this.diagnosisDict );
		this.maintenanceTables.add( this.prescriptionDict );
	}
	
	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillTreat(db);
	}
	
	private void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" ).set( IllTableSchema.DIAGNOSIS, "Flu");	
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" ).set( IllTableSchema.DIAGNOSIS, "Aids");	
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mike" ).set( IllTableSchema.DIAGNOSIS, "Flu");	
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "John" ).set( IllTableSchema.DIAGNOSIS, "Flu");	
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Kate" ).set( IllTableSchema.DIAGNOSIS, "Flu");	
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "James" ).set( IllTableSchema.DIAGNOSIS, "Flu");	
		db.insert(this.ill);		
	}

	private void fillTreat(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" ).set( TreatTableSchema.PRESCRIPTION, "MedicineA");	
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" ).set( TreatTableSchema.PRESCRIPTION, "MedicineB");	
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Lisa" ).set( TreatTableSchema.PRESCRIPTION, "MedicineB");	
		db.insert(this.treat);		
	}
	
	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase mdb) throws DatabaseException {
		super.createDefaultAccessRights(mdb);
		super.createDefaultUsers(mdb);
	
		User user;
		try {
			user = mdb.getUserManagement().load(LYING_ID);
			// LYING_ID entspricht der USER_ID, die dem Lying Zensor zugeordnet wird.
			this.fillPrior(user);
			this.fillConfidentialityPolicy(user);
			
			// REFUSAL_ID entspricht der USER_ID, die dem Refusal Zensor zugeordnet wird.
			user = mdb.getUserManagement().load(REFUSAL_ID);
			this.fillPrior(user);
			this.fillConfidentialityPolicy(user);
			
			// COMBINED_ID entspricht der USER_ID, die dem Combined Zensor zugeordnet wird.
			user = mdb.getUserManagement().load(COMBINED_ID);
			this.fillPrior(user);
			this.fillConfidentialityPolicy(user);
			
			super.linkDictionary( mdb, this.nameDict, IllTableSchema.NAME );
			super.linkDictionary( mdb, this.diagnosisDict, IllTableSchema.DIAGNOSIS );
			super.linkDictionary( mdb, this.nameDict, TreatTableSchema.NAME );
			super.linkDictionary( mdb, this.prescriptionDict, TreatTableSchema.PRESCRIPTION );
		} catch (UnknownUserException e1) {
			throw new DatabaseException("User does not exists, check default user creation.", e1);
		}
	}
	
	public void fillPrior(User user) throws DatabaseException {
		Formula formula;
		PriorUser prior = user.getPriorUser();
		try {
			formula = new Formula("FORALL X,Z (treat(X,Z) IMPL EXISTS Y ill(X,Y))");
			prior.add(formula);

			formula = new Formula("EXISTS X (treat(X,\"MedicineB\") AND ill(X,\"Flu\"))");
			prior.add(formula);

			formula = new Formula("FORALL X ((treat(X,\"MedicineA\") AND treat(X,\"MedicineB\")) IMPL ill(X,\"Aids\"))");
			prior.add(formula);
		} catch (ParserException e) {
			throw new DatabaseException("Formular in test wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		Formula formula;
		ConfidentialityPolicy policy = user.getConfidentialityPolicy();
		try {
			formula = new Formula("EXISTS X ill(X,\"Aids\")");
			policy.add(formula, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);

			formula = new Formula("ill(\"Mary\",\"Flu\")");	
			policy.add(formula, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		
			formula = new Formula("ill(\"Mike\",\"Flu\")");	
			policy.add(formula, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);

			formula = new Formula("ill(\"John\",\"Flu\")");	
			policy.add(formula, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);

			formula = new Formula("ill(\"Kate\",\"Flu\")");
			policy.add(formula, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch (ParserException e) {
			e.printStackTrace();
			fail("ParserException cought");
		}
	}
}
