package server.database.schema;

import java.util.ArrayList;
import java.util.List;

import exception.DatabaseException;

public class TableSchema {

	public final String name;
	public final SchemaColumn[] columns;
	
	public TableSchema( String name, SchemaColumn ... columns ) throws DatabaseException {
		this.name = name;
		this.columns = columns;
		
		if ( columns.length == 0 ) {
			throw new DatabaseException("A SQL Table needs at least one column!", null);
		}
		
		for ( SchemaColumn column : columns ) {
			column.setParent( this );
		}
	}
	
	public List<String> generateCreateSQL() {
		ArrayList<String> result = new ArrayList<String>();
		String primaryKeysSQL = "";
		String primaryKeysSep = "";
		String foreignKeysSQL = "";
		String checkValuesSQL = "";
		
		// Begin table definition
		String createSQL = "CREATE TABLE \""+this.name+"\" ( ";
		
		// Column definition
		String sep = "";
		for ( SchemaColumn column : columns ) {
			// Create SQL for column
			createSQL += sep + "\""+column.name+"\" "+column.type;
			if ( column.defaultValue != null ) {
				createSQL += " DEFAULT "+column.defaultValue;
			}
			if ( column.notNull ) {
				createSQL += " NOT NULL ENABLE";
			}
			sep = ", ";
			
			// Generate Constraint SQL
			if ( column.primary ) {
				primaryKeysSQL += primaryKeysSep + "\""+column.name+"\"";
				primaryKeysSep = ", ";
			}
			if ( column.foreignKey != null ) {
				String keyName = this.name+"_"+column.name+"_FK";
				foreignKeysSQL += sep + "CONSTRAINT \""+keyName+"\" FOREIGN KEY (\""+column.name+"\") REFERENCES \""+column.foreignKey.getParent().name+"\" (\""+column.foreignKey.name+"\") ENABLE";
			}
			if ( column.unique ) {
				String uniqueName = this.name+"_"+column.name+"_UN";
				foreignKeysSQL += sep + "CONSTRAINT \""+uniqueName+"\" UNIQUE (\""+column.name+"\") ENABLE";
			}
			if ( column.allowedValues.length > 0 ) {
				String keyName = this.name+"_"+column.name+"_CHK";
				String values = "";
				String valueSep = "";
				for ( String allowedValue : column.allowedValues ) {
					values += valueSep + "'"+allowedValue+"'";
					valueSep = ", ";
				}
				checkValuesSQL += sep + "CONSTRAINT \""+keyName+"\" CHECK ("+column.name+" IN ("+values+")) ENABLE";
			}
			
			// Handle autoincrement as trigger.
			if ( column.autoincrement ) {
				// Create SEQUENCE for the trigger to use.
				String seqName = this.name+"_"+column.name+"_SQ";
				result.add( "CREATE SEQUENCE \""+seqName+"\" MINVALUE 0 MAXVALUE 9999999 INCREMENT BY 1 START WITH 0" );
				
				String triggerName = this.name+"_"+column.name+"_TRG";
				String triggerSQL = "CREATE OR REPLACE TRIGGER \""+triggerName+"\" before\n";
				triggerSQL += "INSERT ON \""+this.name+"\" FOR EACH row BEGIN IF inserting THEN IF :NEW.\""+column.name+"\" IS NULL THEN\n";
				triggerSQL += "SELECT "+seqName+".nextval INTO :NEW.\""+column.name+"\" FROM dual;\n";
				triggerSQL += "END IF;\n";
				triggerSQL += "END IF;\n";
				triggerSQL += "END;";
				result.add( triggerSQL );
				
				// Activate the trigger
				result.add( "ALTER TRIGGER \""+triggerName+"\" ENABLE" );
			}
		}
		
		// Append constraints (primary keys, foreign keys)
		if ( !primaryKeysSQL.isEmpty() ) {
			createSQL += sep + "CONSTRAINT \""+this.name+"_PK"+"\" PRIMARY KEY ("+primaryKeysSQL+") ENABLE";
		}
		createSQL += foreignKeysSQL;
		createSQL += checkValuesSQL;
		
		// End table definition
		createSQL += " )";
		
		// Add CREATE statement as first item in the list.
		result.add(0, createSQL);
		
		return result;
	}
	
	public List<String> generateDropSQL() {
		ArrayList<String> result = new ArrayList<String>();
		
		// First delete the table (and trigger through cascade).
		result.add( "DROP TABLE \""+this.name+"\" cascade constraints" );
		
		// And now all associated sequences (each trigger has one).
		for ( SchemaColumn column : this.columns ) {
			if ( column.autoincrement ) {
				String seqName = this.name+"_"+column.name+"_SQ";
				result.add( "DROP SEQUENCE \""+seqName+"\"" );
			}
		}
		
		return result;
	}
	
	public List<String> generateTruncateSQL() {
		ArrayList<String> result = new ArrayList<String>();
		result.add( "TRUNCATE TABLE \""+this.name+"\"" );
		return result;
	}
}
