package server.database.cache;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import exception.DatabaseException;

import server.database.NewMaintenanceDatabase;
import server.database.sql.SQLDatabase;
import server.util.Tuple;

/**
 * Provides easy access to schema information with built-in caching.
 */
public class AppDatabaseSchemaCache {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database.cache");

	/**
	 * Maximum size (number of entries) of the dictionary cache.
	 * With 1000*1000 entries you have roughly 30mb cache with each item being 30 characters long.
	 */
	public final static int DICTIONARY_SIZE_LIMIT = 1000*10;

	private Map<String, List<Attribute>> dataCache;
	private SQLDatabase appDB;
	private NewMaintenanceDatabase maintenanceDB;
	private List<Attribute> loadedDictionaryHistory;
	private int dictionarySizeCounter;
	private List<Attribute> loadedActiveDomainHistory;
	private int activeDomainSizeCounter;
	
	public AppDatabaseSchemaCache( SQLDatabase appDB, NewMaintenanceDatabase maintenanceDB ) throws DatabaseException {
		this.appDB = appDB;
		this.maintenanceDB = maintenanceDB;
		this.loadedDictionaryHistory = new LinkedList<Attribute>();
		this.loadedActiveDomainHistory = new LinkedList<Attribute>();
		
		this.reloadRelations();
	}
	
	/**
	 * Reloads the available relations (tables) in the database.
	 * Should only be called if the database relations have changed.
	 * @throws DatabaseException
	 */
	public void reloadRelations() throws DatabaseException {
		this.dataCache = new HashMap<String, List<Attribute>>();

		for ( String relation : this.appDB.getTables() ) {
			List<Attribute> attributes = new LinkedList<Attribute>();
			for ( Tuple<String, String> attribute : this.appDB.getColumnNamesWithAttributeTypes(relation) ) {
				attributes.add( new Attribute(this, appDB, this.maintenanceDB, relation, attribute.getFirstElement(), attribute.getSecondElement()) );
			}
			
			this.dataCache.put(relation, attributes);
		}
	}

	/**
	 * Returns the names of all available relations.
	 * @return Set of all available relations in the app database.
	 */
	public Set<String> getRelations() {
		return this.dataCache.keySet();
	}
	
	/**
	 * Returns the attributes of a specific relation. Each attribute consists of the
	 * corresponding column name and type and a list of possible values loaded
	 * from the dictionaries.
	 * 
	 * @param name Name of the relation.
	 * @return List (the order is important) of attributes.
	 */
	public List<Attribute> getRelation( String name ) {
		return this.dataCache.get(name.toUpperCase());
	}
	
	/**
	 * This method MUST be called whenever a dictionary is loaded on-demand.
	 * @param attribute Reference to the attribute that loaded the dictionary.
	 * @throws DatabaseException 
	 */
	public void loadedDictionary( Attribute attribute ) throws DatabaseException {
		this.dictionarySizeCounter += attribute.getPossibleValues().size();
		
		// Scale the dictionary down until we're in the size limit.
		while ( this.dictionarySizeCounter > DICTIONARY_SIZE_LIMIT ) {
			if ( this.loadedDictionaryHistory.isEmpty() ) {
				// Removed every element but still not enough size for the newly loaded dictionary. Bad luck.
				logger.warn("Trying to scale down cache, but newly loaded dictionary exceeds cache limit.");
				break;
			}
			
			Attribute loadedAttribute = this.loadedDictionaryHistory.get(0);
			this.dictionarySizeCounter -= loadedAttribute.getPossibleValues().size();
			loadedAttribute.resetPossibleValues();
		}
		
		this.loadedDictionaryHistory.add( attribute );
	}
	
	/**
	 * This method MUST be called whenever an active domain is loaded on-demand.
	 * @param attribute Reference to the attribute that loaded the dictionary.
	 * @throws DatabaseException 
	 */
	public void loadedActiveDomain( Attribute attribute ) throws DatabaseException {
		this.activeDomainSizeCounter += attribute.getActiveDomain().size();
		
		// Scale the dictionary down until we're in the size limit.
		while ( this.activeDomainSizeCounter > DICTIONARY_SIZE_LIMIT ) {
			if ( this.loadedActiveDomainHistory.isEmpty() ) {
				// Removed every element but still not enough size for the newly loaded dictionary. Bad luck.
				logger.warn("Trying to scale down cache, but newly loaded active domain exceeds cache limit.");
				break;
			}
			
			Attribute loadedAttribute = this.loadedActiveDomainHistory.get(0);
			this.activeDomainSizeCounter -= loadedAttribute.getActiveDomain().size();
			loadedAttribute.resetActiveDomainFromDB();
		}
		
		this.loadedActiveDomainHistory.add( attribute );
	}
	
	/**
	 * Resets the custom set active domain values for each stored attribute.
	 */
	public void resetCustomActiveDomain() {
		for ( List<Attribute> attributes : dataCache.values() ) {
			for ( Attribute attribute : attributes ) {
				attribute.resetCustomActiveDomain();
			}
		}
	}
}
