package server.data;

import exception.DatabaseException;
import server.database.schema.Schema;
import server.database.schema.SchemaColumn;
import server.database.schema.maintenance.UserTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLUpdate;
import server.util.Observable;

public final class User extends Observable<User,Role> {
	

	private SQLDatabase db;
	private final int id;
	private String username;
	private String password;
	private String censor;
	private Role role;
	private final Log log;
	private final ConfidentialityPolicy confidentialityPolicy;
	private final PriorUser priorUser;
	private final Knowledge knowledge;
	
	/**
	 * Constructs a new user object. ONLY to be used by UserManagement class.
	 * The visibility of the constructor is therefore limited to the data package.
	 * 
	 * @param id
	 * @param username
	 * @param password
	 * @param censor
	 * @param role
	 * @param log
	 * @param confPol
	 * @param priorUser
	 */
	User( SQLDatabase db, int id, String username, String password, String censor, Role role, Log log, ConfidentialityPolicy confPol, PriorUser priorUser, PriorAll priorAll, SchemaConstraints schemaConstraints ) {
		this.db = db;
		this.id = id;
		this.username = username;
		this.password = password;
		this.censor = censor;
		this.role = role;
		this.log = log;
		this.confidentialityPolicy = confPol;
		this.priorUser = priorUser;
		
		this.knowledge = new Knowledge(this.log, priorAll, priorUser, schemaConstraints);
	}
	
	/**
	 * Writes changes of the user to the database. Observers won't be notified.
	 * 
	 * @param column The column which will get a new value.
	 * @param value The new value of the column.
	 * @throws DatabaseException 
	 */
	private void updateValue( SchemaColumn column, String value ) throws DatabaseException {
		SQLUpdate sql = this.db.newUpdate();
		sql.set( column, value );
		sql.where( UserTableSchema.ID, this.id );
		sql.update( Schema.maintenanceDatabase.cqeUser );
	}

	/**
	 * Returns the unique id. The id cannot be changed.
	 * @return
	 */
	public int getId() {
		return this.id;
	}
	
	/**
	 * Returns the username.
	 * @return
	 */
	public String getUsername() {
		return this.username;
	}

	/**
	 * Sets the username to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers won't be notified.
	 * 
	 * @param username
	 * @throws DatabaseException 
	 */
	public void setUsername(String username) throws DatabaseException {
		this.username = username;
		this.updateValue( UserTableSchema.LOGIN, this.username );
	}

	/**
	 * Returns the password.
	 * @return
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * Sets the password to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers won't be notified.
	 * 
	 * @param username
	 * @throws DatabaseException 
	 */
	public void setPassword(String password) throws DatabaseException {
		this.password = password;
		this.updateValue( UserTableSchema.PASSWORD, this.password );
	}
	
	/**
	 * Returns the current censor.
	 * @return
	 */
	public String getCensor() {
		return this.censor;
	}
	
	/**
	 * Sets the censor to a new value.
	 * The automaton will check if the censor is applicable and may change it.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers won't be notified.
	 * 
	 * @param censor
	 * @throws DatabaseException
	 */
	public void setCensor(String censor) throws DatabaseException {
		this.censor = censor;
		this.updateValue( UserTableSchema.CENSOR, this.censor );
	}

	/**
	 * Returns the role assigned to this user.
	 * A user has exactly one role assigned.
	 */
	public Role getRole() {
		return role;
	}

	/**
	 * Sets the role to a new value.
	 * The role must be loaded by {@link Role#load(SQLDatabase, int)} beforehand.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param role
	 * @throws DatabaseException
	 */
	public void setRole(Role role) throws DatabaseException {
		this.role = role;		// FIXME: create copy?
		this.updateValue( UserTableSchema.ROLE_ID, String.valueOf(role.getId()) );
		this.notifyObservers(Action.MODIFY, role);
	}
	
	/**
	 * Returns the log.
	 * @return
	 */
	public Log getLog() {
		return log;
	}

	/**
	 * Returns the confidentiality policy.
	 * @return
	 */
	public ConfidentialityPolicy getConfidentialityPolicy() {
		return confidentialityPolicy;
	}

	/**
	 * Returns the prior of the user.
	 * @return
	 */
	public PriorUser getPriorUser() {
		return priorUser;
	}
	
	/**
	 * Returns the complete knowledge of the user.
	 * This is a collection of the log, the prior of the user,
	 * the prior for all users and the schema constraints.
	 * @return
	 */
	public Knowledge getKnowledge() {
		return knowledge;
	}
}
