package server.data;

import java.util.HashSet;
import java.util.Set;

import exception.DatabaseException;

import server.database.schema.maintenance.LogTableSchema;
import server.parser.Formula;
import server.parser.node.AtomPredicateNode;

/**
 * One Log entry. The get methods will only return immutable
 * objects like Strings or copies of mutable objects like Formulas.
 * This means any changes MUST be done through the set methods to
 * take any effect.
 * 
 * Any call of a set method will lead to a notification for all Log-observers.
 */
public class LogEntry {

	private Log log;
	private int id;
	private String interactionType;
	private Formula interaction;
	private String interactionDate;
	private Formula answer;
	private String answerType;
	private Set<AtomPredicateNode> effectiveUpdates;
	
	/**
	 * Creates a new LogEntry. All mutable values (like formulas) must be copied beforehand!
	 * This constructor has the package visibility because it should only be called by methods of the class Log.
	 * 
	 * @param interaction
	 * @param interactionType
	 * @param interactionDate
	 * @param answer
	 * @param answerType
	 * @param effectiveUpdates
	 */
	LogEntry( Log log, int id, Formula interaction, String interactionType, String interactionDate, Formula answer, String answerType, Set<AtomPredicateNode> effectiveUpdates ) {
		this.log = log;
		this.id = id;
		this.interaction = interaction;
		this.interactionType = interactionType;
		this.interactionDate = interactionDate;
		this.answer = answer;
		this.answerType = answerType;
		this.effectiveUpdates = effectiveUpdates;
	}
		
	/**
	 * Returns the unique id. The id cannot be changed.
	 * @return
	 */
	public int getId() {
		return this.id;
	}

	/**
	 * Returns the interaction type.
	 * @return
	 */
	public String getInteractionType() {
		return this.interactionType;
	}
	
	/**
	 * Sets the interaction type to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param interactionType
	 * @throws DatabaseException
	 */
	public void setInteractionType(String interactionType) throws DatabaseException {
		this.interactionType = interactionType;
		this.log.updateEntry( this, LogTableSchema.INTERACTIONTYPE, this.interactionType );
	}
	
	/**
	 * Returns the a copy of the interaction.
	 * Modifying this copy will have no effect on the log.
	 * @return
	 */
	public Formula getInteraction() {
		return new Formula( this.interaction );
	}
	
	/**
	 * Sets the interaction to a new value.
	 * A copy of the given formula will be created. Modifying the formula will have no effect on the log.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param interaction
	 * @throws DatabaseException
	 */
	public void setInteraction(Formula interaction) throws DatabaseException {
		this.interaction = new Formula(interaction);
		this.log.updateEntry( this, LogTableSchema.INTERACTION, this.interaction.toString() );
	}
	
	/**
	 * Returns the interaction date.
	 * @return
	 */
	public String getInteractionDate() {
		return this.interactionDate;
	}
	
	/**
	 * Sets the interaction date to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param interactionDate
	 * @throws DatabaseException
	 */
	public void setInteractionDate(String interactionDate) throws DatabaseException {
		this.interactionDate = interactionDate;
		this.log.updateEntry( this, LogTableSchema.INTERACTIONDATE, this.interactionDate );
	}
	
	/**
	 * Returns a copy of the answer.
	 * Modifying this copy will have no effect on the log.
	 * @return
	 */
	public Formula getAnswer() {
		return new Formula( this.answer );
	}
	
	/**
	 * Sets the answer to a new value.
	 * A copy of the given formula will be created. Modifying the formula will have no effect on the log.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param answer
	 * @throws DatabaseException
	 */
	public void setAnswer(Formula answer) throws DatabaseException {
		this.answer = new Formula( answer );
		this.log.updateEntry( this, LogTableSchema.ANSWER, this.answer.toString() );
	}
	
	/**
	 * Returns the answer type.
	 * @return
	 */
	public String getAnswerType() {
		return this.answerType;
	}
	
	/**
	 * Sets the answer type to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param answerType
	 * @throws DatabaseException
	 */
	public void setAnswerType(String answerType) throws DatabaseException {
		this.answerType = answerType;
		this.log.updateEntry( this, LogTableSchema.ANSWERFORMULATYPE, this.answerType );
	}
	
	/**
	 * Returns a copy of the effective updates.
	 * Modifying this copy will have no effect on the log.
	 * @return
	 */
	public Set<AtomPredicateNode> getEffectiveUpdates() {
		HashSet<AtomPredicateNode> copy = new HashSet<AtomPredicateNode>();
		for ( AtomPredicateNode update : this.effectiveUpdates ) {
			copy.add( (AtomPredicateNode)update.clone() );
		}
		
		return copy;
	}
	
	/**
	 * Sets the effective updates to a new value.
	 * A copy of the given set and the items inside will be created. Modifying the set or the items inside will have no effect on the log.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param effectiveUpdates
	 * @throws DatabaseException
	 */
	public void setEffectiveUpdates(Set<AtomPredicateNode> effectiveUpdates) throws DatabaseException {
		HashSet<AtomPredicateNode> copy = new HashSet<AtomPredicateNode>();
		for ( AtomPredicateNode update : effectiveUpdates ) {
			copy.add( (AtomPredicateNode)update.clone() );
		}
		
		this.effectiveUpdates = copy;
		this.log.updateEntry( this, LogTableSchema.EFFECTIVE_UPDATES, this.effectiveUpdates.toString() );
	}	
	
	/**
	 * Returns the log this entry belongs to.
	 * @return
	 */
	public Log getLog() {
		return this.log;
	}
}
