package server.data;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import exception.DatabaseException;

import server.database.schema.Schema;
import server.database.schema.SchemaColumn;
import server.database.schema.app.DictTableSchema;
import server.database.schema.maintenance.DictionariesTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.util.Observable;

public class Dictionary extends Observable<Dictionary,DictionaryEntry> {

	private ConcurrentHashMap<String,DictionaryEntry> entries;
	private SQLDatabase db;
	
	public Dictionary( SQLDatabase db ) throws DatabaseException {
		this.db = db;
		this.entries = new ConcurrentHashMap<String, DictionaryEntry>();
		
		// Load entries from database.
		// Determine which dictionary entries to load
		SQLSelect sql = this.db.newSelect();
		sql.select(DictionariesTableSchema.RELATION);
		sql.select(DictionariesTableSchema.ATTRIBUTE);
		sql.select(DictionariesTableSchema.DICT_TABLE);
		sql.from(Schema.maintenanceDatabase.dictionaries);
		
		SQLTable result = sql.get();
		for ( String[] row : result ) {
				String relation = row[0];
				String attribute = row[1];
				String dict_table = row[2];
				
				// Get content of the dictionary
				sql = this.db.newSelect();
				sql.select(DictTableSchema.ENTRY.name);
				sql.from(dict_table);
				sql.order_by(DictTableSchema.ID.name);
				
				SQLTable dictValues = sql.get();
				List<String> values = new ArrayList<String>(dictValues.getRowCount());
				for( String[] valueRow : dictValues ) {
					values.add(valueRow[0]);
				}
				
				// Create new DictionaryEntry and add to list.
				String identifier = this.createId(relation, attribute);
				DictionaryEntry entry = new DictionaryEntry(this, identifier, dict_table, relation, attribute, values);
				this.entries.put( identifier, entry );
		}
	}
	
	/**
	 * Creates a unique identifier for a relation/attribute pair.
	 * @param relation Relation name.
	 * @param attribute Attribute name.
	 * @return Unique identifier.
	 */
	private String createId(String relation, String attribute) {
		return relation.toUpperCase()+"_"+attribute.toUpperCase();
	}
	
	/**
	 * Writes the changes of a DictionaryEntry to the database and notifies all observers.
	 * This method has the package visibility because only methods in the DictionaryEntry class should call it!
	 * 
	 * @param entry The DictionaryEntry that has changed.
	 * @param column The column which will get a new value.
	 * @param value The new value of the column.
	 * @throws DatabaseException
	 */
	void updateEntry( DictionaryEntry entry, SchemaColumn column, String value ) throws DatabaseException {
		SQLUpdate sql = this.db.newUpdate();
		sql.set(column, value);
		sql.where(DictionariesTableSchema.RELATION, entry.getRelation());
		sql.where(DictionariesTableSchema.ATTRIBUTE, entry.getAttribute());
		sql.update( Schema.maintenanceDatabase.dictionaries );
		
		this.notifyObservers( Observable.Action.MODIFY, entry );
	}
	
	/**
	 * Returns a DictTableSchema instance for the given name.
	 * @param identifier
	 * @return
	 * @throws DatabaseException
	 */
	private DictTableSchema getDictionarySchema( String identifier ) throws DatabaseException {
		DictTableSchema schema = new DictTableSchema(identifier) {
			@Override
			public String[] getDictionaryContents() {
				return new String[0];
			}
		};
		
		return schema;
	}
	
	/**
	 * Adds a new entry to the dictionary.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param formula
	 * @param type
	 * @param preservation
	 * @return
	 * @throws DatabaseException
	 */
	public synchronized DictionaryEntry add( String relation, String attribute, List<String> values ) throws DatabaseException {
		String identifier = this.createId(relation, attribute);
		
		// Copy all mutable objects.
		values = new ArrayList<String>(values);
		
		// Create new database entry.
		SQLInsert sql = this.db.newInsert();
		sql.set( DictionariesTableSchema.RELATION, relation );
		sql.set( DictionariesTableSchema.ATTRIBUTE, attribute );
		sql.set( DictionariesTableSchema.DICT_TABLE, identifier );
		
		int num = sql.insert( Schema.maintenanceDatabase.dictionaries );
		if ( num != 1 ) {
			throw new DatabaseException("Failed to insert dictionary entry. Rows inserted: "+num, null );
		}
		
		// Create dictionary table.
		DictTableSchema schema = this.getDictionarySchema(identifier);
		for ( String sqlCreate : schema.generateCreateSQL() ) {
			this.db.rawQuery(sqlCreate);
		}
		
		for ( String value : values ) {
			sql = this.db.newInsert();
			sql.set( DictTableSchema.ENTRY, value );
			sql.insert(identifier);
		}
		
		// Create new LogEntry and add to list.
		DictionaryEntry entry = new DictionaryEntry(this, identifier, identifier, relation, attribute, values);
		this.entries.put( identifier, entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.ADD, entry );
		
		return entry;
	}
	
	/**
	 * Removes a dictionary entry.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param entry
	 * @throws DatabaseException
	 */
	public synchronized void remove( DictionaryEntry entry ) throws DatabaseException {
		// Remove dictionary table.
		DictTableSchema schema = this.getDictionarySchema(entry.getIdentifier());
		for ( String sqlCreate : schema.generateDropSQL() ) {
			this.db.rawQuery(sqlCreate);
		}
		
		// Remove from database.
		SQLDelete sql = this.db.newDelete();
		sql.where(DictionariesTableSchema.RELATION, entry.getRelation());
		sql.where(DictionariesTableSchema.ATTRIBUTE, entry.getAttribute());
		sql.delete(Schema.maintenanceDatabase.dictionaries);
		
		// Remove from list.
		this.entries.remove( entry.getIdentifier() );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.DELETE, entry );
	}
	
	/**
	 * Removes all dictionary entries.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @throws DatabaseException
	 */
	public synchronized void clear() throws DatabaseException {
		// First remove dictionary tables.
		for ( DictionaryEntry entry : this.entries.values() ) {
			DictTableSchema schema = this.getDictionarySchema(entry.getIdentifier());
			for ( String sqlCreate : schema.generateDropSQL() ) {
				this.db.rawQuery(sqlCreate);
			}
		}
		
		// Remove dictionary entries.
		SQLDelete sql = this.db.newDelete();
		sql.delete(Schema.maintenanceDatabase.dictionaries);
		
		// Notify observers.
		for ( DictionaryEntry entry : this.entries.values() ) {
			this.notifyObservers( Observable.Action.DELETE, entry );
		}
		
		// Clear the list.
		this.entries.clear();
	}
	
	/**
	 * Returns the dictionary entry corresponding to the given relation/attribute pair.
	 * 
	 * @param relation Name of the relation.
	 * @param attribute Name of the attribute.
	 * @return
	 */
	public DictionaryEntry getEntry( String relation, String attribute ) {
		return this.entries.get( this.createId(relation, attribute) );
	}
	
	/**
	 * Returns a copy of the dictionary values for the given relation/attribute pair.
	 * Modifying the returned list will have no effect on the dictionary.
	 * 
	 * @param relation Name of the relation.
	 * @param attribute Name of the attribute.
	 * @return
	 */
	public List<String> getValues( String relation, String attribute ) {
		return this.getEntry(relation, attribute).getValues();
	}
	
	/**
	 * Returns the number of dictionary entries.
	 * @return Dictionary count.
	 */
	public int size() {
		return this.entries.size();
	}
	
	/**
	 * Returns whether the dictionary is empty or not.
	 * @return True if the the dictionary has no entries, false otherwise.
	 */
	public boolean isEmpty() {
		return this.entries.isEmpty();
	}
}
