package server.data;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownConfidentialityPolicyException;

import server.database.schema.Schema;
import server.database.schema.SchemaColumn;
import server.database.schema.maintenance.ConfidentialityPolicyTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.parser.Formula;
import server.util.Observable;
import user.ConfidentialityPolicyItem;

public class ConfidentialityPolicy extends Observable<ConfidentialityPolicy,ConfidentialityPolicyEntry> implements Iterable<ConfidentialityPolicyEntry> {

	private CopyOnWriteArrayList<ConfidentialityPolicyEntry> entries;
	private SQLDatabase db;
	private int userId;
	
	/**
	 * Loads the confidentiality policy of a user from the database.
	 * ONLY to be called by {@link User#load(int)} or {@link User#load(String)}.
	 * The visibility of the constructor is therefore limited to the data package.
	 * @param db DB from which the confidentiality policy will be loaded.
	 * @param userId ID of the user.
	 * @throws DatabaseException
	 */
	ConfidentialityPolicy( SQLDatabase db, int userId ) throws DatabaseException {
		this.entries = new CopyOnWriteArrayList<ConfidentialityPolicyEntry>();
		this.db = db;
		this.userId = userId;
		
		// Load entries from database.
		SQLSelect sql = this.db.newSelect();
		sql.select(ConfidentialityPolicyTableSchema.ID).select(ConfidentialityPolicyTableSchema.FORMULA).select(ConfidentialityPolicyTableSchema.POLICY_TYPE).select(ConfidentialityPolicyTableSchema.PRESERVATION);
		sql.from(Schema.maintenanceDatabase.confidentialityPolicy);
		sql.where(ConfidentialityPolicyTableSchema.USER_ID, userId);
		
		SQLTable result = sql.get();
		for ( String[] row : result ) {
			try {
				int id = Integer.valueOf( row[0] );
				Formula formula = new Formula( row[1] );
				CqeType.PolicyType type = CqeType.PolicyType.valueOf( row[2].toUpperCase() );
				CqeType.PolicyPreservation preservation = CqeType.PolicyPreservation.valueOf( row[3].toUpperCase() );

				// Create new ConfidentialityPolicyEntry and add to list.
				ConfidentialityPolicyEntry entry = new ConfidentialityPolicyEntry(this, id, formula, type, preservation);
				this.entries.add( entry );
			} catch (ParserException e) {
				throw new DatabaseException("Unable to parse confidentiality policy entry (malformed formula)", e);
			} catch (NumberFormatException e) {
				throw new DatabaseException("Unable to parse confidentiality policy entry (malformed id)", e);
			} catch (IllegalArgumentException e) {
				throw new DatabaseException("Unable to parse confidentiality policy entry (malformed type or preservation)", e);
			}
		}
	}
	
	/**
	 * Writes the changes of a ConfidentialityPolicyEntry to the database and notifies all observers.
	 * This method has the package visibility because only methods in the ConfidentialityPolicyEntry class should call it!
	 * 
	 * @param entry The ConfidentialityPolicyEntry that has changed.
	 * @param column The column which will get a new value.
	 * @param value The new value of the column.
	 * @throws DatabaseException
	 */
	void updateEntry( ConfidentialityPolicyEntry entry, SchemaColumn column, String value ) throws DatabaseException {
		SQLUpdate sql = this.db.newUpdate();
		sql.set(column, value);
		sql.where(ConfidentialityPolicyTableSchema.ID, entry.getId());
		sql.update( Schema.maintenanceDatabase.confidentialityPolicy );
		
		this.notifyObservers( Observable.Action.MODIFY, entry );
	}
	
	/**
	 * Adds a new entry to the confidentiality policy.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param formula
	 * @param type
	 * @param preservation
	 * @return
	 * @throws DatabaseException
	 */
	public synchronized ConfidentialityPolicyEntry add( Formula formula, CqeType.PolicyType type, CqeType.PolicyPreservation preservation ) throws DatabaseException {
		// Copy all mutable objects.
		formula = new Formula(formula);
		
		// Create new database entry.
		SQLInsert sql = this.db.newInsert();
		sql.setGenerated( ConfidentialityPolicyTableSchema.ID );
		sql.set( ConfidentialityPolicyTableSchema.FORMULA, formula.toString() );
		sql.set( ConfidentialityPolicyTableSchema.POLICY_TYPE, type.toString().toLowerCase() );
		sql.set( ConfidentialityPolicyTableSchema.PRESERVATION, preservation.toString().toLowerCase() );
		sql.set( ConfidentialityPolicyTableSchema.USER_ID, this.userId );
		
		int num = sql.insert( Schema.maintenanceDatabase.confidentialityPolicy );
		SQLTable generatedValues = sql.getGeneratedValues();
		if ( num != 1 || generatedValues == null || generatedValues.getRowCount() != 1 ) {
			throw new DatabaseException("Failed to insert confidentiality policys entry. Rows inserted: "+num, null );
		}
		
		// Get ID for the entry.
		int id = Integer.valueOf( generatedValues.getFirstRow()[0] );
		
		// Create new LogEntry and add to list.
		ConfidentialityPolicyEntry entry = new ConfidentialityPolicyEntry(this, id, formula, type, preservation);
		this.entries.add( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.ADD, entry );
		
		return entry;
	}
	
	/**
	 * Removes a confidentiality policy entry.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param entry
	 * @throws DatabaseException
	 */
	public synchronized void remove( ConfidentialityPolicyEntry entry ) throws DatabaseException {
		if ( entry.getConfidentialityPolicy() != this ) {
			throw new DatabaseException("ConfidentialityPolicyEntry doesn't belong to this ConfidentialityPolicy.", null);
		}
		
		// Remove from database.
		SQLDelete sql = this.db.newDelete();
		sql.where(ConfidentialityPolicyTableSchema.ID, entry.getId());
		sql.delete(Schema.maintenanceDatabase.confidentialityPolicy);
		
		// Remove from list.
		this.entries.remove( entry );
		
		// Notify observers.
		this.notifyObservers( Observable.Action.DELETE, entry );
	}
	
	/**
	 * Removes all confidentiality policy entries of this user.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @throws DatabaseException
	 */
	public synchronized void clear() throws DatabaseException {
		// Remove all confidentiality policy entries for this user.
		SQLDelete sql = this.db.newDelete();
		sql.where(ConfidentialityPolicyTableSchema.USER_ID, this.userId);
		sql.delete(Schema.maintenanceDatabase.confidentialityPolicy);
		
		// Notify observers.
		for ( ConfidentialityPolicyEntry entry : this.entries ) {
			this.notifyObservers( Observable.Action.DELETE, entry );
		}
		
		// Clear the list.
		this.entries.clear();
	}
	
	/**
	 * Returns a specific confidentiality policy entry with the given id.
	 * @param id The Id of the confidentiality policy entry that will be returned.
	 * @return A confidentiality policy entry. Cannot be null.
	 * @throws UnknownConfidentialityPolicyException Throws the Exception if the confidentiality policy entry with the given id does not exist.
	 */
	public synchronized ConfidentialityPolicyEntry get( int id ) throws UnknownConfidentialityPolicyException {
		for ( ConfidentialityPolicyEntry entry : this.entries ) {
			if (entry.getId() == id) {
				return entry;
			}
		}
		throw new UnknownConfidentialityPolicyException( id ); 
	}
	
	/**
	 * Produces a copy of the confidentiality policy.
	 * Currently used to feed the client with information.
	 * @return Copy of the confidentiality policy.
	 */
	public synchronized List<ConfidentialityPolicyItem> getCopyAsList() {
		List<ConfidentialityPolicyItem> result = new LinkedList<ConfidentialityPolicyItem>();
		
		for ( ConfidentialityPolicyEntry entry : this.entries ) {
			ConfidentialityPolicyItem item = new ConfidentialityPolicyItem( entry.getId(), entry.getFormula(), entry.getType(), entry.getPreservation() );
			result.add( item );
		}
		
		return result;
	}
	
	@Override
	public synchronized Iterator<ConfidentialityPolicyEntry> iterator() {
		return this.entries.iterator();
	}
	
	/**
	 * Returns the number of confidentiality policy entries.
	 * @return Confidentiality Policy count.
	 */
	public int size() {
		return this.entries.size();
	}
	
	/**
	 * Returns whether the confidentiality policy is empty or not.
	 * @return True if the the confidentiality policy has no entries, false otherwise.
	 */
	public boolean isEmpty() {
		return this.entries.isEmpty();
	}
}
