package server.core;

import java.io.EOFException;
import java.io.IOException;
import java.net.Socket;

import notification.Notification;
import notification.serverToClient.AvailableDatabasesNotification;
import notification.serverToClient.ExceptionNotification;

import org.apache.log4j.Logger;

import communication.SocketCommunication;

import exception.IllegalFormulaOperationException;

import server.database.AppDatabase;
import server.database.DatabaseConfigurationList;
import server.database.NewMaintenanceDatabase;
import server.data.User;

public class Client extends SocketCommunication implements Runnable {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.core");

	private boolean running = false;
	private Server server = null;
	private User user = null;
	private NewMaintenanceDatabase maintenanceDB = null;
	private AppDatabase applicationDB = null;
	
	/**
	 * This constructor is used for every new established client connection.
	 * @param server
	 * @param clientSocket
	 * @throws IOException
	 */
	public Client(Server server, Socket clientSocket) throws IOException {
		super(clientSocket);
		this.running = true;
		this.server = server;
	}

	@Override
	public void run() {
		Notification notification = null;

		logger.info( "New client connection established." );
		
		// Send the client a list of databases.
		try {
			this.send( new AvailableDatabasesNotification(DatabaseConfigurationList.DEFAULT_LIST) );
		} catch (IOException e) {
			logger.info( "Unable to send database list to the client. Disconnecting.", e );
			this.disconnect();
			return;
		}

		while ( this.running ) {
			try {
				// This call blocks until a new notification is available.
				notification = this.recv();
				
				try {
					// Forward the message to the notification hub.
					server.getNotificationHub().handle(this, notification);
				} catch ( IllegalFormulaOperationException e) {
					// Runtime Exception
					this.send(new ExceptionNotification(e, ExceptionNotification.CRITICAL));
					break;
				}
			} catch ( EOFException e ) {
				// Client disconnected.
				logger.info( "Client "+this.toString()+" disconnected.", e );
				this.disconnect();
				break;
			} catch ( IOException e ) {
				logger.warn( "Communication with client "+this.toString()+" failed. Disconnecting.", e );
				this.disconnect();
				break;
			}
		}

		logger.info( "Stopping thread " + Thread.currentThread().getName() );
	}
	
	@Override
	public void disconnect() {
		this.running = false;
		super.disconnect();
	}
	
	public Server getServer() {
		return this.server;
	}
	
	public void setUser( User user ) {
		this.user = user;
	}
	
	public User getUser() {
		return this.user;
	}
	
	public void setMaintenanceDB( NewMaintenanceDatabase mDB ) {
		this.maintenanceDB = mDB;
	}
	
	public NewMaintenanceDatabase getMaintenanceDB() {
		return this.maintenanceDB;
	}
	
	public void setApplicationDB( AppDatabase appDB ) {
		this.applicationDB = appDB;
	}
	
	public AppDatabase getApplicationDB() {
		return this.applicationDB;
	}
}