package server.censor.signature.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;


import exception.DatabaseException;
import exception.UnsupportedFormulaException;
import server.database.cache.Attribute;
import server.database.schema.Schema;
import server.database.schema.maintenance.SignaturesExperimentEvaluationCumulatedSchema;
import server.database.schema.maintenance.SignaturesExperimentEvaluationSchema;
import server.database.schema.maintenance.SignaturesFlexiblePotSecAVTableSchema;
import server.database.schema.maintenance.SignaturesFlexiblePotSecTableSchema;
import server.database.schema.maintenance.SignaturesFlexibleQueryAVSchema;
import server.database.schema.maintenance.SignaturesFlexibleQuerySchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesPAVSchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesPSchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesSchema;
import server.database.schema.maintenance.SignaturesUnflexiblePotSecSchema;
import server.database.schema.maintenance.SignaturesUnflexibleQuerySchema;
import server.database.schema.maintenance.SignaturesUnflexibleSignaturesKrankheitSchema;
import server.database.schema.maintenance.SignaturesUnflexibleSignaturesSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.parser.Formula;

public class MaintenanceUtil {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.censor.signature");
	
	/**
	 * Methode mit welcher die signaturbasierte Zugriffskontrolle realisiert wird.
	 * s. Diplomarbeit: Kontrollierte Anfrageauswertung mit Hilfe von aus Integritaetsbedingungen erzeugten Signaturen (Torsten Schlotmann)
	 * Die gestellte Anfrage eines Benutzers wird durch einen in PL/SQL implementierten Zensor bearbeitet. Zu diesem Zweck wird die Anfrage
	 * zunaechst in eine Tabelle in der Maintain-Datenbank eingefuegt, sodass der dort als Trigger hinterlegte SignatureCensor ausgefuehrt wird.
	 * Dieser fuehrt die statische Zugriffskontrolle durch.
	 * 
	 * Diese Methode benutzt den UNflexiblen Trigger, der nur mit solchen Relationen umgehen kann fuer die im Vorfeld spezielle Tabellen angelegt
	 * worden sind. Der Name dieser Tabellen besteht aus einem Prefix (SIGS_QUE_) gefolgt von dem Namen der zu schuetzenden Relation.
	 * 
	 * @param CQE_Query: die in ihre einzelnen Teile zerlegte Anfrage
	 * @param interaction: die vom Benutzer eingegebene Anfrage (Interaktion)
	 * @param columnNames: die Namen der Spalten, die in der jeweilig angefragten Relation vorhanden sind
	 * @return String: ist ein potentielles Geheimnis gefaehrdet, wird mum zureck gegeben, andernfalls die korrekte Antwort, kodiert durch 0 (gilt nicht in der DB) bzw. 1 (gilt in der DB)             
	 * @throws DatabaseException, UnsupportedFormulaException, IllegalFormulaOperationException
	 */
	public String writeToCQEQueryUnflexible( SQLDatabase db, ArrayList<String> CQE_Query, Formula interaction, String[] ColumnNames, String relationname, int userID ) throws DatabaseException, UnsupportedFormulaException {		
		// einzelne Anteile der Anfrage in Tabelle SIGS_QUE_RELATIONNAME einfuegen
		SQLInsert sqlInsert = db.newInsert();
		// UserID wird in mit in die Relation geschrieben, damit anhand dieser nur die relevanten Eintraege aus der CONFIDENTIALITYPOLICY
		// ausgelesen werden
		sqlInsert.set( SignaturesUnflexibleQuerySchema.USER_ID, userID );
		for (int i = 0; i < ColumnNames.length; i++) {
			sqlInsert.set( ColumnNames[i], CQE_Query.get(i) );
		}

		// INSERT auf SIGS_QUE_RELATIONNAME, wodurch der dahinterliegende Trigger ausgeloest wird
		sqlInsert.insert( SignaturesUnflexibleQuerySchema.TABLE_PREFIX + relationname.toUpperCase() );
		
		// vom Trigger erzeugte Antwort auslesen: "mum" = verweigert, 1 = wahr (ungefaehrlich), 0 = falsch (ungefaehrlich)
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select(SignaturesUnflexibleQuerySchema.REACTION);
		sqlSelect.from(SignaturesUnflexibleQuerySchema.TABLE_PREFIX + relationname.toUpperCase());
		SQLTable table = sqlSelect.get();
		String reaction = table.getFirstRow()[0];
		if( reaction == null || ! (table.getFirstRow()[0].equals("mum") || table.getFirstRow()[0].equals("0") || table.getFirstRow()[0].equals("1") ) ) {
			throw new DatabaseException("Illegal reaction value.", null);
		}
		return reaction;
	}
	
	/**
	 * Methode mit welcher die signaturbasierte Zugriffskontrolle realisiert wird.
	 * s. Diplomarbeit: Kontrollierte Anfrageauswertung mit Hilfe von aus Integritaetsbedingungen erzeugten Signaturen (Torsten Schlotmann)
	 * Die gestellte Anfrage eines Benutzers wird durch einen in PL/SQL implementierten Zensor bearbeitet. Zu diesem Zweck wird die Anfrage
	 * zunaechst in eine Tabelle in der Maintain-Datenbank eingefuegt, sodass der dort als Trigger hinterlegte SignatureCensor ausgefuehrt wird.
	 * Dieser fuehrt die statische Zugriffskontrolle durch.
	 * 
	 * Diese Methode benutzt den flexiblen Trigger, der mit beliebigen Relationen umgehen kann ohne das hierfuer im Vorfeld spezielle Tabellen angelegt
	 * werden muessen.
	 * 
	 * @param CQE_Query: die in ihre einzelnen Teile zerlegte Anfrage
	 * @param interaction: die vom Benutzer eingegebene Anfrage (Interaktion)
	 * @param columnNames: die Namen der Spalten, die in der jeweilig angefragten Relation vorhanden sind
	 * @return String: ist ein potentielles Geheimnis gefaehrdet, wird mum zureck gegeben, andernfalls die korrekte Antwort, kodiert durch 0 (gilt nicht in der DB) bzw. 1 (gilt in der DB)             
	 * @throws DatabaseException, UnsupportedFormulaException, IllegalFormulaOperationException
	 */
	public String writeToCQEQueryFlexible( SQLDatabase db, ArrayList<String> CQE_Query, Formula interaction, String[] ColumnNames, String relationname, int userID ) throws DatabaseException, UnsupportedFormulaException {		
		// Attribut-Value-Kombinationen in Datenbank eintragen
		// neue Query-ID generieren
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("SEQUENCE_FLEX_QUERY_ID.NEXTVAL");
		sqlSelect.from("DUAL");
		SQLTable resultQueryID = sqlSelect.get();
		String queryID = resultQueryID.getFirstRow()[0];
		if( queryID == null ) {
			throw new DatabaseException("Generation of new query id failed.", null);
		}
		
		// Attribute mit zugehoerigen Values in Datenbank eintragen
		SQLInsert sqlInsert;
		for (int i = 0; i < ColumnNames.length; ++i) {
			sqlInsert = db.newInsert();
			sqlInsert.set( SignaturesFlexibleQueryAVSchema.QUERY_ID, queryID );
			sqlInsert.set( SignaturesFlexibleQueryAVSchema.ATTRIBUTE, ColumnNames[i] );
			sqlInsert.set( SignaturesFlexibleQueryAVSchema.VALUE, CQE_Query.get(i) );
			sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexibleQueryAV );
		}
		
		// uebergeordneten Datensatz in die Datenbank eintragen
		sqlInsert = db.newInsert();
		sqlInsert.set( SignaturesFlexibleQuerySchema.QUERY_ID, queryID );
		// UserID wird mit in die Relation geschrieben. Noetig fuer die Verwaltung der Flags
		sqlInsert.set( SignaturesFlexibleQuerySchema.USER_ID, userID );
		sqlInsert.set( SignaturesFlexibleQuerySchema.RELATIONNAME, relationname.toUpperCase() );
		// insert ausfuehren -> Trigger auf der Tabelle wird ausgeloest
		sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexibleQuery );
		
		// vom Trigger erzeugte Antwort auslesen: "mum" = verweigert, 1 = wahr (ungefaehrlich), 0 = falsch (ungefaehrlich)
		sqlSelect = db.newSelect();
		sqlSelect.select(SignaturesFlexibleQuerySchema.REACTION).from(Schema.maintenanceDatabase.signaturesFlexibleQuery).where(SignaturesFlexibleQuerySchema.QUERY_ID.name, queryID);
		SQLTable table = sqlSelect.get();
		String reaction = table.getFirstRow()[0];
		if( reaction == null || ! (table.getFirstRow()[0].equals("mum") || table.getFirstRow()[0].equals("0") || table.getFirstRow()[0].equals("1") ) ) {
			throw new DatabaseException("Illegal reaction value.", null);
		}
		return reaction;
	}
	
	
	public void addConfidentialityPolicyItemUnflexible(SQLDatabase db, List<Attribute> attributes, SignatureConfidentialityPolicyItem conf_pol_item) throws DatabaseException {
		// Relation und Spaltennamen ermitteln
		String relationname = conf_pol_item.getRelationname().toUpperCase();
		
		if( attributes.size() != conf_pol_item.getAttributeCount() ) {
			String msg = "Error: Number of attributes of the predicate " + relationname + " doesn't match the number of columns in the database.";
			logger.debug(msg);
			throw new DatabaseException(msg, null);
		}
		
		// Pruefung, ob eine PotSec-Relation, auf die sich die Formel bezieht, fuer den SignatureCensor vorhanden ist
		if( !db.getTables().contains(SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname) ) {
			// keine PotSec-Relation vorhanden -> Formel kann nicht aufgenommen werden
			String msg = "Error: Adding confidentiality policy entry for the UnflexibleSignatureCensor failed! Missing confidentiality policy table for relation " + relationname;
			logger.debug(msg);
			throw new DatabaseException(msg, null);
		}
		
		SQLInsert sqlInsert = db.newInsert();
		sqlInsert.set(SignaturesUnflexiblePotSecSchema.ID, conf_pol_item.getID());
		sqlInsert.set(SignaturesUnflexiblePotSecSchema.USER_ID, conf_pol_item.getUserID());
		int i = 0;
		for( Attribute attribute : attributes ) {
			sqlInsert.set( attribute.getName(), conf_pol_item.getAttributeValue(i) );
			++i;
		}
		sqlInsert.insert(SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname);
	}
	
	public void addConfidentialityPolicyItemFlexible(SQLDatabase db, List<Attribute> attributes, SignatureConfidentialityPolicyItem conf_pol_item) throws DatabaseException {
		// Relation und Spaltennamen ermitteln
		String relationname = conf_pol_item.getRelationname().toUpperCase();
		
		if( attributes.size() != conf_pol_item.getAttributeCount() ) {
			String msg = "Error: Number of attributes of the predicate " + relationname + " doesn't match the number of columns in the database.";
			logger.debug(msg);
			throw new DatabaseException(msg, null);
		}
		
		SQLInsert sqlInsert= db.newInsert();
		sqlInsert.set(SignaturesFlexiblePotSecTableSchema.POT_SEC_ID, conf_pol_item.getID());
		sqlInsert.set(SignaturesFlexiblePotSecTableSchema.RELATIONNAME, conf_pol_item.getRelationname());
		sqlInsert.set(SignaturesFlexiblePotSecTableSchema.USER_ID, conf_pol_item.getUserID());
		sqlInsert.insert(Schema.maintenanceDatabase.signaturesFlexiblePotSec);
		
		int i = 0;
		for( Attribute attribute : attributes ) {
			sqlInsert = db.newInsert();
			sqlInsert.set(SignaturesFlexiblePotSecAVTableSchema.POT_SEC_ID, conf_pol_item.getID());
			sqlInsert.set(SignaturesFlexiblePotSecAVTableSchema.ATTRIBUTE, attribute.getName());
			sqlInsert.set(SignaturesFlexiblePotSecAVTableSchema.VALUE, conf_pol_item.getAttributeValue(i));
			sqlInsert.insert(Schema.maintenanceDatabase.signaturesFlexiblePotSecAV);
			++i;
		}
	}
	
	public void addSignatureUnflexible(SQLDatabase db, List<Attribute> attributes, TemplateDependency td, int userId) throws DatabaseException {
		// Relation und Spaltennamen ermitteln
		String relationname = td.getRelationname();
		
		// ermittle die bisher groesste ID
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + ") as MAX_ID").from(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relationname);
		SQLTable result = sqlSelect.get();
		int nextID;
		String maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID ) + 1;
		}
		
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			SQLInsert sqlInsert = db.newInsert();
			sqlInsert.set(SignaturesUnflexibleSignaturesKrankheitSchema.SIG_ID, nextID);
			sqlInsert.set(SignaturesUnflexibleSignaturesKrankheitSchema.USER_ID, userId);
			sqlInsert.set(SignaturesUnflexibleSignaturesKrankheitSchema.FLAG_IMPL, 0);
			sqlInsert.set(SignaturesUnflexibleSignaturesKrankheitSchema.FLAG_OLD, 0);
			for( int attributeIndex=0; attributeIndex<attributes.size(); ++attributeIndex ) {
				TemplateDependencyElement element = td.getPremise(i).getElement(attributeIndex);
				if( element.isConstant() )
					sqlInsert.set(attributes.get(attributeIndex).getName(), element.getValue());
				else
					sqlInsert.set(attributes.get(attributeIndex).getName(), "*");
			}
			
			sqlInsert.insert(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relationname);
		}
	}
	
	public void addSignatureFlexible(SQLDatabase db, List<Attribute> attributes, TemplateDependency td, int userId) throws DatabaseException {
		// Relation und Spaltennamen ermitteln
		String relationname = td.getRelationname();
				
		// ermittle die bisher groesste ID
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexibleSignaturesSchema.SIG_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
		SQLTable result = sqlSelect.get();
		int nextID;
		String maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID ) + 1;
		}
		
		// neue Signatur anlegen (Signature)
		SQLInsert sqlInsert= db.newInsert();
		sqlInsert.set(SignaturesFlexibleSignaturesSchema.SIG_ID, nextID);
		sqlInsert.set(SignaturesFlexibleSignaturesSchema.USER_ID, userId);
		sqlInsert.set(SignaturesFlexibleSignaturesSchema.RELATIONNAME, relationname);
		sqlInsert.insert(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
		
		// Praemissen der Signatur anlegen (SignatureP und SignaturePAV)
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			// SignatureP
			sqlInsert = db.newInsert();
			sqlInsert.set(SignaturesFlexibleSignaturesPSchema.SIG_ID, nextID);
			sqlInsert.set(SignaturesFlexibleSignaturesPSchema.P_ID, i);
			sqlInsert.set(SignaturesFlexibleSignaturesPSchema.FLAG_IMPL, 0);
			sqlInsert.set(SignaturesFlexibleSignaturesPSchema.FLAG_OLD, 0);
			sqlInsert.insert(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP);
			
			// SignaturePAV
			for( int attributeIndex=0; attributeIndex<attributes.size(); ++attributeIndex ) {
				TemplateDependencyElement element = td.getPremise(i).getElement(attributeIndex);
				sqlInsert = db.newInsert();
				sqlInsert.set(SignaturesFlexibleSignaturesPAVSchema.SIG_ID, nextID);
				sqlInsert.set(SignaturesFlexibleSignaturesPAVSchema.P_ID, i);
				sqlInsert.set(SignaturesFlexibleSignaturesPAVSchema.ATTRIBUTE, attributes.get(attributeIndex).getName());
				if( element.isConstant() )
					sqlInsert.set(SignaturesFlexibleSignaturesPAVSchema.VALUE, element.getValue());
				else
					sqlInsert.set(SignaturesFlexibleSignaturesPAVSchema.VALUE, "*");
				sqlInsert.insert(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV);
			}
		}
	}
	
	public void removeSignaturesCoveringPotentialSecretsUnflexible(SQLDatabase db, List<Attribute> attributes, String relation) throws DatabaseException {
		String columnstr = "";
		for( Attribute attribute : attributes ) {
			if( ! columnstr.equals("") )
				columnstr += ", ";
			columnstr += attribute.getName();
		}
		
		// entferne Template Dependencies, die in einer Praemisse ein potentielles Geheimnis ueberdecken
		String exists_subquery = "SELECT " + columnstr + " FROM " + SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relation + " potsec WHERE ";
		exists_subquery += "sig." + SignaturesUnflexibleSignaturesSchema.USER_ID.name + "=potsec." + SignaturesUnflexiblePotSecSchema.USER_ID.name + " AND ";
		String sep = "";
		for( Attribute attribute : attributes ) {
			exists_subquery += sep + "(potsec." + attribute.getName() + "=sig." + attribute.getName() + " OR potsec." + attribute.getName() + "='*')";
			sep = " AND ";
		}
		String query = "DELETE FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " WHERE SIG_ID IN( SELECT SIG_ID FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig WHERE ( EXISTS(" + exists_subquery + ") ) )";
		db.dataManipulationQuery(query);
	}
	
	public void removeSignaturesCoveringPotentialSecretsFlexible(SQLDatabase db, List<Attribute> attributes, String relation) throws DatabaseException {
		/* Ermittle die IDs der Signaturen, die von potentiellen Geheimnissen ueberdeckt werden
		 * 
		 * Zielquery:
		 * 
		 * SELECT sigs.SIG_ID FROM SIGS_FLEX_SIGNATURES sigs, SIGS_FLEX_SIGNATURES_P sigs_p, SIGS_FLEX_SIGNATURES_PAV sigs_pav1, SIGS_FLEX_SIGNATURES_PAV sigs_pav2, SIGS_FLEX_SIGNATURES_PAV sigs_pav3 WHERE ( 
		 *		sigs.RELATIONNAME = 'SIG_KRANKHEIT'
		 *		AND sigs.SIG_ID = sigs_p.SIG_ID
		 *		AND sigs.SIG_ID = sigs_pav1.SIG_ID
		 *		AND sigs_p.P_ID = sigs_pav1.P_ID
		 *		AND sigs.SIG_ID = sigs_pav2.SIG_ID
		 *		AND sigs_p.P_ID = sigs_pav2.P_ID
		 * 		AND sigs.SIG_ID = sigs_pav3.SIG_ID
		 *		AND sigs_p.P_ID = sigs_pav3.P_ID
		 *		AND EXISTS ( 
		 *		      SELECT 'TRUE' AS BOOLEANVALUE
		 *		      FROM SIGS_FLEX_POT_SEC ps, SIGS_FLEX_POT_SEC_AV potsec1, SIGS_FLEX_POT_SEC_AV potsec2, SIGS_FLEX_POT_SEC_AV potsec3 
		 *		      WHERE 
		 *				ps.RELATIONNAME = sigs.RELATIONNAME 
		 *				AND ps.USER_ID = sigs.USER_ID
		 *				AND ps.POT_SEC_ID = potsec1.POT_SEC_ID 
		 *				AND (potsec1.ATTRIBUTE='SYMPTOM'  AND sigs_pav1.ATTRIBUTE='SYMPTOM'  AND (potsec1.VALUE=sigs_pav1.VALUE OR potsec1.VALUE='*')) 
		 *				AND ps.POT_SEC_ID = potsec2.POT_SEC_ID 			    
		 *				AND (potsec2.ATTRIBUTE='DIAGNOSE' AND sigs_pav2.ATTRIBUTE='DIAGNOSE' AND (potsec2.VALUE=sigs_pav2.VALUE OR potsec2.VALUE='*'))
		 *				AND ps.POT_SEC_ID = potsec3.POT_SEC_ID  
		 *			    AND (potsec3.ATTRIBUTE='PATIENT'  AND sigs_pav3.ATTRIBUTE='PATIENT'  AND (potsec3.VALUE=sigs_pav3.VALUE OR potsec3.VALUE='*'))
		 *		      ) 
		 *	    );
		 */
		String query = "SELECT sigs." + SignaturesFlexibleSignaturesSchema.SIG_ID.name;
		query += " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sigs";
		query += ", " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesP.name + " sigs_p";
		for( int i=1; i<= attributes.size(); ++i ) {
			query += ", " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " sigs_pav" + i;
		}
		query += " WHERE (";
		query += "sigs." + SignaturesFlexibleSignaturesSchema.RELATIONNAME.name + "= '" + relation + "'";
		query += " AND sigs." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + "=sigs_p." + SignaturesFlexibleSignaturesPSchema.SIG_ID.name;
		for( int i=1; i<= attributes.size(); ++i ) {
			query += " AND sigs." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + "=sigs_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name;
			query += " AND sigs_p." + SignaturesFlexibleSignaturesPSchema.P_ID.name + "=sigs_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.P_ID.name;
		}
		query += " AND EXISTS (";
		query += "SELECT 'TRUE' AS BOOLEANVALUE";
		query += " FROM " + Schema.maintenanceDatabase.signaturesFlexiblePotSec.name + " ps";
		for( int i=1; i<= attributes.size(); ++i ) {
			query += ", " + Schema.maintenanceDatabase.signaturesFlexiblePotSecAV.name + " potsec" + i;
		}
		query += " WHERE ps." + SignaturesFlexiblePotSecTableSchema.RELATIONNAME.name + "=sigs." + SignaturesFlexibleSignaturesSchema.RELATIONNAME.name;
		query += " AND ps." + SignaturesFlexiblePotSecTableSchema.USER_ID.name + "=sigs." + SignaturesFlexibleSignaturesSchema.USER_ID.name;
		int i = 1;
		for( Attribute attribute : attributes ) {
			query += " AND ps." + SignaturesFlexiblePotSecTableSchema.POT_SEC_ID.name + "=potsec" + i + "." + SignaturesFlexiblePotSecAVTableSchema.POT_SEC_ID.name;
			query += " AND (";
				query += "potsec" + i + "." + SignaturesFlexiblePotSecAVTableSchema.ATTRIBUTE.name + "='" + attribute.getName() + "'";
				query += " AND sigs_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.ATTRIBUTE.name + "='" + attribute.getName() + "'";
				query += " AND (";
					query += "potsec" + i + "." + SignaturesFlexiblePotSecAVTableSchema.VALUE.name + "=sigs_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.VALUE.name;
					query += " OR potsec" + i + "." + SignaturesFlexiblePotSecAVTableSchema.VALUE.name + "='*'";
					query += ")";
			query += ")";
			++i;
		}
		query += ")";
		query += ")";
		
		SQLTable result = db.query(query);
		
		if( result.getRowCount() > 0 ) {
			// Loesche alle ueberdeckten Signaturen
			String whereQuery =  " WHERE " + SignaturesFlexibleSignaturesSchema.SIG_ID.name + " IN (";
			String sep = "";
			Iterator<String[]> iterator = result.iterator();
			while( iterator.hasNext() ) {
				whereQuery += sep + iterator.next()[0];
				sep = ",";
			}
			whereQuery += ")";
			
			// Signature
			String deleteQuery = "DELETE FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + whereQuery;
			db.dataManipulationQuery(deleteQuery);
			// SignatureP
			deleteQuery = "DELETE FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesP.name + whereQuery;
			db.dataManipulationQuery(deleteQuery);
			// SignaturePAV
			deleteQuery = "DELETE FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + whereQuery;
			db.dataManipulationQuery(deleteQuery);
		}
	}
	
	public void removeSubsumedDependenciesUnflexible(SQLDatabase db, List<Attribute> attributes, String relation) throws DatabaseException {
		/* Entferne aequivalente Signaturen (bis auf eine)
		 * 
		 * Zielquery:
		 * 
		 * SELECT DISTINCT sig2.SIG_ID FROM SIGS_SIG_SIG_KRANKHEIT sig1, SIGS_SIG_SIG_KRANKHEIT sig2 
		 *	WHERE sig1.SIG_ID < sig2.SIG_ID 
		 *	    AND sig1.USER_ID = sig2.USER_ID
		 *	    AND NOT EXISTS (
		 *	      (
		 *	        SELECT sig1_values.SYMPTOM AS v1, sig1_values.DIAGNOSIS AS v2, sig1_values.PATIENT AS v3
		 *	        FROM SIGS_SIG_SIG_KRANKHEIT sig1_values
		 *	        WHERE 
		 *	          sig1.SIG_ID = sig1_values.SIG_ID
		 *	      )
		 *	      MINUS
		 *	      (
		 *	        SELECT sig2_values.SYMPTOM AS v1, sig2_values.DIAGNOSIS AS v2, sig2_values.PATIENT AS v3
		 *	        FROM SIGS_SIG_SIG_KRANKHEIT sig2_values
		 *	        WHERE 
		 *            sig2.SIG_ID = sig2_values.SIG_ID
		 *	      )
		 *	    )
		 *      AND NOT EXISTS (
		 *	      (
		 *	        SELECT sig2_values.SYMPTOM AS v1, sig2_values.DIAGNOSIS AS v2, sig2_values.PATIENT AS v3
		 *	        FROM SIGS_SIG_SIG_KRANKHEIT sig2_values
		 *	        WHERE 
		 *           sig2.SIG_ID = sig2_values.SIG_ID
		 *	      )
		 *	      MINUS
		 *	      (
		 *	        SELECT sig1_values.SYMPTOM AS v1, sig1_values.DIAGNOSIS AS v2, sig1_values.PATIENT AS v3
		 *	        FROM SIGS_SIG_SIG_KRANKHEIT sig1_values
		 *	        WHERE 
		 *	          sig1.SIG_ID = sig1_values.SIG_ID
		 *	      )
		 *	    )
		 *	;
		 */
		
		String minus_query_sig1 = "";
		String minus_query_sig2 = "";
		for( int sigIndex=1; sigIndex<=2; ++sigIndex ) {
			String select = "SELECT ";
			String from = " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig" + sigIndex + "_values";
			String where = " WHERE sig" + sigIndex + "." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + "=sig" + sigIndex + "_values." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name;
			String sep = "";
			for( int i=0; i<attributes.size(); ++i ) {
				select += sep + "sig" + sigIndex + "_values." + attributes.get(i).getName() + " as v" + i;
				sep = ", ";
			}
			String minus_query = select + from + where;
			if( sigIndex==1 )
				minus_query_sig1 = minus_query;
			else
				minus_query_sig2 = minus_query;
		}
		
		String query = "SELECT DISTINCT sig2." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name;
		query += " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig1, " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig2";
		query += " WHERE sig1." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + "<sig2." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name;
		query += " AND sig1." + SignaturesUnflexibleSignaturesSchema.USER_ID.name + "=sig2." + SignaturesUnflexibleSignaturesSchema.USER_ID.name;
		query += " AND NOT EXISTS (";
			query += "(";
				query += minus_query_sig2;
			query += ")";
			query += "MINUS";
			query += "(";
				query += minus_query_sig1;
			query += ")";
		query += ")";
		query += " AND NOT EXISTS (";
			query += "(";
				query += minus_query_sig1;
			query += ")";
			query += "MINUS";
			query += "(";
				query += minus_query_sig2;
			query += ")";
		query += ")";
		
		// Ermittle alle IDs von Signaturen, die aufgrund einer weiteren aequivalenten Signatur geloescht werden koennen
		SQLTable result = db.query(query);
		// Loesche diese Signaturen
		if( result.getRowCount() > 0 ) {
			SQLDelete sqlDelete = db.newDelete();
			for( Iterator<String[]> iterator = result.iterator(); iterator.hasNext(); ) {
				int sigID = Integer.parseInt( iterator.next()[0] );
				sqlDelete.or_where(SignaturesUnflexibleSignaturesSchema.SIG_ID.name, sigID);
			}
			sqlDelete.delete(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation);
		}
		
		/* Entferne Signaturen, die andere Signaturen vollstaendig enthalten
		 * 
		 * Zielquery:
		 * 
		 * SELECT DISTINCT sig1.SIG_ID FROM SIGS_SIG_SIG_KRANKHEIT sig1 
		 *	WHERE 
		 *     EXISTS (
		 *       SELECT sig1.SIG_ID, sig2.SIG_ID 
		 *       FROM SIGS_SIG_SIG_KRANKHEIT sig2
		 *       WHERE 
		 *             sig1.SIG_ID != sig2.SIG_ID 
		 *         AND sig1.USER_ID = sig2.USER_ID
		 *         AND NOT EXISTS (
		 *           SELECT sig1.SIG_ID, sig2.SIG_ID, sig3.SIG_ID
		 *           FROM SIGS_SIG_SIG_KRANKHEIT sig3
		 *           WHERE
		 *                 sig2.SIG_ID = sig3.SIG_ID
		 *             AND NOT EXISTS (
		 *               SELECT sig1.SIG_ID, sig2.SIG_ID, sig3.SIG_ID, sig4.SIG_ID
		 *               FROM SIGS_SIG_SIG_KRANKHEIT sig4
		 *               WHERE
		 *                     sig1.SIG_ID = sig4.SIG_ID
		 *                 AND (sig3.SYMPTOM = sig4.SYMPTOM OR sig3.SYMPTOM = '*')
		 *                 AND (sig3.DIAGNOSIS = sig4.DIAGNOSIS OR sig3.DIAGNOSIS = '*')
		 *                 AND (sig3.PATIENT = sig4.PATIENT OR sig3.PATIENT = '*')             
		 *             )
		 *         )
		 *     )
		 */
		String columnSigID = SignaturesUnflexibleSignaturesSchema.SIG_ID.name;
		String columnUserID = SignaturesUnflexibleSignaturesSchema.USER_ID.name;
		
		String querySig4 = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID + ", sig3." + columnSigID + ", sig4." + columnSigID;
		querySig4 += " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig4";
		querySig4 += " WHERE sig1." + columnSigID + "=sig4." + columnSigID;
		for( int i=0; i<attributes.size(); ++i ) {
			querySig4 += " AND (sig3." + attributes.get(i).getName() + "=sig4." + attributes.get(i).getName() + " OR sig3." + attributes.get(i).getName() + "='*')";
		}
		
		String querySig3 = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID + ", sig3." + columnSigID;
		querySig3 += " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig3";
		querySig3 += " WHERE sig2." + columnSigID + "=sig3." + columnSigID;
		querySig3 += " AND NOT EXISTS (";
			querySig3 += querySig4;
		querySig3 += ")";
		
		String querySig2 = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID;
		querySig2 += " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig2";
		querySig2 += " WHERE sig1." + columnSigID + "!=sig2." + columnSigID;
		querySig2 += " AND sig1." + columnUserID + "=sig2." + columnUserID;
		querySig2 += " AND NOT EXISTS (";
			querySig2 += querySig3;
		querySig2 += ")";
		
		query = "SELECT DISTINCT sig1." + columnSigID;
		query += " FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " sig1";
		query += " WHERE EXISTS (";
			query += querySig2;
		query += ")";
		
		// Ermittle alle IDs von Signaturen, die aufgrund des Enthaltens einer weiteren Signatur geloescht werden koennen
		result = db.query(query);

		// Loesche diese Signaturen
		if( result.getRowCount() > 0 ) {
			SQLDelete sqlDelete = db.newDelete();
			for( Iterator<String[]> iterator = result.iterator(); iterator.hasNext(); ) {
				int sigID = Integer.parseInt( iterator.next()[0] );
				sqlDelete.or_where(SignaturesUnflexibleSignaturesSchema.SIG_ID.name, sigID);
			}
			sqlDelete.delete(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation);
		}
	}
	
	/**
	 * Entfernt diejenigen Signaturen, die in anderen Signaturen vollstaendig enthalten sind.
	 * 
	 * @param relation
	 * @throws DatabaseException
	 */
	public void removeSubsumedDependenciesFlexible(SQLDatabase db, List<Attribute> attributes, String relation) throws DatabaseException {
		/* Entferne aequivalente Signaturen (bis auf eine)
		 * 
		 * Zielquery:
		 * 
		 * SELECT DISTINCT sig2.SIG_ID FROM SIGS_FLEX_SIGNATURES sig1, SIGS_FLEX_SIGNATURES sig2 
		 *	WHERE sig1.SIG_ID < sig2.SIG_ID 
		 *	    AND sig1.RELATIONNAME = sig2.RELATIONNAME
		 *	    AND sig1.RELATIONNAME = 'SIG_KRANKHEIT'
		 *	    AND sig1.USER_ID = sig2.USER_ID
		 *	    AND NOT EXISTS (
		 *	      (
		 *	        SELECT sig2_pav1.VALUE AS v1, sig2_pav2.VALUE AS v2, sig2_pav3.VALUE AS v3
		 *	        FROM SIGS_FLEX_SIGNATURES_PAV sig2_pav1, SIGS_FLEX_SIGNATURES_PAV sig2_pav2, SIGS_FLEX_SIGNATURES_PAV sig2_pav3
		 *	        WHERE 
		 *	          sig2.SIG_ID = sig2_pav1.SIG_ID
		 *	          AND sig2.SIG_ID = sig2_pav2.SIG_ID
		 *	          AND sig2.SIG_ID = sig2_pav3.SIG_ID
		 *	          AND sig2_pav1.P_ID = sig2_pav2.P_ID
		 *	          AND sig2_pav1.P_ID = sig2_pav3.P_ID
		 *	          AND sig2_pav1.ATTRIBUTE = 'SYMPTOM'
		 *	          AND sig2_pav2.ATTRIBUTE = 'DIAGNOSE'
		 *	          AND sig2_pav3.ATTRIBUTE = 'PATIENT'
		 *	      )
		 *	      MINUS
		 *	      (
		 *	        SELECT sig1_pav1.VALUE AS v1, sig1_pav2.VALUE AS v2, sig1_pav3.VALUE AS v3
		 *	        FROM SIGS_FLEX_SIGNATURES_PAV sig1_pav1, SIGS_FLEX_SIGNATURES_PAV sig1_pav2, SIGS_FLEX_SIGNATURES_PAV sig1_pav3
		 *	        WHERE 
		 *	          sig1.SIG_ID = sig1_pav1.SIG_ID
		 *	          AND sig1.SIG_ID = sig1_pav2.SIG_ID
		 *	          AND sig1.SIG_ID = sig1_pav3.SIG_ID
		 *	          AND sig1_pav1.P_ID = sig1_pav2.P_ID
		 *	          AND sig1_pav1.P_ID = sig1_pav3.P_ID
		 *	          AND sig1_pav1.ATTRIBUTE = 'SYMPTOM'
		 *	          AND sig1_pav2.ATTRIBUTE = 'DIAGNOSE'
		 *	          AND sig1_pav3.ATTRIBUTE = 'PATIENT'
		 *	      )
		 *	    )
		 *	    AND NOT EXISTS (
		 *	      (
		 *	        SELECT sig1_pav1.VALUE AS v1, sig1_pav2.VALUE AS v2, sig1_pav3.VALUE AS v3
		 *	        FROM SIGS_FLEX_SIGNATURES_PAV sig1_pav1, SIGS_FLEX_SIGNATURES_PAV sig1_pav2, SIGS_FLEX_SIGNATURES_PAV sig1_pav3
		 *	        WHERE 
		 *	          sig1.SIG_ID = sig1_pav1.SIG_ID
		 *	          AND sig1.SIG_ID = sig1_pav2.SIG_ID
		 *	          AND sig1.SIG_ID = sig1_pav3.SIG_ID
		 *	          AND sig1_pav1.P_ID = sig1_pav2.P_ID
		 *	          AND sig1_pav1.P_ID = sig1_pav3.P_ID
		 *	          AND sig1_pav1.ATTRIBUTE = 'SYMPTOM'
		 *	          AND sig1_pav2.ATTRIBUTE = 'DIAGNOSE'
		 *	          AND sig1_pav3.ATTRIBUTE = 'PATIENT'
		 *	      )
		 *	      MINUS
		 *	      (
		 *	        SELECT sig2_pav1.VALUE AS v1, sig2_pav2.VALUE AS v2, sig2_pav3.VALUE AS v3
		 *	        FROM SIGS_FLEX_SIGNATURES_PAV sig2_pav1, SIGS_FLEX_SIGNATURES_PAV sig2_pav2, SIGS_FLEX_SIGNATURES_PAV sig2_pav3
		 *	        WHERE 
		 *	          sig2.SIG_ID = sig2_pav1.SIG_ID
		 *	          AND sig2.SIG_ID = sig2_pav2.SIG_ID
		 *	          AND sig2.SIG_ID = sig2_pav3.SIG_ID
		 *	          AND sig2_pav1.P_ID = sig2_pav2.P_ID
		 *	          AND sig2_pav1.P_ID = sig2_pav3.P_ID
		 *	          AND sig2_pav1.ATTRIBUTE = 'SYMPTOM'
		 *	          AND sig2_pav2.ATTRIBUTE = 'DIAGNOSE'
		 *	          AND sig2_pav3.ATTRIBUTE = 'PATIENT'
		 *	      )
		 *	    )
		 *	;
		 */
		
		String minus_query_sig1 = "";
		String minus_query_sig2 = "";
		for( int sigIndex=1; sigIndex<=2; ++sigIndex ) {
			String select = "SELECT ";
			String from = " FROM ";
			String where = " WHERE 1=1 ";
			String sep = "";
			for( int i=0; i<attributes.size(); ++i ) {
				select += sep + "sig" + sigIndex + "_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.VALUE.name + " AS v" + i;
				from += sep + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " sig" + sigIndex + "_pav" + i;
				where += " AND sig" + sigIndex + "." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + "=sig" + sigIndex + "_pav" + i + "." + SignaturesFlexibleSignaturesSchema.SIG_ID.name;
				where += " AND sig" + sigIndex + "_pav1." + SignaturesFlexibleSignaturesPAVSchema.P_ID.name + "=sig" + sigIndex + "_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.P_ID.name;
				where += " AND sig" + sigIndex + "_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.ATTRIBUTE.name + "='" + attributes.get(i).getName() + "'";
				sep = ", ";
			}
			String minus_query = select + from + where; 
			if( sigIndex==1 )
				minus_query_sig1 = minus_query;
			else
				minus_query_sig2 = minus_query;
		}
		
		String query = "SELECT DISTINCT sig2." + SignaturesFlexibleSignaturesSchema.SIG_ID.name;
		query += " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig1, " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig2";
		query += " WHERE sig1." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + "<sig2." + SignaturesFlexibleSignaturesSchema.SIG_ID.name;
		query += " AND sig1." + SignaturesFlexibleSignaturesSchema.RELATIONNAME.name + "=sig2." + SignaturesFlexibleSignaturesSchema.RELATIONNAME.name;
		query += " AND sig1." + SignaturesFlexibleSignaturesSchema.RELATIONNAME.name + "='" + relation + "'";
		query += " AND sig1." + SignaturesFlexibleSignaturesSchema.USER_ID.name + "=sig2." + SignaturesFlexibleSignaturesSchema.USER_ID.name;
		query += " AND NOT EXISTS (";
			query += "(";
				query += minus_query_sig2;
			query += ")";
			query += "MINUS";
			query += "(";
				query += minus_query_sig1;
			query += ")";
		query += ")";
		query += " AND NOT EXISTS (";
			query += "(";
				query += minus_query_sig1;
			query += ")";
			query += "MINUS";
			query += "(";
				query += minus_query_sig2;
			query += ")";
		query += ")";
		
		// Ermittle alle IDs von Signaturen, die aufgrund einer weiteren aequivalenten Signatur geloescht werden koennen
		SQLTable result = db.query(query);
		// Loesche diese Signaturen
		if( result.getRowCount() > 0 ) {
			SQLDelete sqlDeleteSignatures = db.newDelete();
			SQLDelete sqlDeleteSignaturesP = db.newDelete();
			SQLDelete sqlDeleteSignaturesPAV = db.newDelete();
			for( Iterator<String[]> iterator = result.iterator(); iterator.hasNext(); ) {
				int sigID = Integer.parseInt( iterator.next()[0] );
				// Signatures
				sqlDeleteSignatures.or_where(SignaturesFlexibleSignaturesSchema.SIG_ID, sigID);
				// SignaturesP
				sqlDeleteSignaturesP.or_where(SignaturesFlexibleSignaturesPSchema.SIG_ID, sigID);
				// SignaturesPAV
				sqlDeleteSignaturesPAV.or_where(SignaturesFlexibleSignaturesPAVSchema.SIG_ID, sigID);
			}
			sqlDeleteSignatures.delete(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
			sqlDeleteSignaturesP.delete(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP);
			sqlDeleteSignaturesPAV.delete(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV);
		}
		
		/* Entferne Signaturen, die andere Signaturen vollstaendig enthalten
		 * 
		 * Zielquery:
		 * 
		 * SELECT DISTINCT sig1.SIG_ID FROM SIGS_FLEX_SIGNATURES sig1 
		 *	WHERE 
		 *    EXISTS (
		 *      SELECT sig1.SIG_ID, sig2.SIG_ID 
		 *      FROM SIGS_FLEX_SIGNATURES sig2
		 *      WHERE 
		 *            sig1.SIG_ID != sig2.SIG_ID 
		 *        AND sig1.USER_ID = sig2.USER_ID
		 *        AND NOT EXISTS (
		 *          SELECT sig1.SIG_ID, sig2.SIG_ID, sig3.SIG_ID
		 *          FROM SIGS_FLEX_SIGNATURES sig3, SIGS_FLEX_SIGNATURES_PAV sig3_pav1, SIGS_FLEX_SIGNATURES_PAV sig3_pav2, SIGS_FLEX_SIGNATURES_PAV sig3_pav3
		 *          WHERE
		 *               sig2.SIG_ID = sig3.SIG_ID
		 *           AND sig3.SIG_ID = sig3_pav1.SIG_ID
		 *           AND sig3.SIG_ID = sig3_pav2.SIG_ID
		 *           AND sig3.SIG_ID = sig3_pav3.SIG_ID
		 *           AND sig3_pav1.P_ID = sig3_pav2.P_ID
		 *           AND sig3_pav1.P_ID = sig3_pav3.P_ID
		 *           AND sig3_pav1.ATTRIBUTE = 'SYMPTOM'
		 *           AND sig3_pav2.ATTRIBUTE = 'DIAGNOSIS'
		 *           AND sig3_pav3.ATTRIBUTE = 'PATIENT'
		 *           AND NOT EXISTS (
		 *              SELECT sig1.SIG_ID, sig2.SIG_ID, sig3.SIG_ID, sig4.SIG_ID
		 *              FROM SIGS_FLEX_SIGNATURES sig4, SIGS_FLEX_SIGNATURES_PAV sig4_pav1, SIGS_FLEX_SIGNATURES_PAV sig4_pav2, SIGS_FLEX_SIGNATURES_PAV sig4_pav3
		 *              WHERE
		 *                    sig1.SIG_ID = sig4.SIG_ID
		 *                AND sig4.SIG_ID = sig4_pav1.SIG_ID
		 *                AND sig4.SIG_ID = sig4_pav2.SIG_ID
		 *                AND sig4.SIG_ID = sig4_pav3.SIG_ID
		 *                AND sig4_pav1.P_ID = sig4_pav2.P_ID
		 *                AND sig4_pav1.P_ID = sig4_pav3.P_ID
		 *                AND sig4_pav1.ATTRIBUTE = 'SYMPTOM'
		 *                AND sig4_pav2.ATTRIBUTE = 'DIAGNOSIS'
		 *                AND sig4_pav3.ATTRIBUTE = 'PATIENT'
		 *                AND (sig3_pav1.VALUE = sig4_pav1.VALUE OR sig3_pav1.VALUE = '*')
		 *                AND (sig3_pav2.VALUE = sig4_pav2.VALUE OR sig3_pav2.VALUE = '*')
		 *                AND (sig3_pav3.VALUE = sig4_pav3.VALUE OR sig3_pav3.VALUE = '*')          
		 *           )
		 *        )
		 *    );
		 */
		
		String columnSigID = SignaturesFlexibleSignaturesSchema.SIG_ID.name;
		String columnUserID = SignaturesFlexibleSignaturesSchema.USER_ID.name;
		String columnPAVSigID = SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name;
		String columnPAVPID = SignaturesFlexibleSignaturesPAVSchema.P_ID.name;
		String columnPAVVALUE = SignaturesFlexibleSignaturesPAVSchema.VALUE.name;
		
		String querySig4Select = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID + ", sig3." + columnSigID + ", sig4." + columnSigID;
		String querySig4From = " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig4";
		String querySig4Where = " WHERE sig1." + columnSigID + "=sig4." + columnSigID;
		for( int i=0; i<attributes.size(); ++i ) {
			querySig4From += ", " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " sig4_pav" + i;
			querySig4Where += " AND sig4." + columnSigID + "=sig4_pav" + i + "." + columnPAVSigID;
			if( i > 0)
				querySig4Where += " AND sig4_pav0." + columnPAVPID + "=sig4_pav" + i + "." + columnPAVPID;
			querySig4Where += " AND sig4_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.ATTRIBUTE.name + "='" + attributes.get(i).getName() + "'";
			querySig4Where += " AND (sig3_pav" + i + "." + columnPAVVALUE + "=sig4_pav" + i + "." + columnPAVVALUE + " OR sig3_pav" + i + "." + columnPAVVALUE + "='*')";
		}
		String querySig4 = querySig4Select + querySig4From + querySig4Where;
		
		String querySig3Select = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID + ", sig3." + columnSigID;
		String querySig3From = " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig3";
		String querySig3Where = " WHERE sig2." + columnSigID + "=sig3." + columnSigID;
		for( int i=0; i<attributes.size(); ++i ) {
			querySig3From += ", " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " sig3_pav" + i;
			querySig3Where += " AND sig3." + columnSigID + "=sig3_pav" + i + "." + columnPAVSigID;
			if( i > 0)
				querySig3Where += " AND sig3_pav0." + columnPAVPID + "=sig3_pav" + i + "." + columnPAVPID;
			querySig3Where += " AND sig3_pav" + i + "." + SignaturesFlexibleSignaturesPAVSchema.ATTRIBUTE.name + "='" + attributes.get(i).getName() + "'";
		}
		String querySig3 = querySig3Select + querySig3From + querySig3Where;
		querySig3 += " AND NOT EXISTS (";
			querySig3 += querySig4;
		querySig3 += ")";
		
		String querySig2 = "SELECT sig1." + columnSigID + ", sig2." +  columnSigID;
		querySig2 += " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig2";
		querySig2 += " WHERE sig1." + columnSigID + "!=sig2." + columnSigID;
		querySig2 += " AND sig1." + columnUserID + "=sig2." + columnUserID;
		querySig2 += " AND NOT EXISTS (";
			querySig2 += querySig3;
		querySig2 += ")";
		
		query = "SELECT DISTINCT sig1." + columnSigID;
		query += " FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " sig1";
		query += " WHERE EXISTS (";
			query += querySig2;
		query += ")";
		
		// Ermittle alle IDs von Signaturen, die aufgrund des Enthaltens einer weiteren Signatur geloescht werden koennen
		result = db.query(query);

		// Loesche diese Signaturen
		if( result.getRowCount() > 0 ) {
			SQLDelete sqlDeleteSignatures = db.newDelete();
			SQLDelete sqlDeleteSignaturesP = db.newDelete();
			SQLDelete sqlDeleteSignaturesPAV = db.newDelete();
			for( Iterator<String[]> iterator = result.iterator(); iterator.hasNext(); ) {
				int sigID = Integer.parseInt( iterator.next()[0] );
				// Signatures
				sqlDeleteSignatures.or_where(SignaturesFlexibleSignaturesSchema.SIG_ID, sigID);
				// SignaturesP
				sqlDeleteSignaturesP.or_where(SignaturesFlexibleSignaturesPSchema.SIG_ID, sigID);
				// SignaturesPAV
				sqlDeleteSignaturesPAV.or_where(SignaturesFlexibleSignaturesPAVSchema.SIG_ID, sigID);
			}
			sqlDeleteSignatures.delete(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
			sqlDeleteSignaturesP.delete(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP);
			sqlDeleteSignaturesPAV.delete(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV);
		}
	}
	
	public void removeSignatureIndexGapsUnflexible(SQLDatabase db, String relation) throws DatabaseException {
		String query = "UPDATE " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " S1 SET " + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + " = ( SELECT COUNT(DISTINCT " + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + ") + 1 FROM " + SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation + " S2 WHERE S2." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + " < S1." + SignaturesUnflexibleSignaturesSchema.SIG_ID.name +")";
		db.dataManipulationQuery(query);		
	}
	
	public void removeSignatureIndexGapsFlexible(SQLDatabase db) throws DatabaseException {
		// Signatures
		String query = "UPDATE " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " S1 SET " + SignaturesFlexibleSignaturesSchema.SIG_ID.name + " = ( SELECT COUNT(DISTINCT " + SignaturesFlexibleSignaturesSchema.SIG_ID.name + ") + 1 FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignatures.name + " S2 WHERE S2." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + " < S1." + SignaturesFlexibleSignaturesSchema.SIG_ID.name + " )";
		db.dataManipulationQuery(query);
		// SignaturesP
		query = "UPDATE " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesP.name + " S1 SET " + SignaturesFlexibleSignaturesPSchema.SIG_ID.name + " = ( SELECT COUNT(DISTINCT " + SignaturesFlexibleSignaturesPSchema.SIG_ID.name + ") + 1 FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesP.name + " S2 WHERE S2." + SignaturesFlexibleSignaturesPSchema.SIG_ID.name + " < S1." + SignaturesFlexibleSignaturesPSchema.SIG_ID.name + " )";
		db.dataManipulationQuery(query);
		// SignaturesPAV
		query = "UPDATE " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " S1 SET " + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name + " = ( SELECT COUNT(DISTINCT " + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name + ") + 1 FROM " + Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name + " S2 WHERE S2." + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name + " < S1." + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name + " )";
		db.dataManipulationQuery(query);
	}
	
	public void clearFlagsUnflexible(SQLDatabase db, String relation, int userID) throws DatabaseException {
		SQLUpdate updateSQL = db.newUpdate();
		updateSQL.set(SignaturesUnflexibleSignaturesSchema.FLAG_IMPL, 0);
		updateSQL.set(SignaturesUnflexibleSignaturesSchema.FLAG_OLD, 0);
		updateSQL.where(SignaturesUnflexibleSignaturesSchema.USER_ID.name, userID);
		updateSQL.update(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relation);
	}
	
	public void clearFlagsFlexible(SQLDatabase db, String relation, int userID) throws DatabaseException {
		// Ermittle Signaturen (IDs), die aktualisiert werden sollen
		SQLSelect sql = db.newSelect();
		sql.select(SignaturesFlexibleSignaturesSchema.SIG_ID);
		sql.from(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
		sql.where(SignaturesFlexibleSignaturesSchema.RELATIONNAME.name, relation);
		sql.where(SignaturesFlexibleSignaturesSchema.USER_ID.name, userID);
		SQLTable result = sql.get();
		
		// update Signaturen
		SQLUpdate updateSQL = db.newUpdate();
		updateSQL.set(SignaturesFlexibleSignaturesPSchema.FLAG_IMPL, 0);
		updateSQL.set(SignaturesFlexibleSignaturesPSchema.FLAG_OLD, 0);
		String where = SignaturesFlexibleSignaturesPSchema.SIG_ID.name + " IN (";
		String sep = "";
		for( Iterator<String[]> iterator = result.iterator(); iterator.hasNext(); ) {
			int sigID = Integer.parseInt( iterator.next()[0] );
			where += sep + sigID;
			sep = ",";
		}
		where += ")";
		updateSQL.where(where);
		updateSQL.update(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP);
	}
	
	public String getDatabaseProcessingTimeUnflexible(SQLDatabase db, String relation) throws DatabaseException {
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select(SignaturesUnflexibleQuerySchema.PROCESSING_TIME);
		sqlSelect.from(SignaturesUnflexibleQuerySchema.TABLE_PREFIX + relation);
		SQLTable result = sqlSelect.get();
		
		return result.getFirstRow()[0];
	}
	
	public String getDatabaseProcessingTimeFlexible(SQLDatabase db) throws DatabaseException {
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select(SignaturesFlexibleQuerySchema.PROCESSING_TIME);
		sqlSelect.from(Schema.maintenanceDatabase.signaturesFlexibleQuery);
		String where_subquery = SignaturesFlexibleQuerySchema.QUERY_ID.name + "=(";
		where_subquery += "SELECT MAX(" + SignaturesFlexibleQuerySchema.QUERY_ID.name + ")";
		where_subquery += " FROM " + Schema.maintenanceDatabase.signaturesFlexibleQuery.name;
		where_subquery += ")";
		sqlSelect.where(where_subquery);
		SQLTable result = sqlSelect.get();
		
		return result.getFirstRow()[0];
	}
	
	public void experimentEvaluation_writeQueryExecutionStatistics(SQLDatabase db, int runID, int queryID, String query, String result, long processingTime, int databaseProcessingTime) throws DatabaseException {
		SQLInsert sqlInsert = db.newInsert();
		sqlInsert.set(SignaturesExperimentEvaluationSchema.RUN_ID, runID);
		sqlInsert.set(SignaturesExperimentEvaluationSchema.QUERY_ID, queryID);
		sqlInsert.set(SignaturesExperimentEvaluationSchema.QUERY, query);
		sqlInsert.set(SignaturesExperimentEvaluationSchema.RESULT, result);
		sqlInsert.set(SignaturesExperimentEvaluationSchema.PROCESSING_TIME, String.valueOf(processingTime));
		sqlInsert.set(SignaturesExperimentEvaluationSchema.DATABASE_PROCESSING_TIME, String.valueOf(databaseProcessingTime));
		sqlInsert.insert(SignaturesExperimentEvaluationSchema.TABLE_NAME);
	}
	
	public void experimentEvaluation_writeQueryExecutionStatisticsAccumulated(SQLDatabase db, int runID, int queryID, String query, String result, long processingTime, int databaseProcessingTime) throws DatabaseException {
		// Pruefe, ob es der erste Eintrag fuer den Durchlauf ist und lege ggfs. einen neuen Datensatz an
		SQLSelect sqlSelect= db.newSelect();
		sqlSelect.select(SignaturesExperimentEvaluationCumulatedSchema.RUN_ID);
		sqlSelect.from(SignaturesExperimentEvaluationCumulatedSchema.TABLE_NAME);
		sqlSelect.where(SignaturesExperimentEvaluationCumulatedSchema.RUN_ID, runID);
		SQLTable tableResult = sqlSelect.get();
		if( tableResult == null ) {
			throw new DatabaseException("Invalid result.", null);
		}
		else if( tableResult.getRowCount() == 0 ) {
			// es existiert noch kein Datensatz fuer den Durchlauf -> anlegen
			SQLInsert sqlInsert = db.newInsert();
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.RUN_ID, runID);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.QUERY_COUNT, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.TRUE_COUNT, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.FALSE_COUNT, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.MUM_COUNT, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_MAX, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_SUM, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_MAX, 0);
			sqlInsert.set(SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_SUM, 0);
			sqlInsert.insert(SignaturesExperimentEvaluationCumulatedSchema.TABLE_NAME);
		}
		
		// SQL-Befehl zum Updaten manuell aufbauen, da Datenbank-Abstraktion nicht mit Set-Bedingungen der Form "COLUM = COLUM + 1" umgehen kann (generiert ungueltige Prepared Statements)
		String sqlUpdate = "UPDATE " + SignaturesExperimentEvaluationCumulatedSchema.TABLE_NAME + " SET ";
		sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.QUERY_COUNT.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.QUERY_COUNT.name + "+1, ";
		if( result.equals("true") )
			sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.TRUE_COUNT.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.TRUE_COUNT.name + "+1, ";
		if( result.equals("false") )
			sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.FALSE_COUNT.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.FALSE_COUNT.name + "+1, ";
		if( result.equals("mum") )
			sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.MUM_COUNT.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.MUM_COUNT.name + "+1, ";
		sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_MAX.name + "=GREATEST(" + SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_MAX.name + ", " + processingTime + "), ";
		sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_SUM.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.PROCESSING_TIME_SUM.name + "+" + processingTime + ", ";
		sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_MAX.name + "=GREATEST(" + SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_MAX.name + ", " + databaseProcessingTime + "), ";
		sqlUpdate += SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_SUM.name + "=" + SignaturesExperimentEvaluationCumulatedSchema.DATABASE_PROCESSING_TIME_SUM.name + "+" + databaseProcessingTime;
		sqlUpdate += " WHERE " + SignaturesExperimentEvaluationCumulatedSchema.RUN_ID.name + "=" + runID;
		
		db.dataManipulationQuery(sqlUpdate);
	}
}
