package server.censor.signature.experiment;

import java.util.LinkedList;
import java.util.List;

import server.censor.signature.util.DistinguishedSymbolElement;
import server.censor.signature.util.NonDistinguishedSymbolElement;
import server.censor.signature.util.SignatureConfidentialityPolicyItem;
import server.censor.signature.util.TemplateDependency;
import server.censor.signature.util.TemplateDependencyRow;
import server.database.AppDatabase;
import server.database.DatabaseConfiguration;
import server.database.NewMaintenanceDatabase;
import server.parser.Formula;
import server.util.Tuple;
import exception.DatabaseException;
import exception.ParserException;
import exception.UnsupportedFormulaException;



/**
 *  Diese Klasse dient der Vorbereitung des Experiments zum SignatureCensor.
 *  Das Experiment ist in "Signature-Based Inference-Usability Confinement for Relational 
 *  Databases under Functional and Join Dependencies" von Biskup etc. beschrieben.
 *  Details des Experimentaufbaus koennen der Datei 'Experimentbeschreibung_DBSEC12.txt' 
 *  entnommen werden.
 */
public class ExperimentPreparationDBSEC12 extends ExperimentPreparation {

	/** Name der zu schuetzenden Relation */
	protected static final String RELATION_NAME = "SIG_KRANKHEIT";
	

	public ExperimentPreparationDBSEC12(NewMaintenanceDatabase maintenanceDB, AppDatabase appDB,	boolean flexible, int userID) throws DatabaseException {
		super(maintenanceDB, appDB, flexible, userID, RELATION_NAME);
	}
	
	public ExperimentPreparationDBSEC12(Tuple<DatabaseConfiguration,DatabaseConfiguration> dbConfig, boolean flexible, int userID) throws DatabaseException {
		super(dbConfig, flexible, userID, RELATION_NAME);
	}
	
	/**
	 * Diese Methode implementiert Schritt 1 des Experiment-Aufbaus:
	 * 
	 * Als generische Signaturen (forbidden structures) 
	 * TD_i, i= 1,...,7 
	 * werden die basic implications aus Abbildung 4 und 6
	 * verwendet, also
	 * 
	 * 2 tds mit zwei Hypothesen,
	 * 2 tds mit drei Hypothesen, 
	 * 2 tds mit vier Hypothesen,
	 * 1 td  mit sieben Hypothesen
	 * 
	 * (wobei deren mangelnde Vollständigkeit ignoriert wird).
	 */
	@Override
	protected List<TemplateDependency> generateTemplateDependencies() {
		String relationname = RELATION_NAME.toLowerCase();
		
		List<TemplateDependency> templateDependencies = new LinkedList<TemplateDependency>();
		
		TemplateDependencyRow p1, p2, p3, p4, p5, p6, p7, conclusion;
					
		TemplateDependency td1 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td1.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p2.setElement(1, 			new NonDistinguishedSymbolElement("b2"));
		p2.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td1.addPremise(p2);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td1.setConclusion(conclusion);
		templateDependencies.add(td1);
		
		TemplateDependency td2 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td2.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p2.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p2.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td2.addPremise(p2);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td2.setConclusion(conclusion);
		templateDependencies.add(td2);
		
		TemplateDependency td3 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td3.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p2.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p2.setElement(2, 			new NonDistinguishedSymbolElement("b3"));
		td3.addPremise(p2);
		p3 = new TemplateDependencyRow(3);
		p3.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p3.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p3.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td3.addPremise(p3);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td3.setConclusion(conclusion);
		templateDependencies.add(td3);
		
		TemplateDependency td4 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td4.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p2.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p2.setElement(2, 			new NonDistinguishedSymbolElement("b5"));
		td4.addPremise(p2);
		p3 = new TemplateDependencyRow(3);
		p3.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p3.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p3.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td4.addPremise(p3);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td4.setConclusion(conclusion);
		templateDependencies.add(td4);
		
		TemplateDependency td5 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td5.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p2.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p2.setElement(2, 			new NonDistinguishedSymbolElement("b3"));
		td5.addPremise(p2);
		p3 = new TemplateDependencyRow(3);
		p3.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p3.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p3.setElement(2, 			new NonDistinguishedSymbolElement("b5"));
		td5.addPremise(p3);
		p4 = new TemplateDependencyRow(3);
		p4.setElement(0, 			new NonDistinguishedSymbolElement("b6"));
		p4.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p4.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td5.addPremise(p4);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td5.setConclusion(conclusion);
		templateDependencies.add(td5);
		
		TemplateDependency td6 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td6.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p2.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p2.setElement(2, 			new NonDistinguishedSymbolElement("b5"));
		td6.addPremise(p2);
		p3 = new TemplateDependencyRow(3);
		p3.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p3.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p3.setElement(2, 			new NonDistinguishedSymbolElement("b6"));
		td6.addPremise(p3);
		p4 = new TemplateDependencyRow(3);
		p4.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p4.setElement(1, 			new NonDistinguishedSymbolElement("b7"));
		p4.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td6.addPremise(p4);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td6.setConclusion(conclusion);
		templateDependencies.add(td6);
		
		TemplateDependency td7 = new TemplateDependency(relationname, 3);
		p1 = new TemplateDependencyRow(3);
		p1.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p1.setElement(1, 			new DistinguishedSymbolElement("aD"));
		p1.setElement(2, 			new NonDistinguishedSymbolElement("b1"));
		td7.addPremise(p1);
		p2 = new TemplateDependencyRow(3);
		p2.setElement(0, 			new DistinguishedSymbolElement("aS"));
		p2.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p2.setElement(2, 			new NonDistinguishedSymbolElement("b5"));
		td7.addPremise(p2);
		p3 = new TemplateDependencyRow(3);
		p3.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p3.setElement(1, 			new NonDistinguishedSymbolElement("b4"));
		p3.setElement(2, 			new NonDistinguishedSymbolElement("b6"));
		td7.addPremise(p3);
		p4 = new TemplateDependencyRow(3);
		p4.setElement(0, 			new NonDistinguishedSymbolElement("b2"));
		p4.setElement(1, 			new NonDistinguishedSymbolElement("b7"));
		p4.setElement(2, 			new NonDistinguishedSymbolElement("b9"));
		td7.addPremise(p4);
		p5 = new TemplateDependencyRow(3);
		p5.setElement(0, 			new NonDistinguishedSymbolElement("b8"));
		p5.setElement(1, 			new NonDistinguishedSymbolElement("b7"));
		p5.setElement(2, 			new NonDistinguishedSymbolElement("b11"));
		td7.addPremise(p5);
		p6 = new TemplateDependencyRow(3);
		p6.setElement(0, 			new NonDistinguishedSymbolElement("b8"));
		p6.setElement(1, 			new NonDistinguishedSymbolElement("b10"));
		p6.setElement(2, 			new NonDistinguishedSymbolElement("b13"));
		td7.addPremise(p6);
		p7 = new TemplateDependencyRow(3);
		p7.setElement(0, 			new NonDistinguishedSymbolElement("b12"));
		p7.setElement(1, 			new NonDistinguishedSymbolElement("b10"));
		p7.setElement(2, 			new DistinguishedSymbolElement("aP"));
		td7.addPremise(p7);
		conclusion = new TemplateDependencyRow(3);
		conclusion.setElement(0, 	new DistinguishedSymbolElement("aS"));
		conclusion.setElement(1, 	new DistinguishedSymbolElement("aD"));
		conclusion.setElement(2, 	new DistinguishedSymbolElement("aP"));
		td7.setConclusion(conclusion);
		templateDependencies.add(td7);
		
		return templateDependencies;
	}
	
	protected SignatureConfidentialityPolicyItem getPotSec(int i) throws UnsupportedFormulaException, DatabaseException {
		Formula formula = null;
		try {
			// Ziel-Formel: EXISTS X krankheit(X, "td_i_Cancer", "td_i_Smith")
			formula = new Formula( "EXISTS X " + RELATION_NAME.toLowerCase() + "(X, \"td" + i + "_Cancer\", \"td" + i + "_Smith\")" );
		}
		catch( ParserException e ) {
			throw new UnsupportedFormulaException("formula could not be parsed");
		}
		
		return SignatureConfidentialityPolicyItem.createSignatureConfidentialityPolicyItem(i, this.userID, formula);
	}
}
